package io.github.pigaut.sql.database;

import com.zaxxer.hikari.*;
import io.github.pigaut.sql.*;
import io.github.pigaut.sql.database.statement.*;

import javax.sql.*;
import java.io.*;
import java.sql.*;
import java.util.*;

public class FileDatabase implements Database {

    private final File file;
    private HikariDataSource dataSource = null;

    public FileDatabase(File file) {
        this.file = file;
    }

    @Override
    public boolean isConnected() {
        return dataSource != null && dataSource.isRunning();
    }

    @Override
    public void openConnection() {
        if (dataSource == null || !dataSource.isRunning()) {
            dataSource = SQLib.createDataSource(file);
        }
    }

    @Override
    public void closeConnection() {
        if (dataSource != null && !dataSource.isRunning()) {
            dataSource.close();
            dataSource = null;
        }
    }

    @Override
    public DatabaseStatement createStatement(String sql) {
        return new SimpleDatabaseStatement(this, sql);
    }

    @Override
    public DataSource getDataSource() {
        return dataSource;
    }

    @Override
    public void createTableIfNotExists(String name, String... columns) {
        final String columnsDef = String.join(", ", columns);
        final String createTableSQL = String.format(StatementTemplate.CREATE_TABLE_IF_NOT_EXISTS, name, columnsDef);
        execute(createTableSQL);
    }

    @Override
    public void dropTable(String table) {
        final String dropTableSQL = String.format(StatementTemplate.DROP_TABLE, table);
        execute(dropTableSQL);
    }

    @Override
    public void renameTable(String table, String newName) {
        final String renameTableSQL = String.format(StatementTemplate.RENAME_TABLE, table, newName);
        execute(renameTableSQL);
    }

    @Override
    public void addColumn(String table, String columnDefinition) {
        final String addColumnStmt = String.format(StatementTemplate.ADD_COLUMN, table, columnDefinition);
        execute(addColumnStmt);
    }

    @Override
    public void renameColumn(String table, String column, String newName) {
        final String renameColumnStmt = String.format(StatementTemplate.RENAME_COLUMN, table, column, newName);
        execute(renameColumnStmt);
    }

    @Override
    public void clearTable(String table) {
        final String deleteFromSQL = String.format(StatementTemplate.TRUNCATE, table);
        executeUpdate(deleteFromSQL);
    }

    @Override
    public DatabaseStatement insert(String table, String... columns) {
        final String columnsDef = String.join(", ", columns);
        final String values = String.join(", ", Collections.nCopies(columns.length, "?"));
        final String insertIntoSQL = String.format(StatementTemplate.INSERT_COLUMNS, table, columnsDef, values);
        return new SimpleDatabaseStatement(this, insertIntoSQL);
    }

    @Override
    public DatabaseStatement merge(String table, String primaryKey, String... columns) {
        final String columnsDef = String.join(", ", columns);
        final String values = String.join(", ", Collections.nCopies(columns.length, "?"));
        final String insertIntoSQL = String.format(StatementTemplate.MERGE_COLUMNS, table, columnsDef, primaryKey, values);
        return new SimpleDatabaseStatement(this, insertIntoSQL);
    }

    @Override
    public DatabaseStatement selectAll(String table) {
        final String selectAllSQL = String.format(StatementTemplate.SELECT_ALL, table);
        return new SimpleDatabaseStatement(this, selectAllSQL);
    }

    private <T> T executeStatement(StatementExecutor<T> executor) {
        try (Connection connection = dataSource.getConnection()) {
            try (Statement statement = connection.createStatement()) {
                try {
                    return executor.execute(statement);
                } catch (SQLException e) {
                    throw new DatabaseException("Could not execute database statement", e);
                }
            } catch (SQLException e) {
                throw new DatabaseException("Could not create database statement", e);
            }
        } catch (SQLException e) {
            throw new DatabaseException("Could not establish database connection", e);
        }
    }

    @FunctionalInterface
    private interface StatementExecutor<T> {
        T execute(Statement statement) throws SQLException;
    }

    @Override
    public void execute(String sql) {
        executeStatement(statement -> statement.execute(sql));
    }

    @Override
    public int executeUpdate(String sql) {
        return executeStatement(statement -> statement.executeUpdate(sql));
    }

    @Override
    public long executeLargeUpdate(String sql) {
        return executeStatement(statement -> statement.executeLargeUpdate(sql));
    }

    @Override
    public void executeQuery(String sql, QueryReader reader) {
        executeStatement(statement -> {
            try (ResultSet results = statement.executeQuery(sql)) {
                reader.read(results);
            }
            return null;
        });
    }

    @Override
    public void executeRowQuery(String sql, QueryReader reader) {
        executeStatement(preparedStatement -> {
            try (ResultSet results = preparedStatement.executeQuery(sql)) {
                while (results.next()) {
                    reader.read(results);
                }
            }
            return null;
        });
    }

}
