package javaforce.service;

/** File Sync Server.
 *
 * Proprietary protocol similar in function to rsync.
 *
 * TCP Port 33203
 *
 * @author pquiring
 */

import java.io.*;
import java.net.*;
import java.util.*;

import javaforce.*;

public class FileSyncServer {

  private static boolean debug;

  private String passwd;
  private String root;

  private Server server;
  private Object lock = new Object();
  private ServerSocket ss;
  private ArrayList<Client> clients = new ArrayList<>();

  public FileSyncServer(String passwd, String root) {
    this.passwd = passwd;
    this.root = root;
  }

  private static String getKeyFile() {
    return JF.getConfigPath() + "/jffilesync.key";
  }

  private void addClient(Client client) {
    synchronized(lock) {
      clients.add(client);
    }
  }

  private void removeClient(Client client) {
    synchronized(lock) {
      clients.remove(client);
    }
  }

  private class Server extends Thread {
    public boolean active;
    public void run() {
      active = true;
      JFLog.append(JF.getLogPath() + "/jffilesync.log", true);
      JFLog.setRetention(30);
      JFLog.log("FileSync : Starting service");
      JFLog.log("CreateServerSocketSSL");
      KeyMgmt keys = new KeyMgmt();
      if (new File(getKeyFile()).exists()) {
        try {
          FileInputStream fis = new FileInputStream(getKeyFile());
          keys.open(fis, "password");
          fis.close();
        } catch (Exception e) {
          JFLog.log(e);
        }
      } else {
        JFLog.log("Warning:Server SSL Keys not generated!");
        return;
      }
      ss = JF.createServerSocketSSL(keys);
      try {
        while(active) {
          Socket s = ss.accept();
          Client client = new Client(s);
          addClient(client);
          client.start();
        }
      } catch (Exception e) {
        JFLog.log(e);
      }
    }
  }

  public void start() {
    stop();
    server = new Server();
    server.start();
  }

  public void stop() {
    if (server == null) return;
    server.active = false;
    synchronized(lock) {
      Client[] cs = clients.toArray(new Client[0]);
      for(Client c : cs) {
        c.close();
      }
      clients.clear();
    }
    synchronized(server) {
      server.notify();
    }
    server = null;
  }

  //client requests
  private static final byte REQ_AUTH = 0x01;
  private static final byte REQ_FOLDER_OPEN = 0x02;  //folder name
  private static final byte REQ_FILE_OPEN = 0x03;  //filename, date, size
  private static final byte REQ_FILE_BLOCK_HASH = 0x04;
  private static final byte REQ_FILE_BLOCK_DATA = 0x05;
  private static final byte REQ_FILE_TRUNCATE = 0x06;
  private static final byte REQ_FILE_DELETE = 0x07;
  private static final byte REQ_FOLDER_LIST = 0x10;  //may support mirror function in future

  //server replies
  private static final byte RET_OKAY = -0x01;
  private static final byte RET_NO_EXIST = -0x02;
  private static final byte RET_NO_CHANGE = -0x03;
  private static final byte RET_CHANGE = -0x04;

  private static final int max_passwd_len = 256;
  private static final int max_data_len = 1 * 1024 * 1024;  //1MB data max

  private class Client extends Thread {
    public String remote;
    private Socket s;
    private InputStream is;
    private OutputStream os;
    private boolean active;
    private byte[] data = new byte[max_data_len];
    private String folder;
    public Client(Socket s) {
      this.s = s;
      remote = s.getInetAddress().toString();
    }
    public void run() {
      int read;
      active = true;
      try {
        is = s.getInputStream();
        os = s.getOutputStream();
        //read auth (8)(32)(length)
        {
          int auth_cmd = is.read();
          if (auth_cmd == -1) throw new Exception("read error");
          if (auth_cmd != REQ_AUTH) throw new Exception("auth not received");
          read = is.readNBytes(data, 0, 4);
          if (read != 4) throw new Exception("read error");
          int length = BE.getuint32(data, 0);
          if (length > max_passwd_len) throw new Exception("passwd too long");
          read = is.readNBytes(data, 0, length);
          if (read != length) throw new Exception("passwd not received");
          String auth_passwd_str = new String(data);
          if (!passwd.equals(auth_passwd_str)) throw new Exception("access denied");
        }
        while (active) {
          //read command (8)
          //read length (32)
          //read data (length)
          int cmd = is.read();
          if (cmd == -1) throw new Exception("read error");
          read = is.readNBytes(data, 0, 4);
          if (read != 4) throw new Exception("read error");
          int length = BE.getuint32(data, 0);
          if (length > max_data_len) throw new Exception("data packet too big");
          read = is.readNBytes(data, 0, length);
          if (read != length) throw new Exception("read data error");
          switch (cmd) {
            //TODO
            default: throw new Exception("unknown cmd");
          }
        }
      } catch (Exception e) {
        if (debug) JFLog.log(e);
      }
      close();
      removeClient(this);
    }
    public void close() {
      active = false;
      try { s.close(); } catch (Exception e) {}
    }
  }
}
