package com.jn.sqlhelper.cli.commands;

import com.jn.agileway.shell.command.annotation.Command;
import com.jn.agileway.shell.command.annotation.CommandComponent;
import com.jn.agileway.shell.command.annotation.CommandOption;
import com.jn.langx.configuration.file.directoryfile.DirectoryBasedFileConfigurationRepository;
import com.jn.langx.util.Preconditions;
import com.jn.langx.util.Strings;
import com.jn.langx.util.collection.Collects;
import com.jn.langx.util.collection.StringMap;
import com.jn.langx.util.function.Consumer2;
import com.jn.sqlhelper.common.connection.NamedConnectionConfiguration;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Properties;

@CommandComponent
public class JdbcConnectionCommands {

    @Autowired
    DirectoryBasedFileConfigurationRepository<NamedConnectionConfiguration> repository;

    @Command(value = "jdbc list", desc = "list all connection configuration names")
    public List<String> getConnectionNames() {
        return Collects.asList(repository.getAll().keySet());
    }

    @Command(value = "jdbc create", desc = "create an new connection configuration")
    public NamedConnectionConfiguration addConnection(
            @CommandOption() String name,
            @CommandOption() String driver,
            @CommandOption() String url,
            @CommandOption String username,
            @CommandOption(defaultValue = "") String password,
            @CommandOption(defaultValue = "") String catalog,
            @CommandOption(defaultValue = "") String schema,
            @CommandOption(defaultValue = "") String props) {

        Preconditions.checkTrue(Strings.isNotEmpty(name));
        Preconditions.checkTrue(Strings.isNotEmpty(driver));
        Preconditions.checkTrue(Strings.isNotEmpty(url));
        Preconditions.checkTrue(Strings.isNotEmpty(username));


        StringMap propsMap = new StringMap(props, "=", "&");
        final Properties properties = new Properties();
        Collects.forEach(propsMap, new Consumer2<String, String>() {
            @Override
            public void accept(String key, String value) {
                properties.setProperty(key, value);
            }
        });

        NamedConnectionConfiguration configuration = new NamedConnectionConfiguration();
        configuration.setName(name);
        configuration.setDriver(driver);
        configuration.setUrl(url);
        configuration.setUser(username);
        configuration.setPassword(password);
        configuration.setDriverProps(properties);

        properties.setProperty("user", username);
        properties.setProperty("password", password);

        repository.add(configuration);
        return configuration;
    }

    @Command(value = "jdbc get", desc = "get a connection configuration by name")
    public NamedConnectionConfiguration getConnection(String name) {
        return repository.getById(name);
    }

    @Command(value = "jdbc remove", desc = "remove a connection configuration by name")
    public boolean removeConnection(String name) {
        repository.removeById(name);
        return true;
    }


}
