package io.github.robin.code.util;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static io.github.robin.code.constant.DateTimeConstant.DATE_FORMATTER;
import static java.time.temporal.ChronoUnit.DAYS;
import lombok.experimental.UtilityClass;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

/**
 * @author robin
 */
@UtilityClass
public class DateUtil {

    public static LocalDate toLocalDate(String localDate) {
        if (StringUtils.isBlank(localDate)) {
            return null;
        }
        return LocalDate.parse(localDate, DATE_FORMATTER);
    }

    public static LocalDate getFirstDayOfMonth(LocalDate localDate) {
        return localDate.with(TemporalAdjusters.firstDayOfMonth());
    }

    public static LocalDate getLastDayOfMonth(LocalDate localDate) {
        return localDate.with(TemporalAdjusters.lastDayOfMonth());
    }

    public static List<LocalDate> getDaysBetween(LocalDate startDate, LocalDate endDate) {
        return IntStream.range(0, (int)DAYS.between(startDate, endDate))
            .mapToObj(startDate::plusDays)
            .collect(Collectors.toList());
    }

    @Deprecated
    public static LocalDate date2LocalDate(Date date) {
        if (Objects.isNull(date)) {
            return null;
        }
        return date.toInstant()
            .atZone(ZoneId.systemDefault())
            .toLocalDate();
    }

    @Deprecated
    public static LocalDate asLocalDate(Date date) {
        return Instant.ofEpochMilli(date.getTime())
            .atZone(ZoneId.systemDefault())
            .toLocalDate();
    }

    @Deprecated
    public static Date asDate(LocalDate localDate) {
        return Date.from(localDate.atStartOfDay()
            .atZone(ZoneId.systemDefault())
            .toInstant());
    }

    public static List<Pair<LocalDate, LocalDate>> getMonthFirstLastDate(LocalDate startDate, LocalDate endDate) {
        if (startDate.isAfter(endDate)) {
            return Collections.emptyList();
        }
        LocalDate start = startDate.withDayOfMonth(1);
        LocalDate end = endDate.withDayOfMonth(1);
        List<Pair<LocalDate, LocalDate>> pairArrayList = new ArrayList<>();
        while (!start.isAfter(end)) {
            Pair<LocalDate, LocalDate> of = Pair.of(start, start.with(TemporalAdjusters.lastDayOfMonth()));
            pairArrayList.add(of);
            start = start.plusMonths(1);
        }
        return pairArrayList;
    }
}
