package imgui.extension.implot;

import imgui.ImDrawList;
import imgui.ImVec2;
import imgui.ImVec4;
import imgui.internal.ImGuiContext;
import imgui.type.ImBoolean;
import imgui.type.ImDouble;
import imgui.type.ImFloat;

public final class ImPlot {
    private ImPlot() {
    }

    /*JNI
        #include "_implot.h"
     */

    //-----------------------------------------------------------------------------
    // [SECTION] Contexts
    //-----------------------------------------------------------------------------

    /**
     * Creates a new ImPlot context. Call this after ImGui.createContext().
     */
    public static ImPlotContext createContext() {
        return new ImPlotContext(nCreateContext());
    }

    private static native long nCreateContext(); /*
        return (uintptr_t)ImPlot::CreateContext();
    */

    /**
     * Destroys an ImPlot context. Call this before ImGui.destroyContext(). NULL = destroy current context.
     */
    public static void destroyContext() {
        nDestroyContext();
    }

    /**
     * Destroys an ImPlot context. Call this before ImGui.destroyContext(). NULL = destroy current context.
     */
    public static void destroyContext(final ImPlotContext ctx) {
        nDestroyContext(ctx.ptr);
    }

    private static native void nDestroyContext(); /*
        ImPlot::DestroyContext();
    */

    private static native void nDestroyContext(long ctx); /*
        ImPlot::DestroyContext(reinterpret_cast<ImPlotContext*>(ctx));
    */

    /**
     * Returns the current ImPlot context. NULL if no context has ben set.
     */
    public static ImPlotContext getCurrentContext() {
        return new ImPlotContext(nGetCurrentContext());
    }

    private static native long nGetCurrentContext(); /*
        return (uintptr_t)ImPlot::GetCurrentContext();
    */

    /**
     * Sets the current ImPlot context.
     */
    public static void setCurrentContext(final ImPlotContext ctx) {
        nSetCurrentContext(ctx.ptr);
    }

    private static native void nSetCurrentContext(long ctx); /*
        ImPlot::SetCurrentContext(reinterpret_cast<ImPlotContext*>(ctx));
    */

    public static void setImGuiContext(final ImGuiContext ctx) {
        nSetImGuiContext(ctx.ptr);
    }

    private static native void nSetImGuiContext(long ctx); /*
        ImPlot::SetImGuiContext(reinterpret_cast<ImGuiContext*>(ctx));
    */

    //-----------------------------------------------------------------------------
    // [SECTION] Begin/End Plot
    //-----------------------------------------------------------------------------

    // Starts a 2D plotting context. If this function returns true, EndPlot() MUST
    // be called! You are encouraged to use the following convention:
    //
    // if (BeginPlot(...)) {
    //     PlotLine(...);
    //     ...
    //     EndPlot();
    // }
    //
    // Important notes:
    //
    // - #title_id must be unique to the current ImGui ID scope. If you need to avoid ID
    //   collisions or don't want to display a title in the plot, use double hashes
    //   (e.g. "MyPlot##HiddenIdText" or "##NoTitle").
    // - #size is the **frame** size of the plot widget, not the plot area. The default
    //   size of plots (i.e. when ImVec2(0,0)) can be modified in your ImPlotStyle.

    public static boolean beginPlot(final String titleId) {
        return nBeginPlot(titleId);
    }

    public static boolean beginPlot(final String titleId, final ImVec2 size) {
        return nBeginPlot(titleId, size.x, size.y);
    }

    public static boolean beginPlot(final String titleId, final float sizeX, final float sizeY) {
        return nBeginPlot(titleId, sizeX, sizeY);
    }

    public static boolean beginPlot(final String titleId, final ImVec2 size, final int flags) {
        return nBeginPlot(titleId, size.x, size.y, flags);
    }

    public static boolean beginPlot(final String titleId, final float sizeX, final float sizeY, final int flags) {
        return nBeginPlot(titleId, sizeX, sizeY, flags);
    }

    public static boolean beginPlot(final String titleId, final int flags) {
        return nBeginPlot(titleId, flags);
    }

    private static native boolean nBeginPlot(String obj_titleId); /*MANUAL
        auto titleId = obj_titleId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleId, JNI_FALSE);
        auto _result = ImPlot::BeginPlot(titleId);
        if (titleId != NULL) env->ReleaseStringUTFChars(obj_titleId, titleId);
        return _result;
    */

    private static native boolean nBeginPlot(String obj_titleId, float sizeX, float sizeY); /*MANUAL
        auto titleId = obj_titleId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleId, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginPlot(titleId, size);
        if (titleId != NULL) env->ReleaseStringUTFChars(obj_titleId, titleId);
        return _result;
    */

    private static native boolean nBeginPlot(String obj_titleId, float sizeX, float sizeY, int flags); /*MANUAL
        auto titleId = obj_titleId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleId, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginPlot(titleId, size, flags);
        if (titleId != NULL) env->ReleaseStringUTFChars(obj_titleId, titleId);
        return _result;
    */

    private static native boolean nBeginPlot(String obj_titleId, int flags); /*MANUAL
        auto titleId = obj_titleId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleId, JNI_FALSE);
        auto _result = ImPlot::BeginPlot(titleId, ImVec2(-1,0), flags);
        if (titleId != NULL) env->ReleaseStringUTFChars(obj_titleId, titleId);
        return _result;
    */

    /**
     * Only call EndPlot() if beginPlot() returns true! Typically called at the end
     * of an if statement conditioned on BeginPlot(). See example in beginPlot().
     */
    public static void endPlot() {
        nEndPlot();
    }

    private static native void nEndPlot(); /*
        ImPlot::EndPlot();
    */

    //-----------------------------------------------------------------------------
    // [SECTION] Begin/End Subplots
    //-----------------------------------------------------------------------------

    // Starts a subdivided plotting context. If the function returns true,
    // EndSubplots() MUST be called! Call BeginPlot/EndPlot AT MOST [rows*cols]
    // times in  between the begining and end of the subplot context. Plots are
    // added in row major order.
    //
    // Example:
    //
    // if (BeginSubplots("My Subplot",2,3,ImVec2(800,400)) {
    //     for (int i = 0; i < 6; ++i) {
    //         if (BeginPlot(...)) {
    //             ImPlot::PlotLine(...);
    //             ...
    //             EndPlot();
    //         }
    //     }
    //     EndSubplots();
    // }
    //
    // Produces:
    //
    // [0] | [1] | [2]
    // ----|-----|----
    // [3] | [4] | [5]
    //
    // Important notes:
    //
    // - #title_id must be unique to the current ImGui ID scope. If you need to avoid ID
    //   collisions or don't want to display a title in the plot, use double hashes
    //   (e.g. "MySubplot##HiddenIdText" or "##NoTitle").
    // - #rows and #cols must be greater than 0.
    // - #size is the size of the entire grid of subplots, not the individual plots
    // - #row_ratios and #col_ratios must have AT LEAST #rows and #cols elements,
    //   respectively. These are the sizes of the rows and columns expressed in ratios.
    //   If the user adjusts the dimensions, the arrays are updated with new ratios.
    //
    // Important notes regarding BeginPlot from inside of BeginSubplots:
    //
    // - The #title_id parameter of _BeginPlot_ (see above) does NOT have to be
    //   unique when called inside of a subplot context. Subplot IDs are hashed
    //   for your convenience so you don't have call PushID or generate unique title
    //   strings. Simply pass an empty string to BeginPlot unless you want to title
    //   each subplot.
    // - The #size parameter of _BeginPlot_ (see above) is ignored when inside of a
    //   subplot context. The actual size of the subplot will be based on the
    //   #size value you pass to _BeginSubplots_ and #row/#col_ratios if provided.

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final ImVec2 size) {
        return nBeginSubplots(titleID, rows, cols, size.x, size.y);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final float sizeX, final float sizeY) {
        return nBeginSubplots(titleID, rows, cols, sizeX, sizeY);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final ImVec2 size, final int flags) {
        return nBeginSubplots(titleID, rows, cols, size.x, size.y, flags);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final float sizeX, final float sizeY, final int flags) {
        return nBeginSubplots(titleID, rows, cols, sizeX, sizeY, flags);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final ImVec2 size, final int flags, final float[] rowRatios) {
        return nBeginSubplots(titleID, rows, cols, size.x, size.y, flags, rowRatios);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final float sizeX, final float sizeY, final int flags, final float[] rowRatios) {
        return nBeginSubplots(titleID, rows, cols, sizeX, sizeY, flags, rowRatios);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final ImVec2 size, final int flags, final float[] rowRatios, final float[] colRatios) {
        return nBeginSubplots(titleID, rows, cols, size.x, size.y, flags, rowRatios, colRatios);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final float sizeX, final float sizeY, final int flags, final float[] rowRatios, final float[] colRatios) {
        return nBeginSubplots(titleID, rows, cols, sizeX, sizeY, flags, rowRatios, colRatios);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final ImVec2 size, final float[] rowRatios, final float[] colRatios) {
        return nBeginSubplots(titleID, rows, cols, size.x, size.y, rowRatios, colRatios);
    }

    public static boolean beginSubplots(final String titleID, final int rows, final int cols, final float sizeX, final float sizeY, final float[] rowRatios, final float[] colRatios) {
        return nBeginSubplots(titleID, rows, cols, sizeX, sizeY, rowRatios, colRatios);
    }

    private static native boolean nBeginSubplots(String obj_titleID, int rows, int cols, float sizeX, float sizeY); /*MANUAL
        auto titleID = obj_titleID == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleID, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginSubplots(titleID, rows, cols, size);
        if (titleID != NULL) env->ReleaseStringUTFChars(obj_titleID, titleID);
        return _result;
    */

    private static native boolean nBeginSubplots(String obj_titleID, int rows, int cols, float sizeX, float sizeY, int flags); /*MANUAL
        auto titleID = obj_titleID == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleID, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginSubplots(titleID, rows, cols, size, flags);
        if (titleID != NULL) env->ReleaseStringUTFChars(obj_titleID, titleID);
        return _result;
    */

    private static native boolean nBeginSubplots(String obj_titleID, int rows, int cols, float sizeX, float sizeY, int flags, float[] obj_rowRatios); /*MANUAL
        auto titleID = obj_titleID == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleID, JNI_FALSE);
        auto rowRatios = obj_rowRatios == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_rowRatios, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginSubplots(titleID, rows, cols, size, flags, &rowRatios[0]);
        if (titleID != NULL) env->ReleaseStringUTFChars(obj_titleID, titleID);
        if (rowRatios != NULL) env->ReleasePrimitiveArrayCritical(obj_rowRatios, rowRatios, JNI_FALSE);
        return _result;
    */

    private static native boolean nBeginSubplots(String obj_titleID, int rows, int cols, float sizeX, float sizeY, int flags, float[] obj_rowRatios, float[] obj_colRatios); /*MANUAL
        auto titleID = obj_titleID == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleID, JNI_FALSE);
        auto rowRatios = obj_rowRatios == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_rowRatios, JNI_FALSE);
        auto colRatios = obj_colRatios == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_colRatios, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginSubplots(titleID, rows, cols, size, flags, &rowRatios[0], &colRatios[0]);
        if (titleID != NULL) env->ReleaseStringUTFChars(obj_titleID, titleID);
        if (rowRatios != NULL) env->ReleasePrimitiveArrayCritical(obj_rowRatios, rowRatios, JNI_FALSE);
        if (colRatios != NULL) env->ReleasePrimitiveArrayCritical(obj_colRatios, colRatios, JNI_FALSE);
        return _result;
    */

    private static native boolean nBeginSubplots(String obj_titleID, int rows, int cols, float sizeX, float sizeY, float[] obj_rowRatios, float[] obj_colRatios); /*MANUAL
        auto titleID = obj_titleID == NULL ? NULL : (char*)env->GetStringUTFChars(obj_titleID, JNI_FALSE);
        auto rowRatios = obj_rowRatios == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_rowRatios, JNI_FALSE);
        auto colRatios = obj_colRatios == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_colRatios, JNI_FALSE);
        ImVec2 size = ImVec2(sizeX, sizeY);
        auto _result = ImPlot::BeginSubplots(titleID, rows, cols, size, ImPlotSubplotFlags_None, &rowRatios[0], &colRatios[0]);
        if (titleID != NULL) env->ReleaseStringUTFChars(obj_titleID, titleID);
        if (rowRatios != NULL) env->ReleasePrimitiveArrayCritical(obj_rowRatios, rowRatios, JNI_FALSE);
        if (colRatios != NULL) env->ReleasePrimitiveArrayCritical(obj_colRatios, colRatios, JNI_FALSE);
        return _result;
    */

    /**
     * Only call EndSubplots() if BeginSubplots() returns true! Typically called at the end
     * of an if statement conditioned on BeginSublots(). See example above.
     */
    public static void endSubplots() {
        nEndSubplots();
    }

    private static native void nEndSubplots(); /*
        ImPlot::EndSubplots();
    */

    //-----------------------------------------------------------------------------
    // [SECTION] Setup
    //-----------------------------------------------------------------------------

    // The following API allows you to setup and customize various aspects of the
    // current plot. The functions should be called immediately after BeginPlot
    // and before any other API calls. Typical usage is as follows:

    // if (BeginPlot(...)) {                     1) begin a new plot
    //     SetupAxis(ImAxis_X1, "My X-Axis");    2) make Setup calls
    //     SetupAxis(ImAxis_Y1, "My Y-Axis");
    //     SetupLegend(ImPlotLocation_North);
    //     ...
    //     SetupFinish();                        3) [optional] explicitly finish setup
    //     PlotLine(...);                        4) plot items
    //     ...
    //     EndPlot();                            5) end the plot
    // }
    //
    // Important notes:
    //
    // - Always call Setup code at the top of your BeginPlot conditional statement.
    // - Setup is locked once you start plotting or explicitly call SetupFinish.
    //   Do NOT call Setup code after you begin plotting or after you make
    //   any non-Setup API calls (e.g. utils like PlotToPixels also lock Setup)
    // - Calling SetupFinish is OPTIONAL, but probably good practice. If you do not
    //   call it yourself, then the first subsequent plotting or utility function will
    //   call it for you.

    /**
     * Enables an axis or sets the label and/or flags for an existing axis.
     * Leave `label` as NULL for no label.
     */
    public static void setupAxis(final int axis) {
        nSetupAxis(axis);
    }

    /**
     * Enables an axis or sets the label and/or flags for an existing axis.
     * Leave `label` as NULL for no label.
     */
    public static void setupAxis(final int axis, final String label) {
        nSetupAxis(axis, label);
    }

    /**
     * Enables an axis or sets the label and/or flags for an existing axis.
     * Leave `label` as NULL for no label.
     */
    public static void setupAxis(final int axis, final String label, final int flags) {
        nSetupAxis(axis, label, flags);
    }

    /**
     * Enables an axis or sets the label and/or flags for an existing axis.
     * Leave `label` as NULL for no label.
     */
    public static void setupAxis(final int axis, final int flags) {
        nSetupAxis(axis, flags);
    }

    private static native void nSetupAxis(int axis); /*
        ImPlot::SetupAxis(axis);
    */

    private static native void nSetupAxis(int axis, String label); /*MANUAL
        auto label = obj_label == NULL ? NULL : (char*)env->GetStringUTFChars(obj_label, JNI_FALSE);
        ImPlot::SetupAxis(axis, label);
        if (label != NULL) env->ReleaseStringUTFChars(obj_label, label);
    */

    private static native void nSetupAxis(int axis, String label, int flags); /*MANUAL
        auto label = obj_label == NULL ? NULL : (char*)env->GetStringUTFChars(obj_label, JNI_FALSE);
        ImPlot::SetupAxis(axis, label, flags);
        if (label != NULL) env->ReleaseStringUTFChars(obj_label, label);
    */

    private static native void nSetupAxis(int axis, int flags); /*
        ImPlot::SetupAxis(axis, NULL, flags);
    */

    /**
     * Sets an axis range limits. If ImPlotCond_Always is used, the axes limits will be locked.
     * Inversion with {@code v_min > v_max} is not supported; use SetupAxisLimits instead.
     */
    public static void setupAxisLimits(final int axis, final double vMin, final double vMax) {
        nSetupAxisLimits(axis, vMin, vMax);
    }

    /**
     * Sets an axis range limits. If ImPlotCond_Always is used, the axes limits will be locked.
     * Inversion with {@code v_min > v_max} is not supported; use SetupAxisLimits instead.
     */
    public static void setupAxisLimits(final int axis, final double vMin, final double vMax, final int cond) {
        nSetupAxisLimits(axis, vMin, vMax, cond);
    }

    private static native void nSetupAxisLimits(int axis, double vMin, double vMax); /*
        ImPlot::SetupAxisLimits(axis, vMin, vMax);
    */

    private static native void nSetupAxisLimits(int axis, double vMin, double vMax, int cond); /*
        ImPlot::SetupAxisLimits(axis, vMin, vMax, cond);
    */

    /**
     * Links an axis range limits to external values. Set to NULL for no linkage.
     * The pointer data must remain valid until EndPlot.
     */
    public static void setupAxisLinks(final int axis, final ImDouble linkMin, final ImDouble linkMax) {
        nSetupAxisLinks(axis, linkMin != null ? linkMin.getData() : null, linkMax != null ? linkMax.getData() : null);
    }

    private static native void nSetupAxisLinks(int axis, double[] linkMin, double[] linkMax); /*MANUAL
        auto linkMin = obj_linkMin == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_linkMin, JNI_FALSE);
        auto linkMax = obj_linkMax == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_linkMax, JNI_FALSE);
        ImPlot::SetupAxisLinks(axis, (linkMin != NULL ? &linkMin[0] : NULL), (linkMax != NULL ? &linkMax[0] : NULL));
        if (linkMin != NULL) env->ReleasePrimitiveArrayCritical(obj_linkMin, linkMin, JNI_FALSE);
        if (linkMax != NULL) env->ReleasePrimitiveArrayCritical(obj_linkMax, linkMax, JNI_FALSE);
    */

    /**
     * Sets the format of numeric axis labels via formatter specifier (default="%g").
     * Formatted values will be double (i.e. use %f).
     */
    public static void setupAxisFormat(final int axis, final String fmt) {
        nSetupAxisFormat(axis, fmt);
    }

    private static native void nSetupAxisFormat(int axis, String fmt); /*MANUAL
        auto fmt = obj_fmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_fmt, JNI_FALSE);
        ImPlot::SetupAxisFormat(axis, fmt);
        if (fmt != NULL) env->ReleaseStringUTFChars(obj_fmt, fmt);
    */

    // TODO: support ImPlotFormatter

    /**
     * Sets an axis' ticks and optionally the labels. To keep the default ticks,
     * set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double[] values, final int nTicks) {
        nSetupAxisTicks(axis, values, nTicks);
    }

    /**
     * Sets an axis' ticks and optionally the labels. To keep the default ticks,
     * set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double[] values, final int nTicks, final String[] labels) {
        nSetupAxisTicks(axis, values, nTicks, labels, labels.length);
    }

    /**
     * Sets an axis' ticks and optionally the labels. To keep the default ticks,
     * set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double[] values, final int nTicks, final String[] labels, final boolean keepDefault) {
        nSetupAxisTicks(axis, values, nTicks, labels, labels.length, keepDefault);
    }

    /**
     * Sets an axis' ticks and optionally the labels. To keep the default ticks,
     * set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double[] values, final int nTicks, final boolean keepDefault) {
        nSetupAxisTicks(axis, values, nTicks, keepDefault);
    }

    private static native void nSetupAxisTicks(int axis, double[] values, int nTicks); /*MANUAL
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::SetupAxisTicks(axis, &values[0], nTicks);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nSetupAxisTicks(int axis, double[] values, int nTicks, String[] obj_labels, int labelsCount); /*MANUAL
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        const char* labels[labelsCount];
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labels[i] = rawStr;
        };
        ImPlot::SetupAxisTicks(axis, &values[0], nTicks, labels);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            env->ReleaseStringUTFChars(str, labels[i]);
        };
    */

    private static native void nSetupAxisTicks(int axis, double[] values, int nTicks, String[] obj_labels, int labelsCount, boolean keepDefault); /*MANUAL
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        const char* labels[labelsCount];
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labels[i] = rawStr;
        };
        ImPlot::SetupAxisTicks(axis, &values[0], nTicks, labels, keepDefault);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            env->ReleaseStringUTFChars(str, labels[i]);
        };
    */

    private static native void nSetupAxisTicks(int axis, double[] values, int nTicks, boolean keepDefault); /*MANUAL
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::SetupAxisTicks(axis, &values[0], nTicks, NULL, keepDefault);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Sets an axis' ticks and optionally the labels for the next plot.
     * To keep the default ticks, set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double vMin, final double vMax, final int nTicks) {
        nSetupAxisTicks(axis, vMin, vMax, nTicks);
    }

    /**
     * Sets an axis' ticks and optionally the labels for the next plot.
     * To keep the default ticks, set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double vMin, final double vMax, final int nTicks, final String[] labels) {
        nSetupAxisTicks(axis, vMin, vMax, nTicks, labels, labels.length);
    }

    /**
     * Sets an axis' ticks and optionally the labels for the next plot.
     * To keep the default ticks, set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double vMin, final double vMax, final int nTicks, final String[] labels, final boolean keepDefault) {
        nSetupAxisTicks(axis, vMin, vMax, nTicks, labels, labels.length, keepDefault);
    }

    /**
     * Sets an axis' ticks and optionally the labels for the next plot.
     * To keep the default ticks, set `keepDefault=true`.
     */
    public static void setupAxisTicks(final int axis, final double vMin, final double vMax, final int nTicks, final boolean keepDefault) {
        nSetupAxisTicks(axis, vMin, vMax, nTicks, keepDefault);
    }

    private static native void nSetupAxisTicks(int axis, double vMin, double vMax, int nTicks); /*
        ImPlot::SetupAxisTicks(axis, vMin, vMax, nTicks);
    */

    private static native void nSetupAxisTicks(int axis, double vMin, double vMax, int nTicks, String[] obj_labels, int labelsCount); /*MANUAL
        const char* labels[labelsCount];
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labels[i] = rawStr;
        };
        ImPlot::SetupAxisTicks(axis, vMin, vMax, nTicks, labels);
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            env->ReleaseStringUTFChars(str, labels[i]);
        };
    */

    private static native void nSetupAxisTicks(int axis, double vMin, double vMax, int nTicks, String[] obj_labels, int labelsCount, boolean keepDefault); /*MANUAL
        const char* labels[labelsCount];
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labels[i] = rawStr;
        };
        ImPlot::SetupAxisTicks(axis, vMin, vMax, nTicks, labels, keepDefault);
        for (int i = 0; i < labelsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labels, i);
            env->ReleaseStringUTFChars(str, labels[i]);
        };
    */

    private static native void nSetupAxisTicks(int axis, double vMin, double vMax, int nTicks, boolean keepDefault); /*
        ImPlot::SetupAxisTicks(axis, vMin, vMax, nTicks, NULL, keepDefault);
    */

    /**
     * Sets an axis' scale using built-in options.
     */
    public static void setupAxisScale(final int axis, final int scale) {
        nSetupAxisScale(axis, scale);
    }

    private static native void nSetupAxisScale(int axis, int scale); /*
        ImPlot::SetupAxisScale(axis, scale);
    */

    /**
     * Sets an axis' limits constraints.
     */
    public static void setupAxisLimitsConstraints(final int axis, final double vMin, final double vMax) {
        nSetupAxisLimitsConstraints(axis, vMin, vMax);
    }

    private static native void nSetupAxisLimitsConstraints(int axis, double vMin, double vMax); /*
        ImPlot::SetupAxisLimitsConstraints(axis, vMin, vMax);
    */

    /**
     * Sets an axis' zoom constraints.
     */
    public static void setupAxisZoomConstraints(final int axis, final double vMin, final double vMax) {
        nSetupAxisZoomConstraints(axis, vMin, vMax);
    }

    private static native void nSetupAxisZoomConstraints(int axis, double vMin, double vMax); /*
        ImPlot::SetupAxisZoomConstraints(axis, vMin, vMax);
    */

    /**
     * Sets the label and/or flags for primary X and Y axes (shorthand for two calls to SetupAxis).
     */
    public static void setupAxes(final String xLabel, final String yLabel) {
        nSetupAxes(xLabel, yLabel);
    }

    /**
     * Sets the label and/or flags for primary X and Y axes (shorthand for two calls to SetupAxis).
     */
    public static void setupAxes(final String xLabel, final String yLabel, final int xFlags) {
        nSetupAxes(xLabel, yLabel, xFlags);
    }

    /**
     * Sets the label and/or flags for primary X and Y axes (shorthand for two calls to SetupAxis).
     */
    public static void setupAxes(final String xLabel, final String yLabel, final int xFlags, final int yFlags) {
        nSetupAxes(xLabel, yLabel, xFlags, yFlags);
    }

    private static native void nSetupAxes(String xLabel, String yLabel); /*MANUAL
        auto xLabel = obj_xLabel == NULL ? NULL : (char*)env->GetStringUTFChars(obj_xLabel, JNI_FALSE);
        auto yLabel = obj_yLabel == NULL ? NULL : (char*)env->GetStringUTFChars(obj_yLabel, JNI_FALSE);
        ImPlot::SetupAxes(xLabel, yLabel);
        if (xLabel != NULL) env->ReleaseStringUTFChars(obj_xLabel, xLabel);
        if (yLabel != NULL) env->ReleaseStringUTFChars(obj_yLabel, yLabel);
    */

    private static native void nSetupAxes(String xLabel, String yLabel, int xFlags); /*MANUAL
        auto xLabel = obj_xLabel == NULL ? NULL : (char*)env->GetStringUTFChars(obj_xLabel, JNI_FALSE);
        auto yLabel = obj_yLabel == NULL ? NULL : (char*)env->GetStringUTFChars(obj_yLabel, JNI_FALSE);
        ImPlot::SetupAxes(xLabel, yLabel, xFlags);
        if (xLabel != NULL) env->ReleaseStringUTFChars(obj_xLabel, xLabel);
        if (yLabel != NULL) env->ReleaseStringUTFChars(obj_yLabel, yLabel);
    */

    private static native void nSetupAxes(String xLabel, String yLabel, int xFlags, int yFlags); /*MANUAL
        auto xLabel = obj_xLabel == NULL ? NULL : (char*)env->GetStringUTFChars(obj_xLabel, JNI_FALSE);
        auto yLabel = obj_yLabel == NULL ? NULL : (char*)env->GetStringUTFChars(obj_yLabel, JNI_FALSE);
        ImPlot::SetupAxes(xLabel, yLabel, xFlags, yFlags);
        if (xLabel != NULL) env->ReleaseStringUTFChars(obj_xLabel, xLabel);
        if (yLabel != NULL) env->ReleaseStringUTFChars(obj_yLabel, yLabel);
    */

    /**
     * Sets the primary X and Y axes range limits. If ImPlotCond_Always is used,
     * the axes limits will be locked (shorthand for two calls to SetupAxisLimits).
     */
    public static void setupAxesLimits(final double xMin, final double xMax, final double yMin, final double yMax) {
        nSetupAxesLimits(xMin, xMax, yMin, yMax);
    }

    /**
     * Sets the primary X and Y axes range limits. If ImPlotCond_Always is used,
     * the axes limits will be locked (shorthand for two calls to SetupAxisLimits).
     */
    public static void setupAxesLimits(final double xMin, final double xMax, final double yMin, final double yMax, final int cond) {
        nSetupAxesLimits(xMin, xMax, yMin, yMax, cond);
    }

    private static native void nSetupAxesLimits(double xMin, double xMax, double yMin, double yMax); /*
        ImPlot::SetupAxesLimits(xMin, xMax, yMin, yMax);
    */

    private static native void nSetupAxesLimits(double xMin, double xMax, double yMin, double yMax, int cond); /*
        ImPlot::SetupAxesLimits(xMin, xMax, yMin, yMax, cond);
    */

    /**
     * Sets up the plot legend.
     */
    public static void setupLegend(final int location) {
        nSetupLegend(location);
    }

    /**
     * Sets up the plot legend.
     */
    public static void setupLegend(final int location, final int flags) {
        nSetupLegend(location, flags);
    }

    private static native void nSetupLegend(int location); /*
        ImPlot::SetupLegend(location);
    */

    private static native void nSetupLegend(int location, int flags); /*
        ImPlot::SetupLegend(location, flags);
    */

    /**
     * Sets the location of the current plot's mouse position text (default = South|East).
     */
    public static void setupMouseText(final int location) {
        nSetupMouseText(location);
    }

    /**
     * Sets the location of the current plot's mouse position text (default = South|East).
     */
    public static void setupMouseText(final int location, final int flags) {
        nSetupMouseText(location, flags);
    }

    private static native void nSetupMouseText(int location); /*
        ImPlot::SetupMouseText(location);
    */

    private static native void nSetupMouseText(int location, int flags); /*
        ImPlot::SetupMouseText(location, flags);
    */

    /**
     * Explicitly finalize plot setup. Once you call this, you cannot make any more
     * Setup calls for the current plot! Note that calling this function is OPTIONAL;
     * it will be called by the first subsequent setup-locking API call.
     */
    public static void setupFinish() {
        nSetupFinish();
    }

    private static native void nSetupFinish(); /*
        ImPlot::SetupFinish();
    */

    //-----------------------------------------------------------------------------
    // [SECTION] SetNext
    //-----------------------------------------------------------------------------

    // Though you should default to the `Setup` API above, there are some scenarios
    // where (re)configuring a plot or axis before `BeginPlot` is needed (e.g. if
    // using a preceding button or slider widget to change the plot limits). In
    // this case, you can use the `SetNext` API below. While this is not as feature
    // rich as the Setup API, most common needs are provided. These functions can be
    // called anwhere except for inside of `Begin/EndPlot`. For example:

    // if (ImGui::Button("Center Plot"))
    //     ImPlot::SetNextPlotLimits(-1,1,-1,1);
    // if (ImPlot::BeginPlot(...)) {
    //     ...
    //     ImPlot::EndPlot();
    // }
    //
    // Important notes:
    //
    // - You must still enable non-default axes with SetupAxis for these functions
    //   to work properly.

    /**
     * Sets an upcoming axis range limits. If ImPlotCond_Always is used, the axes limits will be locked.
     */
    public static void setNextAxisLimits(final int axis, final double vMin, final double vMax) {
        nSetNextAxisLimits(axis, vMin, vMax);
    }

    /**
     * Sets an upcoming axis range limits. If ImPlotCond_Always is used, the axes limits will be locked.
     */
    public static void setNextAxisLimits(final int axis, final double vMin, final double vMax, final int cond) {
        nSetNextAxisLimits(axis, vMin, vMax, cond);
    }

    private static native void nSetNextAxisLimits(int axis, double vMin, double vMax); /*
        ImPlot::SetNextAxisLimits(axis, vMin, vMax);
    */

    private static native void nSetNextAxisLimits(int axis, double vMin, double vMax, int cond); /*
        ImPlot::SetNextAxisLimits(axis, vMin, vMax, cond);
    */

    /**
     * Links an upcoming axis range limits to external values. Set to NULL for no linkage.
     * The pointer data must remain valid until EndPlot!
     */
    public static void setNextAxisLinks(final int axis, final ImDouble linkMin, final ImDouble linkMax) {
        nSetNextAxisLinks(axis, linkMin != null ? linkMin.getData() : null, linkMax != null ? linkMax.getData() : null);
    }

    private static native void nSetNextAxisLinks(int axis, double[] linkMin, double[] linkMax); /*MANUAL
        auto linkMin = obj_linkMin == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_linkMin, JNI_FALSE);
        auto linkMax = obj_linkMax == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_linkMax, JNI_FALSE);
        ImPlot::SetNextAxisLinks(axis, (linkMin != NULL ? &linkMin[0] : NULL), (linkMax != NULL ? &linkMax[0] : NULL));
        if (linkMin != NULL) env->ReleasePrimitiveArrayCritical(obj_linkMin, linkMin, JNI_FALSE);
        if (linkMax != NULL) env->ReleasePrimitiveArrayCritical(obj_linkMax, linkMax, JNI_FALSE);
    */

    /**
     * Set an upcoming axis to auto fit to its data.
     */
    public static void setNextAxisToFit(final int axis) {
        nSetNextAxisToFit(axis);
    }

    private static native void nSetNextAxisToFit(int axis); /*
        ImPlot::SetNextAxisToFit(axis);
    */

    /**
     * Sets the upcoming primary X and Y axes range limits. If ImPlotCond_Always is used,
     * the axes limits will be locked (shorthand for two calls to SetupAxisLimits).
     */
    public static void setNextAxesLimits(final double xMin, final double xMax, final double yMin, final double yMax) {
        nSetNextAxesLimits(xMin, xMax, yMin, yMax);
    }

    /**
     * Sets the upcoming primary X and Y axes range limits. If ImPlotCond_Always is used,
     * the axes limits will be locked (shorthand for two calls to SetupAxisLimits).
     */
    public static void setNextAxesLimits(final double xMin, final double xMax, final double yMin, final double yMax, final int cond) {
        nSetNextAxesLimits(xMin, xMax, yMin, yMax, cond);
    }

    private static native void nSetNextAxesLimits(double xMin, double xMax, double yMin, double yMax); /*
        ImPlot::SetNextAxesLimits(xMin, xMax, yMin, yMax);
    */

    private static native void nSetNextAxesLimits(double xMin, double xMax, double yMin, double yMax, int cond); /*
        ImPlot::SetNextAxesLimits(xMin, xMax, yMin, yMax, cond);
    */

    /**
     * Sets all upcoming axes to auto fit to their data.
     */
    public static void setNextAxesToFit() {
        nSetNextAxesToFit();
    }

    private static native void nSetNextAxesToFit(); /*
        ImPlot::SetNextAxesToFit();
    */

    //-----------------------------------------------------------------------------
    // [SECTION] Plot Items
    //-----------------------------------------------------------------------------

    // The main plotting API is provied below. Call these functions between
    // Begin/EndPlot and after any Setup API calls. Each plots data on the current
    // x and y axes, which can be changed with `SetAxis/Axes`.
    //
    // The templated functions are explicitly instantiated in implot_items.cpp.
    // They are not intended to be used generically with custom types. You will get
    // a linker error if you try! All functions support the following scalar types:
    //
    // float, double, ImS8, ImU8, ImS16, ImU16, ImS32, ImU32, ImS64, ImU64
    //
    //
    // If you need to plot custom or non-homogenous data you have a few options:
    //
    // 1. If your data is a simple struct/class (e.g. Vector2f), you can use striding.
    //    This is the most performant option if applicable.
    //
    //    struct Vector2f { float X, Y; };
    //    ...
    //    Vector2f data[42];
    //    ImPlot::PlotLine("line", &data[0].x, &data[0].y, 42, 0, sizeof(Vector2f)); // or sizeof(float)*2
    //
    // 2. Write a custom getter C function or C++ lambda and pass it and optionally your data to
    //    an ImPlot function post-fixed with a G (e.g. PlotScatterG). This has a slight performance
    //    cost, but probably not enough to worry about unless your data is very large. Examples:
    //
    //    ImPlotPoint MyDataGetter(void* data, int idx) {
    //        MyData* my_data = (MyData*)data;
    //        ImPlotPoint p;
    //        p.x = my_data->GetTime(idx);
    //        p.y = my_data->GetValue(idx);
    //        return p
    //    }
    //    ...
    //    auto my_lambda = [](void*, int idx) {
    //        double t = idx / 999.0;
    //        return ImPlotPoint(t, 0.5+0.5*std::sin(2*PI*10*t));
    //    };
    //    ...
    //    if (ImPlot::BeginPlot("MyPlot")) {
    //        MyData my_data;
    //        ImPlot::PlotScatterG("scatter", MyDataGetter, &my_data, my_data.Size());
    //        ImPlot::PlotLineG("line", my_lambda, nullptr, 1000);
    //        ImPlot::EndPlot();
    //    }
    //
    // NB: All types are converted to double before plotting. You may lose information
    // if you try plotting extremely large 64-bit integral types. Proceed with caution!

    /*JNI
        // For a proper type conversion, since C++ doesn't have a "long" type.
        #define long ImS64
        #define LEN(arg) (int)env->GetArrayLength(obj_##arg)
     */

    // values

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] values) {
        nPlotLine(labelId, values);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] values, final double xscale) {
        nPlotLine(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] values, final double xscale, final double xstart) {
        nPlotLine(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] values, final double xscale, final double xstart, final int flags) {
        nPlotLine(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLine(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotLine(String labelId, short[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, short[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, short[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, short[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, short[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] values) {
        nPlotLine(labelId, values);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] values, final double xscale) {
        nPlotLine(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] values, final double xscale, final double xstart) {
        nPlotLine(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] values, final double xscale, final double xstart, final int flags) {
        nPlotLine(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLine(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotLine(String labelId, int[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, int[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, int[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, int[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, int[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] values) {
        nPlotLine(labelId, values);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] values, final double xscale) {
        nPlotLine(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] values, final double xscale, final double xstart) {
        nPlotLine(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] values, final double xscale, final double xstart, final int flags) {
        nPlotLine(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLine(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotLine(String labelId, long[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, long[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, long[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, long[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, long[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] values) {
        nPlotLine(labelId, values);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] values, final double xscale) {
        nPlotLine(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] values, final double xscale, final double xstart) {
        nPlotLine(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] values, final double xscale, final double xstart, final int flags) {
        nPlotLine(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLine(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotLine(String labelId, float[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, float[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, float[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, float[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, float[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] values) {
        nPlotLine(labelId, values);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] values, final double xscale) {
        nPlotLine(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] values, final double xscale, final double xstart) {
        nPlotLine(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] values, final double xscale, final double xstart, final int flags) {
        nPlotLine(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLine(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotLine(String labelId, double[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, double[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, double[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, double[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, double[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] values, final int count) {
        nPlotLineV(labelId, values, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] values, final int count, final double xscale) {
        nPlotLineV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] values, final int count, final double xscale, final double xstart) {
        nPlotLineV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotLineV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, short[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, short[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, short[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, short[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] values, final int count) {
        nPlotLineV(labelId, values, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] values, final int count, final double xscale) {
        nPlotLineV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] values, final int count, final double xscale, final double xstart) {
        nPlotLineV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotLineV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, int[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, int[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, int[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, int[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] values, final int count) {
        nPlotLineV(labelId, values, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] values, final int count, final double xscale) {
        nPlotLineV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] values, final int count, final double xscale, final double xstart) {
        nPlotLineV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotLineV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, long[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, long[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, long[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, long[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] values, final int count) {
        nPlotLineV(labelId, values, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] values, final int count, final double xscale) {
        nPlotLineV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] values, final int count, final double xscale, final double xstart) {
        nPlotLineV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotLineV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, float[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, float[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, float[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, float[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] values, final int count) {
        nPlotLineV(labelId, values, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] values, final int count, final double xscale) {
        nPlotLineV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] values, final int count, final double xscale, final double xstart) {
        nPlotLineV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotLineV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotLineV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, double[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, double[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, double[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, double[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotLine(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    // xs,ys

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] xs, final short[] ys) {
        nPlotLine(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final short[] xs, final short[] ys, final int flags) {
        nPlotLine(labelId, xs, ys, flags);
    }

    private static native void nPlotLine(String labelId, short[] xs, short[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, short[] xs, short[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] xs, final int[] ys) {
        nPlotLine(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final int[] xs, final int[] ys, final int flags) {
        nPlotLine(labelId, xs, ys, flags);
    }

    private static native void nPlotLine(String labelId, int[] xs, int[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, int[] xs, int[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] xs, final long[] ys) {
        nPlotLine(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final long[] xs, final long[] ys, final int flags) {
        nPlotLine(labelId, xs, ys, flags);
    }

    private static native void nPlotLine(String labelId, long[] xs, long[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, long[] xs, long[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] xs, final float[] ys) {
        nPlotLine(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final float[] xs, final float[] ys, final int flags) {
        nPlotLine(labelId, xs, ys, flags);
    }

    private static native void nPlotLine(String labelId, float[] xs, float[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, float[] xs, float[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] xs, final double[] ys) {
        nPlotLine(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLine(final String labelId, final double[] xs, final double[] ys, final int flags) {
        nPlotLine(labelId, xs, ys, flags);
    }

    private static native void nPlotLine(String labelId, double[] xs, double[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLine(String labelId, double[] xs, double[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] xs, final short[] ys, final int count) {
        nPlotLineV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] xs, final short[] ys, final int count, final int flags) {
        nPlotLineV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final short[] xs, final short[] ys, final int count, final int flags, final int offset) {
        nPlotLineV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotLineV(String labelId, short[] xs, short[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, short[] xs, short[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, short[] xs, short[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] xs, final int[] ys, final int count) {
        nPlotLineV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] xs, final int[] ys, final int count, final int flags) {
        nPlotLineV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final int[] xs, final int[] ys, final int count, final int flags, final int offset) {
        nPlotLineV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotLineV(String labelId, int[] xs, int[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, int[] xs, int[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, int[] xs, int[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] xs, final long[] ys, final int count) {
        nPlotLineV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] xs, final long[] ys, final int count, final int flags) {
        nPlotLineV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final long[] xs, final long[] ys, final int count, final int flags, final int offset) {
        nPlotLineV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotLineV(String labelId, long[] xs, long[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, long[] xs, long[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, long[] xs, long[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] xs, final float[] ys, final int count) {
        nPlotLineV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] xs, final float[] ys, final int count, final int flags) {
        nPlotLineV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final float[] xs, final float[] ys, final int count, final int flags, final int offset) {
        nPlotLineV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotLineV(String labelId, float[] xs, float[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, float[] xs, float[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, float[] xs, float[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] xs, final double[] ys, final int count) {
        nPlotLineV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] xs, final double[] ys, final int count, final int flags) {
        nPlotLineV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D line plot.
     */
    public static void plotLineV(final String labelId, final double[] xs, final double[] ys, final int count, final int flags, final int offset) {
        nPlotLineV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotLineV(String labelId, double[] xs, double[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, double[] xs, double[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotLineV(String labelId, double[] xs, double[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotLine(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    // values

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] values) {
        nPlotScatter(labelId, values);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] values, final double xscale) {
        nPlotScatter(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] values, final double xscale, final double xstart) {
        nPlotScatter(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] values, final double xscale, final double xstart, final int flags) {
        nPlotScatter(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatter(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatter(String labelId, short[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, short[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, short[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, short[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, short[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] values) {
        nPlotScatter(labelId, values);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] values, final double xscale) {
        nPlotScatter(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] values, final double xscale, final double xstart) {
        nPlotScatter(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] values, final double xscale, final double xstart, final int flags) {
        nPlotScatter(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatter(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatter(String labelId, int[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, int[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, int[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, int[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, int[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] values) {
        nPlotScatter(labelId, values);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] values, final double xscale) {
        nPlotScatter(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] values, final double xscale, final double xstart) {
        nPlotScatter(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] values, final double xscale, final double xstart, final int flags) {
        nPlotScatter(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatter(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatter(String labelId, long[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, long[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, long[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, long[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, long[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] values) {
        nPlotScatter(labelId, values);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] values, final double xscale) {
        nPlotScatter(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] values, final double xscale, final double xstart) {
        nPlotScatter(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] values, final double xscale, final double xstart, final int flags) {
        nPlotScatter(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatter(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatter(String labelId, float[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, float[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, float[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, float[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, float[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] values) {
        nPlotScatter(labelId, values);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] values, final double xscale) {
        nPlotScatter(labelId, values, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] values, final double xscale, final double xstart) {
        nPlotScatter(labelId, values, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] values, final double xscale, final double xstart, final int flags) {
        nPlotScatter(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatter(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatter(String labelId, double[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, double[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, double[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, double[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, double[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] values, final int count) {
        nPlotScatterV(labelId, values, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] values, final int count, final double xscale) {
        nPlotScatterV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] values, final int count, final double xscale, final double xstart) {
        nPlotScatterV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, short[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, short[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, short[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, short[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] values, final int count) {
        nPlotScatterV(labelId, values, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] values, final int count, final double xscale) {
        nPlotScatterV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] values, final int count, final double xscale, final double xstart) {
        nPlotScatterV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, int[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, int[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, int[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, int[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] values, final int count) {
        nPlotScatterV(labelId, values, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] values, final int count, final double xscale) {
        nPlotScatterV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] values, final int count, final double xscale, final double xstart) {
        nPlotScatterV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, long[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, long[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, long[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, long[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] values, final int count) {
        nPlotScatterV(labelId, values, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] values, final int count, final double xscale) {
        nPlotScatterV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] values, final int count, final double xscale, final double xstart) {
        nPlotScatterV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, float[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, float[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, float[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, float[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] values, final int count) {
        nPlotScatterV(labelId, values, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] values, final int count, final double xscale) {
        nPlotScatterV(labelId, values, count, xscale);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] values, final int count, final double xscale, final double xstart) {
        nPlotScatterV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotScatterV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, double[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, double[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, double[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, double[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    // xs,ys

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] xs, final short[] ys) {
        nPlotScatter(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final short[] xs, final short[] ys, final int flags) {
        nPlotScatter(labelId, xs, ys, flags);
    }

    private static native void nPlotScatter(String labelId, short[] xs, short[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, short[] xs, short[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] xs, final int[] ys) {
        nPlotScatter(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final int[] xs, final int[] ys, final int flags) {
        nPlotScatter(labelId, xs, ys, flags);
    }

    private static native void nPlotScatter(String labelId, int[] xs, int[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, int[] xs, int[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] xs, final long[] ys) {
        nPlotScatter(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final long[] xs, final long[] ys, final int flags) {
        nPlotScatter(labelId, xs, ys, flags);
    }

    private static native void nPlotScatter(String labelId, long[] xs, long[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, long[] xs, long[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] xs, final float[] ys) {
        nPlotScatter(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final float[] xs, final float[] ys, final int flags) {
        nPlotScatter(labelId, xs, ys, flags);
    }

    private static native void nPlotScatter(String labelId, float[] xs, float[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, float[] xs, float[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] xs, final double[] ys) {
        nPlotScatter(labelId, xs, ys);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatter(final String labelId, final double[] xs, final double[] ys, final int flags) {
        nPlotScatter(labelId, xs, ys, flags);
    }

    private static native void nPlotScatter(String labelId, double[] xs, double[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatter(String labelId, double[] xs, double[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] xs, final short[] ys, final int count) {
        nPlotScatterV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] xs, final short[] ys, final int count, final int flags) {
        nPlotScatterV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final short[] xs, final short[] ys, final int count, final int flags, final int offset) {
        nPlotScatterV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, short[] xs, short[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, short[] xs, short[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, short[] xs, short[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] xs, final int[] ys, final int count) {
        nPlotScatterV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] xs, final int[] ys, final int count, final int flags) {
        nPlotScatterV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final int[] xs, final int[] ys, final int count, final int flags, final int offset) {
        nPlotScatterV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, int[] xs, int[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, int[] xs, int[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, int[] xs, int[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] xs, final long[] ys, final int count) {
        nPlotScatterV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] xs, final long[] ys, final int count, final int flags) {
        nPlotScatterV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final long[] xs, final long[] ys, final int count, final int flags, final int offset) {
        nPlotScatterV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, long[] xs, long[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, long[] xs, long[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, long[] xs, long[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] xs, final float[] ys, final int count) {
        nPlotScatterV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] xs, final float[] ys, final int count, final int flags) {
        nPlotScatterV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final float[] xs, final float[] ys, final int count, final int flags, final int offset) {
        nPlotScatterV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, float[] xs, float[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, float[] xs, float[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, float[] xs, float[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] xs, final double[] ys, final int count) {
        nPlotScatterV(labelId, xs, ys, count);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] xs, final double[] ys, final int count, final int flags) {
        nPlotScatterV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a standard 2D scatter plot. Default marker is ImPlotMarker_Circle.
     */
    public static void plotScatterV(final String labelId, final double[] xs, final double[] ys, final int count, final int flags, final int offset) {
        nPlotScatterV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotScatterV(String labelId, double[] xs, double[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, double[] xs, double[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotScatterV(String labelId, double[] xs, double[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotScatter(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    // values

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] values) {
        nPlotStairs(labelId, values);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] values, final double xscale) {
        nPlotStairs(labelId, values, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] values, final double xscale, final double xstart) {
        nPlotStairs(labelId, values, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] values, final double xscale, final double xstart, final int flags) {
        nPlotStairs(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairs(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairs(String labelId, short[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, short[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, short[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, short[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, short[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] values) {
        nPlotStairs(labelId, values);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] values, final double xscale) {
        nPlotStairs(labelId, values, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] values, final double xscale, final double xstart) {
        nPlotStairs(labelId, values, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] values, final double xscale, final double xstart, final int flags) {
        nPlotStairs(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairs(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairs(String labelId, int[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, int[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, int[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, int[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, int[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] values) {
        nPlotStairs(labelId, values);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] values, final double xscale) {
        nPlotStairs(labelId, values, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] values, final double xscale, final double xstart) {
        nPlotStairs(labelId, values, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] values, final double xscale, final double xstart, final int flags) {
        nPlotStairs(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairs(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairs(String labelId, long[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, long[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, long[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, long[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, long[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] values) {
        nPlotStairs(labelId, values);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] values, final double xscale) {
        nPlotStairs(labelId, values, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] values, final double xscale, final double xstart) {
        nPlotStairs(labelId, values, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] values, final double xscale, final double xstart, final int flags) {
        nPlotStairs(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairs(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairs(String labelId, float[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, float[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, float[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, float[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, float[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] values) {
        nPlotStairs(labelId, values);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] values, final double xscale) {
        nPlotStairs(labelId, values, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] values, final double xscale, final double xstart) {
        nPlotStairs(labelId, values, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] values, final double xscale, final double xstart, final int flags) {
        nPlotStairs(labelId, values, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] values, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairs(labelId, values, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairs(String labelId, double[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, double[] values, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, double[] values, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, double[] values, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, double[] values, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], LEN(values), xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] values, final int count) {
        nPlotStairsV(labelId, values, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] values, final int count, final double xscale) {
        nPlotStairsV(labelId, values, count, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] values, final int count, final double xscale, final double xstart) {
        nPlotStairsV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, short[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, short[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, short[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, short[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] values, final int count) {
        nPlotStairsV(labelId, values, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] values, final int count, final double xscale) {
        nPlotStairsV(labelId, values, count, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] values, final int count, final double xscale, final double xstart) {
        nPlotStairsV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, int[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, int[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, int[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, int[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] values, final int count) {
        nPlotStairsV(labelId, values, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] values, final int count, final double xscale) {
        nPlotStairsV(labelId, values, count, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] values, final int count, final double xscale, final double xstart) {
        nPlotStairsV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, long[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, long[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, long[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, long[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] values, final int count) {
        nPlotStairsV(labelId, values, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] values, final int count, final double xscale) {
        nPlotStairsV(labelId, values, count, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] values, final int count, final double xscale, final double xstart) {
        nPlotStairsV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, float[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, float[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, float[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, float[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] values, final int count) {
        nPlotStairsV(labelId, values, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] values, final int count, final double xscale) {
        nPlotStairsV(labelId, values, count, xscale);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] values, final int count, final double xscale, final double xstart) {
        nPlotStairsV(labelId, values, count, xscale, xstart);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] values, final int count, final double xscale, final double xstart, final int flags) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] values, final int count, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStairsV(labelId, values, count, xscale, xstart, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, double[] values, int count, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, double[] values, int count, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, double[] values, int count, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, double[] values, int count, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &values[0], count, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    // xs,ys

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] xs, final short[] ys) {
        nPlotStairs(labelId, xs, ys);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final short[] xs, final short[] ys, final int flags) {
        nPlotStairs(labelId, xs, ys, flags);
    }

    private static native void nPlotStairs(String labelId, short[] xs, short[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, short[] xs, short[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] xs, final int[] ys) {
        nPlotStairs(labelId, xs, ys);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final int[] xs, final int[] ys, final int flags) {
        nPlotStairs(labelId, xs, ys, flags);
    }

    private static native void nPlotStairs(String labelId, int[] xs, int[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, int[] xs, int[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] xs, final long[] ys) {
        nPlotStairs(labelId, xs, ys);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final long[] xs, final long[] ys, final int flags) {
        nPlotStairs(labelId, xs, ys, flags);
    }

    private static native void nPlotStairs(String labelId, long[] xs, long[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, long[] xs, long[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] xs, final float[] ys) {
        nPlotStairs(labelId, xs, ys);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final float[] xs, final float[] ys, final int flags) {
        nPlotStairs(labelId, xs, ys, flags);
    }

    private static native void nPlotStairs(String labelId, float[] xs, float[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, float[] xs, float[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] xs, final double[] ys) {
        nPlotStairs(labelId, xs, ys);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairs(final String labelId, final double[] xs, final double[] ys, final int flags) {
        nPlotStairs(labelId, xs, ys, flags);
    }

    private static native void nPlotStairs(String labelId, double[] xs, double[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairs(String labelId, double[] xs, double[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] xs, final short[] ys, final int count) {
        nPlotStairsV(labelId, xs, ys, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] xs, final short[] ys, final int count, final int flags) {
        nPlotStairsV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final short[] xs, final short[] ys, final int count, final int flags, final int offset) {
        nPlotStairsV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, short[] xs, short[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, short[] xs, short[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, short[] xs, short[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] xs, final int[] ys, final int count) {
        nPlotStairsV(labelId, xs, ys, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] xs, final int[] ys, final int count, final int flags) {
        nPlotStairsV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final int[] xs, final int[] ys, final int count, final int flags, final int offset) {
        nPlotStairsV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, int[] xs, int[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, int[] xs, int[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, int[] xs, int[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] xs, final long[] ys, final int count) {
        nPlotStairsV(labelId, xs, ys, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] xs, final long[] ys, final int count, final int flags) {
        nPlotStairsV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final long[] xs, final long[] ys, final int count, final int flags, final int offset) {
        nPlotStairsV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, long[] xs, long[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, long[] xs, long[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, long[] xs, long[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] xs, final float[] ys, final int count) {
        nPlotStairsV(labelId, xs, ys, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] xs, final float[] ys, final int count, final int flags) {
        nPlotStairsV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final float[] xs, final float[] ys, final int count, final int flags, final int offset) {
        nPlotStairsV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, float[] xs, float[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, float[] xs, float[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, float[] xs, float[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] xs, final double[] ys, final int count) {
        nPlotStairsV(labelId, xs, ys, count);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] xs, final double[] ys, final int count, final int flags) {
        nPlotStairsV(labelId, xs, ys, count, flags);
    }

    /**
     * Plots a stairstep graph. The y value is continued constantly from every x position, i.e. the interval [x[i], x[i+1]) has the value y[i].
     */
    public static void plotStairsV(final String labelId, final double[] xs, final double[] ys, final int count, final int flags, final int offset) {
        nPlotStairsV(labelId, xs, ys, count, flags, offset);
    }

    private static native void nPlotStairsV(String labelId, double[] xs, double[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, double[] xs, double[] ys, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStairsV(String labelId, double[] xs, double[] ys, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStairs(labelId, &xs[0], &ys[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    // values

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] values) {
        nPlotShaded(labelId, values);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] values, final double yRef) {
        nPlotShaded(labelId, values, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] values, final double yRef, final double xscale) {
        nPlotShaded(labelId, values, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] values, final double yRef, final double xscale, final double xstart) {
        nPlotShaded(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShaded(String labelId, short[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] values) {
        nPlotShaded(labelId, values);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] values, final double yRef) {
        nPlotShaded(labelId, values, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] values, final double yRef, final double xscale) {
        nPlotShaded(labelId, values, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] values, final double yRef, final double xscale, final double xstart) {
        nPlotShaded(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShaded(String labelId, int[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] values) {
        nPlotShaded(labelId, values);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] values, final double yRef) {
        nPlotShaded(labelId, values, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] values, final double yRef, final double xscale) {
        nPlotShaded(labelId, values, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] values, final double yRef, final double xscale, final double xstart) {
        nPlotShaded(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShaded(String labelId, long[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] values) {
        nPlotShaded(labelId, values);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] values, final double yRef) {
        nPlotShaded(labelId, values, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] values, final double yRef, final double xscale) {
        nPlotShaded(labelId, values, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] values, final double yRef, final double xscale, final double xstart) {
        nPlotShaded(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShaded(String labelId, float[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] values) {
        nPlotShaded(labelId, values);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] values, final double yRef) {
        nPlotShaded(labelId, values, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] values, final double yRef, final double xscale) {
        nPlotShaded(labelId, values, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] values, final double yRef, final double xscale, final double xstart) {
        nPlotShaded(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShaded(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShaded(String labelId, double[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] values, final int count) {
        nPlotShadedV(labelId, values, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] values, final int count, final double yRef) {
        nPlotShadedV(labelId, values, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] values, final int count, final double yRef, final double xscale) {
        nPlotShadedV(labelId, values, count, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] values, final int count, final double yRef, final double xscale, final double xstart) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] values, int count, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] values, int count, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] values, int count, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] values, int count, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] values, final int count) {
        nPlotShadedV(labelId, values, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] values, final int count, final double yRef) {
        nPlotShadedV(labelId, values, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] values, final int count, final double yRef, final double xscale) {
        nPlotShadedV(labelId, values, count, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] values, final int count, final double yRef, final double xscale, final double xstart) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] values, int count, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] values, int count, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] values, int count, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] values, int count, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] values, final int count) {
        nPlotShadedV(labelId, values, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] values, final int count, final double yRef) {
        nPlotShadedV(labelId, values, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] values, final int count, final double yRef, final double xscale) {
        nPlotShadedV(labelId, values, count, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] values, final int count, final double yRef, final double xscale, final double xstart) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] values, int count, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] values, int count, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] values, int count, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] values, int count, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] values, final int count) {
        nPlotShadedV(labelId, values, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] values, final int count, final double yRef) {
        nPlotShadedV(labelId, values, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] values, final int count, final double yRef, final double xscale) {
        nPlotShadedV(labelId, values, count, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] values, final int count, final double yRef, final double xscale, final double xstart) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] values, int count, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] values, int count, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] values, int count, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] values, int count, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] values, final int count) {
        nPlotShadedV(labelId, values, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] values, final int count, final double yRef) {
        nPlotShadedV(labelId, values, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] values, final int count, final double yRef, final double xscale) {
        nPlotShadedV(labelId, values, count, yRef, xscale);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] values, final int count, final double yRef, final double xscale, final double xstart) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] values, final int count, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotShadedV(labelId, values, count, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] values, int count, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] values, int count, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] values, int count, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] values, int count, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &values[0], count, yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    // xs,ys

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] xs, final short[] ys) {
        nPlotShaded(labelId, xs, ys);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] xs, final short[] ys, final double yRef) {
        nPlotShaded(labelId, xs, ys, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] xs, final short[] ys, final double yRef, final int flags) {
        nPlotShaded(labelId, xs, ys, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] xs, final short[] ys, final double yRef, final int flags, final int offset) {
        nPlotShaded(labelId, xs, ys, yRef, flags, offset);
    }

    private static native void nPlotShaded(String labelId, short[] xs, short[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] xs, short[] ys, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] xs, short[] ys, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] xs, short[] ys, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] xs, final int[] ys) {
        nPlotShaded(labelId, xs, ys);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] xs, final int[] ys, final double yRef) {
        nPlotShaded(labelId, xs, ys, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] xs, final int[] ys, final double yRef, final int flags) {
        nPlotShaded(labelId, xs, ys, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] xs, final int[] ys, final double yRef, final int flags, final int offset) {
        nPlotShaded(labelId, xs, ys, yRef, flags, offset);
    }

    private static native void nPlotShaded(String labelId, int[] xs, int[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] xs, int[] ys, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] xs, int[] ys, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] xs, int[] ys, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] xs, final long[] ys) {
        nPlotShaded(labelId, xs, ys);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] xs, final long[] ys, final double yRef) {
        nPlotShaded(labelId, xs, ys, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] xs, final long[] ys, final double yRef, final int flags) {
        nPlotShaded(labelId, xs, ys, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] xs, final long[] ys, final double yRef, final int flags, final int offset) {
        nPlotShaded(labelId, xs, ys, yRef, flags, offset);
    }

    private static native void nPlotShaded(String labelId, long[] xs, long[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] xs, long[] ys, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] xs, long[] ys, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] xs, long[] ys, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] xs, final float[] ys) {
        nPlotShaded(labelId, xs, ys);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] xs, final float[] ys, final double yRef) {
        nPlotShaded(labelId, xs, ys, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] xs, final float[] ys, final double yRef, final int flags) {
        nPlotShaded(labelId, xs, ys, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] xs, final float[] ys, final double yRef, final int flags, final int offset) {
        nPlotShaded(labelId, xs, ys, yRef, flags, offset);
    }

    private static native void nPlotShaded(String labelId, float[] xs, float[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] xs, float[] ys, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] xs, float[] ys, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] xs, float[] ys, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] xs, final double[] ys) {
        nPlotShaded(labelId, xs, ys);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] xs, final double[] ys, final double yRef) {
        nPlotShaded(labelId, xs, ys, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] xs, final double[] ys, final double yRef, final int flags) {
        nPlotShaded(labelId, xs, ys, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] xs, final double[] ys, final double yRef, final int flags, final int offset) {
        nPlotShaded(labelId, xs, ys, yRef, flags, offset);
    }

    private static native void nPlotShaded(String labelId, double[] xs, double[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] xs, double[] ys, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] xs, double[] ys, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] xs, double[] ys, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], LEN(xs), yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys, final int count) {
        nPlotShadedV(labelId, xs, ys, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys, final int count, final double yRef) {
        nPlotShadedV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys, final int count, final double yRef, final int flags) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys, final int count) {
        nPlotShadedV(labelId, xs, ys, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys, final int count, final double yRef) {
        nPlotShadedV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys, final int count, final double yRef, final int flags) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys, final int count) {
        nPlotShadedV(labelId, xs, ys, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys, final int count, final double yRef) {
        nPlotShadedV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys, final int count, final double yRef, final int flags) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys, final int count) {
        nPlotShadedV(labelId, xs, ys, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys, final int count, final double yRef) {
        nPlotShadedV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys, final int count, final double yRef, final int flags) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys, final int count) {
        nPlotShadedV(labelId, xs, ys, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys, final int count, final double yRef) {
        nPlotShadedV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys, final int count, final double yRef, final int flags) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    // xs,ys1,ys2

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] xs, final short[] ys1, final short[] ys2) {
        nPlotShaded(labelId, xs, ys1, ys2);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final short[] xs, final short[] ys1, final short[] ys2, final int flags) {
        nPlotShaded(labelId, xs, ys1, ys2, flags);
    }

    private static native void nPlotShaded(String labelId, short[] xs, short[] ys1, short[] ys2); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, short[] xs, short[] ys1, short[] ys2, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] xs, final int[] ys1, final int[] ys2) {
        nPlotShaded(labelId, xs, ys1, ys2);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final int[] xs, final int[] ys1, final int[] ys2, final int flags) {
        nPlotShaded(labelId, xs, ys1, ys2, flags);
    }

    private static native void nPlotShaded(String labelId, int[] xs, int[] ys1, int[] ys2); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, int[] xs, int[] ys1, int[] ys2, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] xs, final long[] ys1, final long[] ys2) {
        nPlotShaded(labelId, xs, ys1, ys2);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final long[] xs, final long[] ys1, final long[] ys2, final int flags) {
        nPlotShaded(labelId, xs, ys1, ys2, flags);
    }

    private static native void nPlotShaded(String labelId, long[] xs, long[] ys1, long[] ys2); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, long[] xs, long[] ys1, long[] ys2, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] xs, final float[] ys1, final float[] ys2) {
        nPlotShaded(labelId, xs, ys1, ys2);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final float[] xs, final float[] ys1, final float[] ys2, final int flags) {
        nPlotShaded(labelId, xs, ys1, ys2, flags);
    }

    private static native void nPlotShaded(String labelId, float[] xs, float[] ys1, float[] ys2); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, float[] xs, float[] ys1, float[] ys2, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] xs, final double[] ys1, final double[] ys2) {
        nPlotShaded(labelId, xs, ys1, ys2);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShaded(final String labelId, final double[] xs, final double[] ys1, final double[] ys2, final int flags) {
        nPlotShaded(labelId, xs, ys1, ys2, flags);
    }

    private static native void nPlotShaded(String labelId, double[] xs, double[] ys1, double[] ys2); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShaded(String labelId, double[] xs, double[] ys1, double[] ys2, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys1, final short[] ys2, final int count) {
        nPlotShadedV(labelId, xs, ys1, ys2, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys1, final short[] ys2, final int count, final int flags) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final short[] xs, final short[] ys1, final short[] ys2, final int count, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys1, short[] ys2, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys1, short[] ys2, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, short[] xs, short[] ys1, short[] ys2, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys1, final int[] ys2, final int count) {
        nPlotShadedV(labelId, xs, ys1, ys2, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys1, final int[] ys2, final int count, final int flags) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final int[] xs, final int[] ys1, final int[] ys2, final int count, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys1, int[] ys2, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys1, int[] ys2, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, int[] xs, int[] ys1, int[] ys2, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys1, final long[] ys2, final int count) {
        nPlotShadedV(labelId, xs, ys1, ys2, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys1, final long[] ys2, final int count, final int flags) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final long[] xs, final long[] ys1, final long[] ys2, final int count, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys1, long[] ys2, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys1, long[] ys2, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, long[] xs, long[] ys1, long[] ys2, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys1, final float[] ys2, final int count) {
        nPlotShadedV(labelId, xs, ys1, ys2, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys1, final float[] ys2, final int count, final int flags) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final float[] xs, final float[] ys1, final float[] ys2, final int count, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys1, float[] ys2, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys1, float[] ys2, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, float[] xs, float[] ys1, float[] ys2, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys1, final double[] ys2, final int count) {
        nPlotShadedV(labelId, xs, ys1, ys2, count);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys1, final double[] ys2, final int count, final int flags) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags);
    }

    /**
     * Plots a shaded (filled) region between two lines, or a line and a horizontal reference. Set y_ref to +/-INFINITY for infinite fill extents.
     */
    public static void plotShadedV(final String labelId, final double[] xs, final double[] ys1, final double[] ys2, final int count, final int flags, final int offset) {
        nPlotShadedV(labelId, xs, ys1, ys2, count, flags, offset);
    }

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys1, double[] ys2, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys1, double[] ys2, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    private static native void nPlotShadedV(String labelId, double[] xs, double[] ys1, double[] ys2, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys1 = obj_ys1 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys1, JNI_FALSE);
        auto ys2 = obj_ys2 == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys2, JNI_FALSE);
        ImPlot::PlotShaded(labelId, &xs[0], &ys1[0], &ys2[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys1 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys1, ys1, JNI_FALSE);
        if (ys2 != NULL) env->ReleasePrimitiveArrayCritical(obj_ys2, ys2, JNI_FALSE);
    */

    // values

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] values) {
        nPlotBars(labelId, values);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] values, final double barWidth) {
        nPlotBars(labelId, values, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] values, final double barWidth, final double xstart) {
        nPlotBars(labelId, values, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] values, final double barWidth, final double xstart, final int flags) {
        nPlotBars(labelId, values, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] values, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBars(labelId, values, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBars(String labelId, short[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, short[] values, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, short[] values, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, short[] values, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, short[] values, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] values) {
        nPlotBars(labelId, values);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] values, final double barWidth) {
        nPlotBars(labelId, values, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] values, final double barWidth, final double xstart) {
        nPlotBars(labelId, values, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] values, final double barWidth, final double xstart, final int flags) {
        nPlotBars(labelId, values, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] values, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBars(labelId, values, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBars(String labelId, int[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, int[] values, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, int[] values, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, int[] values, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, int[] values, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] values) {
        nPlotBars(labelId, values);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] values, final double barWidth) {
        nPlotBars(labelId, values, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] values, final double barWidth, final double xstart) {
        nPlotBars(labelId, values, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] values, final double barWidth, final double xstart, final int flags) {
        nPlotBars(labelId, values, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] values, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBars(labelId, values, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBars(String labelId, long[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, long[] values, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, long[] values, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, long[] values, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, long[] values, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] values) {
        nPlotBars(labelId, values);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] values, final double barWidth) {
        nPlotBars(labelId, values, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] values, final double barWidth, final double xstart) {
        nPlotBars(labelId, values, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] values, final double barWidth, final double xstart, final int flags) {
        nPlotBars(labelId, values, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] values, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBars(labelId, values, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBars(String labelId, float[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, float[] values, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, float[] values, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, float[] values, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, float[] values, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] values) {
        nPlotBars(labelId, values);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] values, final double barWidth) {
        nPlotBars(labelId, values, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] values, final double barWidth, final double xstart) {
        nPlotBars(labelId, values, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] values, final double barWidth, final double xstart, final int flags) {
        nPlotBars(labelId, values, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] values, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBars(labelId, values, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBars(String labelId, double[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, double[] values, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, double[] values, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, double[] values, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, double[] values, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], LEN(values), barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] values, final int count) {
        nPlotBarsV(labelId, values, count);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] values, final int count, final double barWidth) {
        nPlotBarsV(labelId, values, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] values, final int count, final double barWidth, final double xstart) {
        nPlotBarsV(labelId, values, count, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] values, final int count, final double barWidth, final double xstart, final int flags) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] values, final int count, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] values, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] values, int count, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] values, int count, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] values, int count, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] values, final int count) {
        nPlotBarsV(labelId, values, count);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] values, final int count, final double barWidth) {
        nPlotBarsV(labelId, values, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] values, final int count, final double barWidth, final double xstart) {
        nPlotBarsV(labelId, values, count, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] values, final int count, final double barWidth, final double xstart, final int flags) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] values, final int count, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] values, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] values, int count, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] values, int count, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] values, int count, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] values, final int count) {
        nPlotBarsV(labelId, values, count);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] values, final int count, final double barWidth) {
        nPlotBarsV(labelId, values, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] values, final int count, final double barWidth, final double xstart) {
        nPlotBarsV(labelId, values, count, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] values, final int count, final double barWidth, final double xstart, final int flags) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] values, final int count, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] values, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] values, int count, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] values, int count, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] values, int count, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] values, final int count) {
        nPlotBarsV(labelId, values, count);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] values, final int count, final double barWidth) {
        nPlotBarsV(labelId, values, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] values, final int count, final double barWidth, final double xstart) {
        nPlotBarsV(labelId, values, count, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] values, final int count, final double barWidth, final double xstart, final int flags) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] values, final int count, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] values, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] values, int count, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] values, int count, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] values, int count, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] values, final int count) {
        nPlotBarsV(labelId, values, count);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] values, final int count, final double barWidth) {
        nPlotBarsV(labelId, values, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] values, final int count, final double barWidth, final double xstart) {
        nPlotBarsV(labelId, values, count, barWidth, xstart);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] values, final int count, final double barWidth, final double xstart, final int flags) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] values, final int count, final double barWidth, final double xstart, final int flags, final int offset) {
        nPlotBarsV(labelId, values, count, barWidth, xstart, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] values, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] values, int count, double barWidth, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] values, int count, double barWidth, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] values, int count, double barWidth, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBars(labelId, &values[0], count, barWidth, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    // xs,ys

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] xs, final short[] ys) {
        nPlotBars(labelId, xs, ys);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] xs, final short[] ys, final int flags) {
        nPlotBars(labelId, xs, ys, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final short[] xs, final short[] ys, final int flags, final int offset) {
        nPlotBars(labelId, xs, ys, flags, offset);
    }

    private static native void nPlotBars(String labelId, short[] xs, short[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, short[] xs, short[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, short[] xs, short[] ys, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] xs, final int[] ys) {
        nPlotBars(labelId, xs, ys);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] xs, final int[] ys, final int flags) {
        nPlotBars(labelId, xs, ys, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final int[] xs, final int[] ys, final int flags, final int offset) {
        nPlotBars(labelId, xs, ys, flags, offset);
    }

    private static native void nPlotBars(String labelId, int[] xs, int[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, int[] xs, int[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, int[] xs, int[] ys, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] xs, final long[] ys) {
        nPlotBars(labelId, xs, ys);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] xs, final long[] ys, final int flags) {
        nPlotBars(labelId, xs, ys, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final long[] xs, final long[] ys, final int flags, final int offset) {
        nPlotBars(labelId, xs, ys, flags, offset);
    }

    private static native void nPlotBars(String labelId, long[] xs, long[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, long[] xs, long[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, long[] xs, long[] ys, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] xs, final float[] ys) {
        nPlotBars(labelId, xs, ys);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] xs, final float[] ys, final int flags) {
        nPlotBars(labelId, xs, ys, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final float[] xs, final float[] ys, final int flags, final int offset) {
        nPlotBars(labelId, xs, ys, flags, offset);
    }

    private static native void nPlotBars(String labelId, float[] xs, float[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, float[] xs, float[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, float[] xs, float[] ys, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] xs, final double[] ys) {
        nPlotBars(labelId, xs, ys);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] xs, final double[] ys, final int flags) {
        nPlotBars(labelId, xs, ys, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBars(final String labelId, final double[] xs, final double[] ys, final int flags, final int offset) {
        nPlotBars(labelId, xs, ys, flags, offset);
    }

    private static native void nPlotBars(String labelId, double[] xs, double[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, double[] xs, double[] ys, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBars(String labelId, double[] xs, double[] ys, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), 0.67, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] xs, final short[] ys, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] xs, final short[] ys, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] xs, final short[] ys, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, short[] xs, short[] ys, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] xs, short[] ys, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] xs, short[] ys, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] xs, final int[] ys, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] xs, final int[] ys, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] xs, final int[] ys, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, int[] xs, int[] ys, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] xs, int[] ys, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] xs, int[] ys, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] xs, final long[] ys, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] xs, final long[] ys, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] xs, final long[] ys, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, long[] xs, long[] ys, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] xs, long[] ys, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] xs, long[] ys, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] xs, final float[] ys, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] xs, final float[] ys, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] xs, final float[] ys, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, float[] xs, float[] ys, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] xs, float[] ys, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] xs, float[] ys, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] xs, final double[] ys, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] xs, final double[] ys, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] xs, final double[] ys, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, double[] xs, double[] ys, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] xs, double[] ys, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] xs, double[] ys, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], LEN(xs), barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] xs, final short[] ys, final int count, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] xs, final short[] ys, final int count, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final short[] xs, final short[] ys, final int count, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, short[] xs, short[] ys, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] xs, short[] ys, int count, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, short[] xs, short[] ys, int count, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] xs, final int[] ys, final int count, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] xs, final int[] ys, final int count, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final int[] xs, final int[] ys, final int count, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, int[] xs, int[] ys, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] xs, int[] ys, int count, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, int[] xs, int[] ys, int count, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] xs, final long[] ys, final int count, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] xs, final long[] ys, final int count, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final long[] xs, final long[] ys, final int count, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, long[] xs, long[] ys, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] xs, long[] ys, int count, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, long[] xs, long[] ys, int count, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] xs, final float[] ys, final int count, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] xs, final float[] ys, final int count, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final float[] xs, final float[] ys, final int count, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, float[] xs, float[] ys, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] xs, float[] ys, int count, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, float[] xs, float[] ys, int count, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] xs, final double[] ys, final int count, final double barWidth) {
        nPlotBarsV(labelId, xs, ys, count, barWidth);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] xs, final double[] ys, final int count, final double barWidth, final int flags) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags);
    }

    /**
     * Plots a vertical bar graph. #bar_width and #xstart are in X units.
     */
    public static void plotBarsV(final String labelId, final double[] xs, final double[] ys, final int count, final double barWidth, final int flags, final int offset) {
        nPlotBarsV(labelId, xs, ys, count, barWidth, flags, offset);
    }

    private static native void nPlotBarsV(String labelId, double[] xs, double[] ys, int count, double barWidth); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] xs, double[] ys, int count, double barWidth, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotBarsV(String labelId, double[] xs, double[] ys, int count, double barWidth, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotBars(labelId, &xs[0], &ys[0], count, barWidth, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final short[] values, final int groupCount) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final short[] values, final int groupCount, final double groupSize) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final short[] values, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final short[] values, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, short[] values, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, short[] values, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, short[] values, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, short[] values, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final int[] values, final int groupCount) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final int[] values, final int groupCount, final double groupSize) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final int[] values, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final int[] values, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, int[] values, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, int[] values, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, int[] values, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, int[] values, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final long[] values, final int groupCount) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final long[] values, final int groupCount, final double groupSize) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final long[] values, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final long[] values, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, long[] values, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, long[] values, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, long[] values, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, long[] values, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final float[] values, final int groupCount) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final float[] values, final int groupCount, final double groupSize) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final float[] values, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final float[] values, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, float[] values, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, float[] values, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, float[] values, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, float[] values, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final double[] values, final int groupCount) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final double[] values, final int groupCount, final double groupSize) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final double[] values, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroups(final String[] labelIds, final double[] values, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroups(labelIds, labelIds.length, values, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, double[] values, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, double[] values, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, double[] values, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroups(String[] obj_labelIds, int labelIdsCount, double[] values, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], LEN(values), groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final short[] values, final int itemCount, final int groupCount) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final short[] values, final int itemCount, final int groupCount, final double groupSize) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final short[] values, final int itemCount, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final short[] values, final int itemCount, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, short[] values, int itemCount, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, short[] values, int itemCount, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, short[] values, int itemCount, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, short[] values, int itemCount, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final int[] values, final int itemCount, final int groupCount) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final int[] values, final int itemCount, final int groupCount, final double groupSize) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final int[] values, final int itemCount, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final int[] values, final int itemCount, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, int[] values, int itemCount, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, int[] values, int itemCount, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, int[] values, int itemCount, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, int[] values, int itemCount, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final long[] values, final int itemCount, final int groupCount) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final long[] values, final int itemCount, final int groupCount, final double groupSize) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final long[] values, final int itemCount, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final long[] values, final int itemCount, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, long[] values, int itemCount, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, long[] values, int itemCount, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, long[] values, int itemCount, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, long[] values, int itemCount, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final float[] values, final int itemCount, final int groupCount) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final float[] values, final int itemCount, final int groupCount, final double groupSize) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final float[] values, final int itemCount, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final float[] values, final int itemCount, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, float[] values, int itemCount, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, float[] values, int itemCount, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, float[] values, int itemCount, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, float[] values, int itemCount, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final double[] values, final int itemCount, final int groupCount) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final double[] values, final int itemCount, final int groupCount, final double groupSize) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final double[] values, final int itemCount, final int groupCount, final double groupSize, final double shift) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift);
    }

    /**
     * Plots a group of vertical bars. #values is a row-major matrix with #item_count rows and #group_count cols. #label_ids should have #item_count elements.
     */
    public static void plotBarGroupsV(final String[] labelIds, final double[] values, final int itemCount, final int groupCount, final double groupSize, final double shift, final int flags) {
        nPlotBarGroupsV(labelIds, labelIds.length, values, itemCount, groupCount, groupSize, shift, flags);
    }

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, double[] values, int itemCount, int groupCount); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, double[] values, int itemCount, int groupCount, double groupSize); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, double[] values, int itemCount, int groupCount, double groupSize, double shift); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotBarGroupsV(String[] obj_labelIds, int labelIdsCount, double[] values, int itemCount, int groupCount, double groupSize, double shift, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotBarGroups(labelIds, &values[0], itemCount, groupCount, groupSize, shift, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final short[] xs, final short[] ys, final short[] err) {
        nPlotErrorBars(labelId, xs, ys, err);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final short[] xs, final short[] ys, final short[] err, final int flags) {
        nPlotErrorBars(labelId, xs, ys, err, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final short[] xs, final short[] ys, final short[] err, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, err, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, short[] xs, short[] ys, short[] err); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, short[] xs, short[] ys, short[] err, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, short[] xs, short[] ys, short[] err, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final int[] xs, final int[] ys, final int[] err) {
        nPlotErrorBars(labelId, xs, ys, err);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final int[] xs, final int[] ys, final int[] err, final int flags) {
        nPlotErrorBars(labelId, xs, ys, err, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final int[] xs, final int[] ys, final int[] err, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, err, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, int[] xs, int[] ys, int[] err); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, int[] xs, int[] ys, int[] err, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, int[] xs, int[] ys, int[] err, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final long[] xs, final long[] ys, final long[] err) {
        nPlotErrorBars(labelId, xs, ys, err);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final long[] xs, final long[] ys, final long[] err, final int flags) {
        nPlotErrorBars(labelId, xs, ys, err, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final long[] xs, final long[] ys, final long[] err, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, err, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, long[] xs, long[] ys, long[] err); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, long[] xs, long[] ys, long[] err, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, long[] xs, long[] ys, long[] err, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final float[] xs, final float[] ys, final float[] err) {
        nPlotErrorBars(labelId, xs, ys, err);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final float[] xs, final float[] ys, final float[] err, final int flags) {
        nPlotErrorBars(labelId, xs, ys, err, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final float[] xs, final float[] ys, final float[] err, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, err, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, float[] xs, float[] ys, float[] err); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, float[] xs, float[] ys, float[] err, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, float[] xs, float[] ys, float[] err, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final double[] xs, final double[] ys, final double[] err) {
        nPlotErrorBars(labelId, xs, ys, err);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final double[] xs, final double[] ys, final double[] err, final int flags) {
        nPlotErrorBars(labelId, xs, ys, err, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final double[] xs, final double[] ys, final double[] err, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, err, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, double[] xs, double[] ys, double[] err); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, double[] xs, double[] ys, double[] err, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, double[] xs, double[] ys, double[] err, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final short[] xs, final short[] ys, final short[] err, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, err, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final short[] xs, final short[] ys, final short[] err, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final short[] xs, final short[] ys, final short[] err, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, short[] xs, short[] ys, short[] err, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, short[] xs, short[] ys, short[] err, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, short[] xs, short[] ys, short[] err, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final int[] xs, final int[] ys, final int[] err, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, err, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final int[] xs, final int[] ys, final int[] err, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final int[] xs, final int[] ys, final int[] err, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, int[] xs, int[] ys, int[] err, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, int[] xs, int[] ys, int[] err, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, int[] xs, int[] ys, int[] err, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final long[] xs, final long[] ys, final long[] err, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, err, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final long[] xs, final long[] ys, final long[] err, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final long[] xs, final long[] ys, final long[] err, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, long[] xs, long[] ys, long[] err, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, long[] xs, long[] ys, long[] err, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, long[] xs, long[] ys, long[] err, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final float[] xs, final float[] ys, final float[] err, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, err, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final float[] xs, final float[] ys, final float[] err, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final float[] xs, final float[] ys, final float[] err, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, float[] xs, float[] ys, float[] err, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, float[] xs, float[] ys, float[] err, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, float[] xs, float[] ys, float[] err, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final double[] xs, final double[] ys, final double[] err, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, err, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final double[] xs, final double[] ys, final double[] err, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final double[] xs, final double[] ys, final double[] err, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, err, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, double[] xs, double[] ys, double[] err, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, double[] xs, double[] ys, double[] err, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, double[] xs, double[] ys, double[] err, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto err = obj_err == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_err, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &err[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (err != NULL) env->ReleasePrimitiveArrayCritical(obj_err, err, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final short[] xs, final short[] ys, final short[] neg, final short[] pos) {
        nPlotErrorBars(labelId, xs, ys, neg, pos);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final short[] xs, final short[] ys, final short[] neg, final short[] pos, final int flags) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final short[] xs, final short[] ys, final short[] neg, final short[] pos, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, short[] xs, short[] ys, short[] neg, short[] pos); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, short[] xs, short[] ys, short[] neg, short[] pos, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, short[] xs, short[] ys, short[] neg, short[] pos, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final int[] xs, final int[] ys, final int[] neg, final int[] pos) {
        nPlotErrorBars(labelId, xs, ys, neg, pos);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final int[] xs, final int[] ys, final int[] neg, final int[] pos, final int flags) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final int[] xs, final int[] ys, final int[] neg, final int[] pos, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, int[] xs, int[] ys, int[] neg, int[] pos); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, int[] xs, int[] ys, int[] neg, int[] pos, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, int[] xs, int[] ys, int[] neg, int[] pos, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final long[] xs, final long[] ys, final long[] neg, final long[] pos) {
        nPlotErrorBars(labelId, xs, ys, neg, pos);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final long[] xs, final long[] ys, final long[] neg, final long[] pos, final int flags) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final long[] xs, final long[] ys, final long[] neg, final long[] pos, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, long[] xs, long[] ys, long[] neg, long[] pos); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, long[] xs, long[] ys, long[] neg, long[] pos, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, long[] xs, long[] ys, long[] neg, long[] pos, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final float[] xs, final float[] ys, final float[] neg, final float[] pos) {
        nPlotErrorBars(labelId, xs, ys, neg, pos);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final float[] xs, final float[] ys, final float[] neg, final float[] pos, final int flags) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final float[] xs, final float[] ys, final float[] neg, final float[] pos, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, float[] xs, float[] ys, float[] neg, float[] pos); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, float[] xs, float[] ys, float[] neg, float[] pos, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, float[] xs, float[] ys, float[] neg, float[] pos, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final double[] xs, final double[] ys, final double[] neg, final double[] pos) {
        nPlotErrorBars(labelId, xs, ys, neg, pos);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final double[] xs, final double[] ys, final double[] neg, final double[] pos, final int flags) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBars(final String labelId, final double[] xs, final double[] ys, final double[] neg, final double[] pos, final int flags, final int offset) {
        nPlotErrorBars(labelId, xs, ys, neg, pos, flags, offset);
    }

    private static native void nPlotErrorBars(String labelId, double[] xs, double[] ys, double[] neg, double[] pos); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, double[] xs, double[] ys, double[] neg, double[] pos, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBars(String labelId, double[] xs, double[] ys, double[] neg, double[] pos, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], LEN(xs), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final short[] xs, final short[] ys, final short[] neg, final short[] pos, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final short[] xs, final short[] ys, final short[] neg, final short[] pos, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final short[] xs, final short[] ys, final short[] neg, final short[] pos, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, short[] xs, short[] ys, short[] neg, short[] pos, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, short[] xs, short[] ys, short[] neg, short[] pos, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, short[] xs, short[] ys, short[] neg, short[] pos, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final int[] xs, final int[] ys, final int[] neg, final int[] pos, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final int[] xs, final int[] ys, final int[] neg, final int[] pos, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final int[] xs, final int[] ys, final int[] neg, final int[] pos, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, int[] xs, int[] ys, int[] neg, int[] pos, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, int[] xs, int[] ys, int[] neg, int[] pos, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, int[] xs, int[] ys, int[] neg, int[] pos, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final long[] xs, final long[] ys, final long[] neg, final long[] pos, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final long[] xs, final long[] ys, final long[] neg, final long[] pos, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final long[] xs, final long[] ys, final long[] neg, final long[] pos, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, long[] xs, long[] ys, long[] neg, long[] pos, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, long[] xs, long[] ys, long[] neg, long[] pos, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, long[] xs, long[] ys, long[] neg, long[] pos, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final float[] xs, final float[] ys, final float[] neg, final float[] pos, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final float[] xs, final float[] ys, final float[] neg, final float[] pos, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final float[] xs, final float[] ys, final float[] neg, final float[] pos, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, float[] xs, float[] ys, float[] neg, float[] pos, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, float[] xs, float[] ys, float[] neg, float[] pos, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, float[] xs, float[] ys, float[] neg, float[] pos, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final double[] xs, final double[] ys, final double[] neg, final double[] pos, final int count) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final double[] xs, final double[] ys, final double[] neg, final double[] pos, final int count, final int flags) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags);
    }

    /**
     * Plots vertical error bar. The label_id should be the same as the label_id of the associated line or bar plot.
     */
    public static void plotErrorBarsV(final String labelId, final double[] xs, final double[] ys, final double[] neg, final double[] pos, final int count, final int flags, final int offset) {
        nPlotErrorBarsV(labelId, xs, ys, neg, pos, count, flags, offset);
    }

    private static native void nPlotErrorBarsV(String labelId, double[] xs, double[] ys, double[] neg, double[] pos, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, double[] xs, double[] ys, double[] neg, double[] pos, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    private static native void nPlotErrorBarsV(String labelId, double[] xs, double[] ys, double[] neg, double[] pos, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        auto neg = obj_neg == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_neg, JNI_FALSE);
        auto pos = obj_pos == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_pos, JNI_FALSE);
        ImPlot::PlotErrorBars(labelId, &xs[0], &ys[0], &neg[0], &pos[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
        if (neg != NULL) env->ReleasePrimitiveArrayCritical(obj_neg, neg, JNI_FALSE);
        if (pos != NULL) env->ReleasePrimitiveArrayCritical(obj_pos, pos, JNI_FALSE);
    */

    // values

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] values) {
        nPlotStems(labelId, values);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] values, final double yRef) {
        nPlotStems(labelId, values, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] values, final double yRef, final double xscale) {
        nPlotStems(labelId, values, yRef, xscale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] values, final double yRef, final double xscale, final double xstart) {
        nPlotStems(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotStems(String labelId, short[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] values) {
        nPlotStems(labelId, values);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] values, final double yRef) {
        nPlotStems(labelId, values, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] values, final double yRef, final double xscale) {
        nPlotStems(labelId, values, yRef, xscale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] values, final double yRef, final double xscale, final double xstart) {
        nPlotStems(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotStems(String labelId, int[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] values) {
        nPlotStems(labelId, values);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] values, final double yRef) {
        nPlotStems(labelId, values, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] values, final double yRef, final double xscale) {
        nPlotStems(labelId, values, yRef, xscale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] values, final double yRef, final double xscale, final double xstart) {
        nPlotStems(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotStems(String labelId, long[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] values) {
        nPlotStems(labelId, values);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] values, final double yRef) {
        nPlotStems(labelId, values, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] values, final double yRef, final double xscale) {
        nPlotStems(labelId, values, yRef, xscale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] values, final double yRef, final double xscale, final double xstart) {
        nPlotStems(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotStems(String labelId, float[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] values) {
        nPlotStems(labelId, values);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] values, final double yRef) {
        nPlotStems(labelId, values, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] values, final double yRef, final double xscale) {
        nPlotStems(labelId, values, yRef, xscale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] values, final double yRef, final double xscale, final double xstart) {
        nPlotStems(labelId, values, yRef, xscale, xstart);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] values, final double yRef, final double xscale, final double xstart, final int flags) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] values, final double yRef, final double xscale, final double xstart, final int flags, final int offset) {
        nPlotStems(labelId, values, yRef, xscale, xstart, flags, offset);
    }

    private static native void nPlotStems(String labelId, double[] values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] values, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] values, double yRef, double xscale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] values, double yRef, double xscale, double xstart); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] values, double yRef, double xscale, double xstart, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] values, double yRef, double xscale, double xstart, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], LEN(values), yRef, xscale, xstart, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] values, final int count) {
        nPlotStemsV(labelId, values, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] values, final int count, final double yRef) {
        nPlotStemsV(labelId, values, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] values, final int count, final double yRef, final double scale) {
        nPlotStemsV(labelId, values, count, yRef, scale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] values, final int count, final double yRef, final double scale, final double start) {
        nPlotStemsV(labelId, values, count, yRef, scale, start);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] values, final int count, final double yRef, final double scale, final double start, final int flags) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] values, final int count, final double yRef, final double scale, final double start, final int flags, final int offset) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] values, int count, double yRef, double scale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] values, int count, double yRef, double scale, double start); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] values, int count, double yRef, double scale, double start, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] values, int count, double yRef, double scale, double start, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] values, final int count) {
        nPlotStemsV(labelId, values, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] values, final int count, final double yRef) {
        nPlotStemsV(labelId, values, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] values, final int count, final double yRef, final double scale) {
        nPlotStemsV(labelId, values, count, yRef, scale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] values, final int count, final double yRef, final double scale, final double start) {
        nPlotStemsV(labelId, values, count, yRef, scale, start);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] values, final int count, final double yRef, final double scale, final double start, final int flags) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] values, final int count, final double yRef, final double scale, final double start, final int flags, final int offset) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] values, int count, double yRef, double scale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] values, int count, double yRef, double scale, double start); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] values, int count, double yRef, double scale, double start, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] values, int count, double yRef, double scale, double start, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] values, final int count) {
        nPlotStemsV(labelId, values, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] values, final int count, final double yRef) {
        nPlotStemsV(labelId, values, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] values, final int count, final double yRef, final double scale) {
        nPlotStemsV(labelId, values, count, yRef, scale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] values, final int count, final double yRef, final double scale, final double start) {
        nPlotStemsV(labelId, values, count, yRef, scale, start);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] values, final int count, final double yRef, final double scale, final double start, final int flags) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] values, final int count, final double yRef, final double scale, final double start, final int flags, final int offset) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] values, int count, double yRef, double scale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] values, int count, double yRef, double scale, double start); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] values, int count, double yRef, double scale, double start, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] values, int count, double yRef, double scale, double start, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] values, final int count) {
        nPlotStemsV(labelId, values, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] values, final int count, final double yRef) {
        nPlotStemsV(labelId, values, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] values, final int count, final double yRef, final double scale) {
        nPlotStemsV(labelId, values, count, yRef, scale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] values, final int count, final double yRef, final double scale, final double start) {
        nPlotStemsV(labelId, values, count, yRef, scale, start);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] values, final int count, final double yRef, final double scale, final double start, final int flags) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] values, final int count, final double yRef, final double scale, final double start, final int flags, final int offset) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] values, int count, double yRef, double scale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] values, int count, double yRef, double scale, double start); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] values, int count, double yRef, double scale, double start, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] values, int count, double yRef, double scale, double start, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] values, final int count) {
        nPlotStemsV(labelId, values, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] values, final int count, final double yRef) {
        nPlotStemsV(labelId, values, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] values, final int count, final double yRef, final double scale) {
        nPlotStemsV(labelId, values, count, yRef, scale);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] values, final int count, final double yRef, final double scale, final double start) {
        nPlotStemsV(labelId, values, count, yRef, scale, start);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] values, final int count, final double yRef, final double scale, final double start, final int flags) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] values, final int count, final double yRef, final double scale, final double start, final int flags, final int offset) {
        nPlotStemsV(labelId, values, count, yRef, scale, start, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] values, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] values, int count, double yRef, double scale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] values, int count, double yRef, double scale, double start); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] values, int count, double yRef, double scale, double start, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] values, int count, double yRef, double scale, double start, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotStems(labelId, &values[0], count, yRef, scale, start, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    // xs,ys

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] xs, final short[] ys) {
        nPlotStems(labelId, xs, ys);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] xs, final short[] ys, final double ref) {
        nPlotStems(labelId, xs, ys, ref);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] xs, final short[] ys, final double ref, final int flags) {
        nPlotStems(labelId, xs, ys, ref, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final short[] xs, final short[] ys, final double ref, final int flags, final int offset) {
        nPlotStems(labelId, xs, ys, ref, flags, offset);
    }

    private static native void nPlotStems(String labelId, short[] xs, short[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] xs, short[] ys, double ref); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] xs, short[] ys, double ref, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, short[] xs, short[] ys, double ref, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] xs, final int[] ys) {
        nPlotStems(labelId, xs, ys);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] xs, final int[] ys, final double ref) {
        nPlotStems(labelId, xs, ys, ref);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] xs, final int[] ys, final double ref, final int flags) {
        nPlotStems(labelId, xs, ys, ref, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final int[] xs, final int[] ys, final double ref, final int flags, final int offset) {
        nPlotStems(labelId, xs, ys, ref, flags, offset);
    }

    private static native void nPlotStems(String labelId, int[] xs, int[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] xs, int[] ys, double ref); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] xs, int[] ys, double ref, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, int[] xs, int[] ys, double ref, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] xs, final long[] ys) {
        nPlotStems(labelId, xs, ys);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] xs, final long[] ys, final double ref) {
        nPlotStems(labelId, xs, ys, ref);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] xs, final long[] ys, final double ref, final int flags) {
        nPlotStems(labelId, xs, ys, ref, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final long[] xs, final long[] ys, final double ref, final int flags, final int offset) {
        nPlotStems(labelId, xs, ys, ref, flags, offset);
    }

    private static native void nPlotStems(String labelId, long[] xs, long[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] xs, long[] ys, double ref); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] xs, long[] ys, double ref, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, long[] xs, long[] ys, double ref, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] xs, final float[] ys) {
        nPlotStems(labelId, xs, ys);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] xs, final float[] ys, final double ref) {
        nPlotStems(labelId, xs, ys, ref);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] xs, final float[] ys, final double ref, final int flags) {
        nPlotStems(labelId, xs, ys, ref, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final float[] xs, final float[] ys, final double ref, final int flags, final int offset) {
        nPlotStems(labelId, xs, ys, ref, flags, offset);
    }

    private static native void nPlotStems(String labelId, float[] xs, float[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] xs, float[] ys, double ref); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] xs, float[] ys, double ref, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, float[] xs, float[] ys, double ref, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] xs, final double[] ys) {
        nPlotStems(labelId, xs, ys);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] xs, final double[] ys, final double ref) {
        nPlotStems(labelId, xs, ys, ref);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] xs, final double[] ys, final double ref, final int flags) {
        nPlotStems(labelId, xs, ys, ref, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStems(final String labelId, final double[] xs, final double[] ys, final double ref, final int flags, final int offset) {
        nPlotStems(labelId, xs, ys, ref, flags, offset);
    }

    private static native void nPlotStems(String labelId, double[] xs, double[] ys); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] xs, double[] ys, double ref); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] xs, double[] ys, double ref, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStems(String labelId, double[] xs, double[] ys, double ref, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], LEN(xs), ref, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] xs, final short[] ys, final int count) {
        nPlotStemsV(labelId, xs, ys, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] xs, final short[] ys, final int count, final double yRef) {
        nPlotStemsV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] xs, final short[] ys, final int count, final double yRef, final int flags) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final short[] xs, final short[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, short[] xs, short[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] xs, short[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] xs, short[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, short[] xs, short[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] xs, final int[] ys, final int count) {
        nPlotStemsV(labelId, xs, ys, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] xs, final int[] ys, final int count, final double yRef) {
        nPlotStemsV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] xs, final int[] ys, final int count, final double yRef, final int flags) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final int[] xs, final int[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, int[] xs, int[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] xs, int[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] xs, int[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, int[] xs, int[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] xs, final long[] ys, final int count) {
        nPlotStemsV(labelId, xs, ys, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] xs, final long[] ys, final int count, final double yRef) {
        nPlotStemsV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] xs, final long[] ys, final int count, final double yRef, final int flags) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final long[] xs, final long[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, long[] xs, long[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] xs, long[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] xs, long[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, long[] xs, long[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] xs, final float[] ys, final int count) {
        nPlotStemsV(labelId, xs, ys, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] xs, final float[] ys, final int count, final double yRef) {
        nPlotStemsV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] xs, final float[] ys, final int count, final double yRef, final int flags) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final float[] xs, final float[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, float[] xs, float[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] xs, float[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] xs, float[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, float[] xs, float[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] xs, final double[] ys, final int count) {
        nPlotStemsV(labelId, xs, ys, count);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] xs, final double[] ys, final int count, final double yRef) {
        nPlotStemsV(labelId, xs, ys, count, yRef);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] xs, final double[] ys, final int count, final double yRef, final int flags) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags);
    }

    /**
     * Plots vertical stems.
     */
    public static void plotStemsV(final String labelId, final double[] xs, final double[] ys, final int count, final double yRef, final int flags, final int offset) {
        nPlotStemsV(labelId, xs, ys, count, yRef, flags, offset);
    }

    private static native void nPlotStemsV(String labelId, double[] xs, double[] ys, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] xs, double[] ys, int count, double yRef); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] xs, double[] ys, int count, double yRef, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    private static native void nPlotStemsV(String labelId, double[] xs, double[] ys, int count, double yRef, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto xs = obj_xs == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_xs, JNI_FALSE);
        auto ys = obj_ys == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_ys, JNI_FALSE);
        ImPlot::PlotStems(labelId, &xs[0], &ys[0], count, yRef, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (xs != NULL) env->ReleasePrimitiveArrayCritical(obj_xs, xs, JNI_FALSE);
        if (ys != NULL) env->ReleasePrimitiveArrayCritical(obj_ys, ys, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final short[] values, final int count) {
        nPlotInfLines(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final short[] values, final int count, final int flags) {
        nPlotInfLines(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final short[] values, final int count, final int flags, final int offset) {
        nPlotInfLines(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLines(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, short[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, short[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final int[] values, final int count) {
        nPlotInfLines(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final int[] values, final int count, final int flags) {
        nPlotInfLines(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final int[] values, final int count, final int flags, final int offset) {
        nPlotInfLines(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLines(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, int[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, int[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final long[] values, final int count) {
        nPlotInfLines(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final long[] values, final int count, final int flags) {
        nPlotInfLines(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final long[] values, final int count, final int flags, final int offset) {
        nPlotInfLines(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLines(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, long[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, long[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final float[] values, final int count) {
        nPlotInfLines(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final float[] values, final int count, final int flags) {
        nPlotInfLines(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final float[] values, final int count, final int flags, final int offset) {
        nPlotInfLines(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLines(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, float[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, float[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final double[] values, final int count) {
        nPlotInfLines(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final double[] values, final int count, final int flags) {
        nPlotInfLines(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLines(final String labelId, final double[] values, final int count, final int flags, final int offset) {
        nPlotInfLines(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLines(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, double[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLines(String labelId, double[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], LEN(values), flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final short[] values, final int count) {
        nPlotInfLinesV(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final short[] values, final int count, final int flags) {
        nPlotInfLinesV(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final short[] values, final int count, final int flags, final int offset) {
        nPlotInfLinesV(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLinesV(String labelId, short[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, short[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, short[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final int[] values, final int count) {
        nPlotInfLinesV(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final int[] values, final int count, final int flags) {
        nPlotInfLinesV(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final int[] values, final int count, final int flags, final int offset) {
        nPlotInfLinesV(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLinesV(String labelId, int[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, int[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, int[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final long[] values, final int count) {
        nPlotInfLinesV(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final long[] values, final int count, final int flags) {
        nPlotInfLinesV(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final long[] values, final int count, final int flags, final int offset) {
        nPlotInfLinesV(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLinesV(String labelId, long[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, long[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, long[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final float[] values, final int count) {
        nPlotInfLinesV(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final float[] values, final int count, final int flags) {
        nPlotInfLinesV(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final float[] values, final int count, final int flags, final int offset) {
        nPlotInfLinesV(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLinesV(String labelId, float[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, float[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, float[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final double[] values, final int count) {
        nPlotInfLinesV(labelId, values, count);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final double[] values, final int count, final int flags) {
        nPlotInfLinesV(labelId, values, count, flags);
    }

    /**
     * Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).
     */
    public static void plotInfLinesV(final String labelId, final double[] values, final int count, final int flags, final int offset) {
        nPlotInfLinesV(labelId, values, count, flags, offset);
    }

    private static native void nPlotInfLinesV(String labelId, double[] values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, double[] values, int count, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotInfLinesV(String labelId, double[] values, int count, int flags, int offset); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotInfLines(labelId, &values[0], count, flags, offset);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final short[] values, final double x, final double y, final double radius) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final short[] values, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final short[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final short[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final short[] values, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, short[] values, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, short[] values, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, short[] values, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, short[] values, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, short[] values, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final int[] values, final double x, final double y, final double radius) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final int[] values, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final int[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final int[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final int[] values, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, int[] values, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, int[] values, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, int[] values, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, int[] values, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, int[] values, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final long[] values, final double x, final double y, final double radius) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final long[] values, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final long[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final long[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final long[] values, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, long[] values, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, long[] values, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, long[] values, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, long[] values, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, long[] values, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final float[] values, final double x, final double y, final double radius) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final float[] values, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final float[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final float[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final float[] values, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, float[] values, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, float[] values, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, float[] values, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, float[] values, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, float[] values, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final double[] values, final double x, final double y, final double radius) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final double[] values, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final double[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final double[] values, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChart(final String[] labelIds, final double[] values, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChart(labelIds, labelIds.length, values, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, double[] values, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, double[] values, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, double[] values, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, double[] values, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChart(String[] obj_labelIds, int labelIdsCount, double[] values, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], LEN(values), x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final short[] values, final int count, final double x, final double y, final double radius) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final short[] values, final int count, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final short[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final short[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final short[] values, final int count, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, short[] values, int count, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, short[] values, int count, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, short[] values, int count, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, short[] values, int count, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, short[] values, int count, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final int[] values, final int count, final double x, final double y, final double radius) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final int[] values, final int count, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final int[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final int[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final int[] values, final int count, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, int[] values, int count, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, int[] values, int count, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, int[] values, int count, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, int[] values, int count, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, int[] values, int count, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final long[] values, final int count, final double x, final double y, final double radius) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final long[] values, final int count, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final long[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final long[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final long[] values, final int count, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, long[] values, int count, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, long[] values, int count, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, long[] values, int count, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, long[] values, int count, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, long[] values, int count, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final float[] values, final int count, final double x, final double y, final double radius) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final float[] values, final int count, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final float[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final float[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final float[] values, final int count, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, float[] values, int count, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, float[] values, int count, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, float[] values, int count, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, float[] values, int count, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, float[] values, int count, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final double[] values, final int count, final double x, final double y, final double radius) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final double[] values, final int count, final double x, final double y, final double radius, final String labelFmt) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final double[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final double[] values, final int count, final double x, final double y, final double radius, final String labelFmt, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, labelFmt, angle0, flags);
    }

    /**
     * Plots a pie chart. If the sum of values{@code >}1 or normalize is true, each value will be normalized. Center and radius are in plot units. #label_fmt can be set to NULL for no labels.
     */
    public static void plotPieChartV(final String[] labelIds, final double[] values, final int count, final double x, final double y, final double radius, final double angle0, final int flags) {
        nPlotPieChartV(labelIds, labelIds.length, values, count, x, y, radius, angle0, flags);
    }

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, double[] values, int count, double x, double y, double radius); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, double[] values, int count, double x, double y, double radius, String labelFmt); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, double[] values, int count, double x, double y, double radius, String labelFmt, double angle0); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, double[] values, int count, double x, double y, double radius, String labelFmt, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, labelFmt, angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotPieChartV(String[] obj_labelIds, int labelIdsCount, double[] values, int count, double x, double y, double radius, double angle0, int flags); /*MANUAL
        const char* labelIds[labelIdsCount];
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            auto rawStr = (char*)env->GetStringUTFChars(str, JNI_FALSE);
            labelIds[i] = rawStr;
        };
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotPieChart(labelIds, &values[0], count, x, y, radius, "%.1f", angle0, flags);
        for (int i = 0; i < labelIdsCount; i++) {
            const jstring str = (jstring)env->GetObjectArrayElement(obj_labelIds, i);
            env->ReleaseStringUTFChars(str, labelIds[i]);
        };
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols) {
        nPlotHeatmap(labelId, values, rows, cols);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final short[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin, double scaleMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, short[] values, int rows, int cols, double scaleMin, double scaleMax, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, "%.1f", ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols) {
        nPlotHeatmap(labelId, values, rows, cols);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final int[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin, double scaleMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, int[] values, int rows, int cols, double scaleMin, double scaleMax, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, "%.1f", ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols) {
        nPlotHeatmap(labelId, values, rows, cols);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final long[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin, double scaleMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, long[] values, int rows, int cols, double scaleMin, double scaleMax, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, "%.1f", ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols) {
        nPlotHeatmap(labelId, values, rows, cols);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final float[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin, double scaleMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, float[] values, int rows, int cols, double scaleMin, double scaleMax, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, "%.1f", ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols) {
        nPlotHeatmap(labelId, values, rows, cols);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final String labelFmt, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, labelFmt, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final ImPlotPoint boundsMin, final ImPlotPoint boundsMax, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMin.x, boundsMin.y, boundsMax.x, boundsMax.y, flags);
    }

    /**
     * Plots a 2D heatmap chart. Values are expected to be in row-major order. Leave #scale_min and scale_max both at 0 for automatic color scaling, or set them to a predefined range. #label_fmt can be set to NULL for no labels.
     */
    public static void plotHeatmap(final String labelId, final double[] values, final int rows, final int cols, final double scaleMin, final double scaleMax, final double boundsMinX, final double boundsMinY, final double boundsMaxX, final double boundsMaxY, final int flags) {
        nPlotHeatmap(labelId, values, rows, cols, scaleMin, scaleMax, boundsMinX, boundsMinY, boundsMaxX, boundsMaxY, flags);
    }

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin, double scaleMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin, double scaleMax, String labelFmt, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto labelFmt = obj_labelFmt == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelFmt, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, labelFmt, ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        if (labelFmt != NULL) env->ReleaseStringUTFChars(obj_labelFmt, labelFmt);
    */

    private static native void nPlotHeatmap(String labelId, double[] values, int rows, int cols, double scaleMin, double scaleMax, double boundsMinX, double boundsMinY, double boundsMaxX, double boundsMaxY, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        ImPlot::PlotHeatmap(labelId, &values[0], rows, cols, scaleMin, scaleMax, "%.1f", ImPlotPoint(boundsMinX, boundsMinY), ImPlotPoint(boundsMaxX, boundsMaxY), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values) {
        return nPlotHistogram(labelId, values);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins) {
        return nPlotHistogram(labelId, values, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins, final double barScale) {
        return nPlotHistogram(labelId, values, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final short[] values, final int bins, final double barScale, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, flags);
    }

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), ImPlotBin_Sturges, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, short[] obj_values, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values) {
        return nPlotHistogram(labelId, values);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins) {
        return nPlotHistogram(labelId, values, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins, final double barScale) {
        return nPlotHistogram(labelId, values, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final int[] values, final int bins, final double barScale, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, flags);
    }

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), ImPlotBin_Sturges, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, int[] obj_values, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values) {
        return nPlotHistogram(labelId, values);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins) {
        return nPlotHistogram(labelId, values, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins, final double barScale) {
        return nPlotHistogram(labelId, values, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final long[] values, final int bins, final double barScale, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, flags);
    }

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), ImPlotBin_Sturges, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, long[] obj_values, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values) {
        return nPlotHistogram(labelId, values);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins) {
        return nPlotHistogram(labelId, values, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins, final double barScale) {
        return nPlotHistogram(labelId, values, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final float[] values, final int bins, final double barScale, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, flags);
    }

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), ImPlotBin_Sturges, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, float[] obj_values, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values) {
        return nPlotHistogram(labelId, values);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins) {
        return nPlotHistogram(labelId, values, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins, final double barScale) {
        return nPlotHistogram(labelId, values, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogram(labelId, values, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogram(labelId, values, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram(final String labelId, final double[] values, final int bins, final double barScale, final int flags) {
        return nPlotHistogram(labelId, values, bins, barScale, flags);
    }

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), ImPlotBin_Sturges, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogram(String obj_labelId, double[] obj_values, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], LEN(values), bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count) {
        return nPlotHistogramV(labelId, values, count);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins) {
        return nPlotHistogramV(labelId, values, count, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins, final double barScale) {
        return nPlotHistogramV(labelId, values, count, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final short[] values, final int count, final int bins, final double barScale, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, flags);
    }

    private static native double nPlotHistogramV(String obj_labelId, short[] obj_values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, short[] obj_values, int count, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, short[] obj_values, int count, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, short[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, short[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, short[] obj_values, int count, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (short*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count) {
        return nPlotHistogramV(labelId, values, count);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins) {
        return nPlotHistogramV(labelId, values, count, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins, final double barScale) {
        return nPlotHistogramV(labelId, values, count, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final int[] values, final int count, final int bins, final double barScale, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, flags);
    }

    private static native double nPlotHistogramV(String obj_labelId, int[] obj_values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, int[] obj_values, int count, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, int[] obj_values, int count, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, int[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, int[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, int[] obj_values, int count, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (int*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count) {
        return nPlotHistogramV(labelId, values, count);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins) {
        return nPlotHistogramV(labelId, values, count, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins, final double barScale) {
        return nPlotHistogramV(labelId, values, count, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final long[] values, final int count, final int bins, final double barScale, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, flags);
    }

    private static native double nPlotHistogramV(String obj_labelId, long[] obj_values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, long[] obj_values, int count, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, long[] obj_values, int count, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, long[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, long[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, long[] obj_values, int count, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (long*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count) {
        return nPlotHistogramV(labelId, values, count);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins) {
        return nPlotHistogramV(labelId, values, count, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins, final double barScale) {
        return nPlotHistogramV(labelId, values, count, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final float[] values, final int count, final int bins, final double barScale, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, flags);
    }

    private static native double nPlotHistogramV(String obj_labelId, float[] obj_values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, float[] obj_values, int count, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, float[] obj_values, int count, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, float[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, float[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, float[] obj_values, int count, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (float*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count) {
        return nPlotHistogramV(labelId, values, count);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins) {
        return nPlotHistogramV(labelId, values, count, bins);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins, final double barScale) {
        return nPlotHistogramV(labelId, values, count, bins, barScale);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins, final double barScale, final ImPlotRange range) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins, final double barScale, final ImPlotRange range, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, range.min, range.max, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins, final double barScale, final double rangeMin, final double rangeMax, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, rangeMin, rangeMax, flags);
    }

    /**
     * Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #cumulative is true, each bin contains its count plus the counts of all previous bins.
     * If #density is true, the PDF is visualized. If both are true, the CDF is visualized. If #range is left unspecified, the min/max of #values will be used as the range.
     * If #range is specified, outlier values outside of the range are not binned. However, outliers still count toward normalizing and cumulative counts unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogramV(final String labelId, final double[] values, final int count, final int bins, final double barScale, final int flags) {
        return nPlotHistogramV(labelId, values, count, bins, barScale, flags);
    }

    private static native double nPlotHistogramV(String obj_labelId, double[] obj_values, int count); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, double[] obj_values, int count, int bins); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, double[] obj_values, int count, int bins, double barScale); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, double[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax));
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, double[] obj_values, int count, int bins, double barScale, double rangeMin, double rangeMax, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(rangeMin, rangeMax), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    private static native double nPlotHistogramV(String obj_labelId, double[] obj_values, int count, int bins, double barScale, int flags); /*MANUAL
        auto labelId = obj_labelId == NULL ? NULL : (char*)env->GetStringUTFChars(obj_labelId, JNI_FALSE);
        auto values = obj_values == NULL ? NULL : (double*)env->GetPrimitiveArrayCritical(obj_values, JNI_FALSE);
        auto _result = ImPlot::PlotHistogram(labelId, &values[0], count, bins, barScale, ImPlotRange(), flags);
        if (labelId != NULL) env->ReleaseStringUTFChars(obj_labelId, labelId);
        if (values != NULL) env->ReleasePrimitiveArrayCritical(obj_values, values, JNI_FALSE);
        return _result;
    */

    /**
     * Plots two dimensional, bivariate histogram as a heatmap. #x_bins and #y_bins can be a positive integer or an ImPlotBin. If #density is true, the PDF is visualized.
     * If #range is left unspecified, the min/max of #xs an #ys will be used as the ranges. If #range is specified, outlier values outside of range are not binned.
     * However, outliers still count toward the normalizing count for density plots unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram2D(final String labelId, final short[] xs, final short[] ys) {
        return nPlotHistogram2D(labelId, xs, ys);
    }

    /**
     * Plots two dimensional, bivariate histogram as a heatmap. #x_bins and #y_bins can be a positive integer or an ImPlotBin. If #density is true, the PDF is visualized.
     * If #range is left unspecified, the min/max of #xs an #ys will be used as the ranges. If #range is specified, outlier values outside of range are not binned.
     * However, outliers still count toward the normalizing count for density plots unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram2D(final String labelId, final short[] xs, final short[] ys, final int xBins) {
        return nPlotHistogram2D(labelId, xs, ys, xBins);
    }

    /**
     * Plots two dimensional, bivariate histogram as a heatmap. #x_bins and #y_bins can be a positive integer or an ImPlotBin. If #density is true, the PDF is visualized.
     * If #range is left unspecified, the min/max of #xs an #ys will be used as the ranges. If #range is specified, outlier values outside of range are not binned.
     * However, outliers still count toward the normalizing count for density plots unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram2D(final String labelId, final short[] xs, final short[] ys, final int xBins, final int yBins) {
        return nPlotHistogram2D(labelId, xs, ys, xBins, yBins);
    }

    /**
     * Plots two dimensional, bivariate histogram as a heatmap. #x_bins and #y_bins can be a positive integer or an ImPlotBin. If #density is true, the PDF is visualized.
     * If #range is left unspecified, the min/max of #xs an #ys will be used as the ranges. If #range is specified, outlier values outside of range are not binned.
     * However, outliers still count toward the normalizing count for density plots unless #outliers is false. The largest bin count or density is returned.
     */
    public static double plotHistogram2D(final String labelId, final short[] xs, final short[] ys, final int xBins, final int yBins, final ImPlotRect range) {
        return nPlotHistogram2D(labelId