package com.xmfuncoding.fundialog

import android.content.Context
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.util.DisplayMetrics
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.annotation.StringRes
import androidx.core.view.isVisible
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.FragmentTransaction
import com.xmfuncoding.fundialog.databinding.FunDialogBinding

/**
 *  Author:      XmFunCoding
 *  Email:       xmfuncoding@163.com
 *  Date:        2022-10-04 16:30
 *  Description: 统一对话框
 */
class FunDialog : DialogFragment() {

    companion object {
        const val TITLE_KEY         = "title"
        const val MESSAGE_KEY       = "message"
        const val CONFIRM_TEXT_KEY  = "confirmText"
        const val CANCEL_TEXT_KEY   = "cancelText"
        const val CANCELABLE_KEY    = "cancelable"

        const val NO_TEXT = 0

        fun newInstance(
            context: Context,
            @StringRes title: Int = NO_TEXT,
            @StringRes message: Int = NO_TEXT,
            @StringRes confirmText: Int = NO_TEXT,
            @StringRes cancelText: Int = NO_TEXT,
            cancelable: Boolean = true,
        ) = newInstance(
            context = context,
            title = if (title != NO_TEXT) context.getString(title) else "",
            message = if (message != NO_TEXT) context.getString(message) else "",
            confirmText = if (confirmText != NO_TEXT) context.getString(confirmText) else "",
            cancelText = if (cancelText != NO_TEXT) context.getString(cancelText) else "",
            cancelable = cancelable)

        fun newInstance(
            context: Context,
            title: String? = "",
            message: String?,
            confirmText: String?,
            cancelText: String? = "",
            cancelable: Boolean = true,
        ): FunDialog {
            val fragment = FunDialog()
            val bundle = Bundle()
            bundle.putString(TITLE_KEY, title)
            bundle.putString(MESSAGE_KEY, message)
            bundle.putString(CONFIRM_TEXT_KEY, confirmText)
            bundle.putString(CANCEL_TEXT_KEY, cancelText)
            bundle.putBoolean(CANCELABLE_KEY, cancelable)
            fragment.arguments = bundle
            return fragment
        }
    }

    private lateinit var binding: FunDialogBinding

    private var mTitle: String? = null
    private var mMessage: String? = null
    private var mConfirmText: String? = null
    private var mCancelText: String? = null
    private var mCancelable = false

    private var mOnConfirmClickListener: View.OnClickListener? = null
    private var mOnCancelClickListener: View.OnClickListener? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        binding = FunDialogBinding.inflate(layoutInflater)
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        mTitle = arguments?.getString(TITLE_KEY)
        mMessage = arguments?.getString(MESSAGE_KEY)
        mConfirmText = arguments?.getString(CONFIRM_TEXT_KEY)
        mCancelText = arguments?.getString(CANCEL_TEXT_KEY)
        mCancelable = arguments?.getBoolean(CANCELABLE_KEY) ?: true

        isCancelable = mCancelable

        if (mTitle.isNullOrEmpty()) {
            binding.tvTitle.isVisible = false
        } else {
            binding.tvTitle.isVisible = true
            binding.tvTitle.text = mTitle
        }

        binding.tvMessage.text = mMessage

        binding.btnCancel.isVisible = !mCancelText.isNullOrEmpty()
        binding.viewDivider.isVisible = !mCancelText.isNullOrEmpty()

        binding.btnCancel.text = mCancelText
        binding.btnConfirm.text = mConfirmText

        binding.btnCancel.setOnClickListener {
            mOnCancelClickListener?.onClick(it)
            dismiss()
        }

        binding.btnConfirm.setOnClickListener {
            mOnConfirmClickListener?.onClick(it)
            dismiss()
        }
    }

    override fun onStart() {
        /*
        因为View在添加后,对话框最外层的ViewGroup并不知道我们导入的View所需要的的宽度。 所以我们需要在onStart生命周期里修改对话框尺寸参数
        */
        super.onStart()
        val displayMetrics = DisplayMetrics()
        requireActivity().windowManager.defaultDisplay.getMetrics(displayMetrics)
        dialog?.window?.setLayout((displayMetrics.widthPixels * 0.9).toInt(), ViewGroup.LayoutParams.WRAP_CONTENT)
        dialog?.window?.decorView?.background = ColorDrawable(Color.TRANSPARENT)
    }

    fun setOnConfirmClickListener(onConfirmClickListener: View.OnClickListener?) {
        mOnConfirmClickListener = onConfirmClickListener
    }

    fun setOnCancelClickListener(onCancelClickListener: View.OnClickListener?) {
        mOnCancelClickListener = onCancelClickListener
    }

    fun show(manager: FragmentManager) {
        show(manager, "dialog")
    }

    override fun show(manager: FragmentManager, tag: String?) {
        try {
            val ft: FragmentTransaction = manager.beginTransaction()
            ft.add(this, tag)
            ft.commitAllowingStateLoss()
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        }
    }

    fun isShowing() : Boolean {
        return dialog?.isShowing ?: false
    }
}