package com.xmfuncoding.fundialog

import android.app.Dialog
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.View.OnClickListener
import android.view.ViewGroup
import androidx.core.content.ContextCompat
import androidx.core.view.isVisible
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.FragmentTransaction
import com.chad.library.adapter.base.BaseQuickAdapter
import com.chad.library.adapter.base.viewholder.BaseViewHolder
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import com.xmfuncoding.funbottomselectdialog.ISelection
import com.xmfuncoding.fundialog.FunConstants.SELECTED_POSITION_KEY
import com.xmfuncoding.fundialog.FunConstants.SELECTION_LIST_KEY
import com.xmfuncoding.fundialog.FunConstants.TITLE_KEY
import com.xmfuncoding.fundialog.databinding.FunDialogBottomSelectBinding
import java.io.Serializable
import java.util.*

/**
 *  Author:      XmFunCoding
 *  Email:       xmfuncoding@163.com
 *  Date:        2022-10-15 18:37
 *  Description: 底部选择对话框
 */
class FunBottomSelectDialog<T : ISelection> : BottomSheetDialogFragment() {

    companion object {
        fun newInstance(
            title: String?,
            list: List<*>?,
            position: Int,
        ): FunBottomSelectDialog<*> {
            val fragment = FunBottomSelectDialog<ISelection>()
            val bundle = Bundle()
            bundle.putString(TITLE_KEY, title)
            bundle.putSerializable(SELECTION_LIST_KEY, list as Serializable?)
            bundle.putInt(SELECTED_POSITION_KEY, position)
            fragment.arguments = bundle
            return fragment
        }
    }

    private lateinit var binding: FunDialogBottomSelectBinding

    // 对话框标题
    private var mTitle: String? = null

    // 选择列表
    private lateinit var mAdapter: BaseQuickAdapter<T, BaseViewHolder>

    private var mList: List<T>? = ArrayList<T>()

    // 选中位置
    private var mSelectedPosition = -1

    private var mOnItemSelectListener: OnItemSelectListener? = null
    private var mOnCloseClickListener: OnClickListener? = null

    interface OnItemSelectListener {
        fun onItemSelect(position: Int)
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        binding = FunDialogBottomSelectBinding.inflate(layoutInflater)

        val view = binding.root
        view.post {
            // R.id.design_bottom_sheet 基本是固定的,不用担心后面 API的更改
            val behavior = BottomSheetBehavior.from(dialog!!.findViewById(com.google.android.material.R.id.design_bottom_sheet))
            behavior.setHideable(false) // 禁止滑动收起底部栏
        }
        return view
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        mTitle = arguments?.getString(TITLE_KEY)
        mList = arguments?.getSerializable(SELECTION_LIST_KEY) as List<T>?
        mSelectedPosition = arguments?.getInt(SELECTED_POSITION_KEY) ?: -1
        if (mList == null) {
            mList = ArrayList()
        }
        if (mSelectedPosition != -1) {
            if (mSelectedPosition < mList!!.size) {
                mList!![mSelectedPosition].setItemChecked(true)
            }
        }

        // 对话框标题
        binding.tvTitle.text = mTitle

        // 关闭按钮
        binding.ivClose.setOnClickListener {
            mOnCloseClickListener?.onClick(it)
            dismiss()
        }

        // 选择列表
        mAdapter = object : BaseQuickAdapter<T, BaseViewHolder>(R.layout.fun_dialog_item_selection_list) {
            override fun convert(holder: BaseViewHolder, item: T) {
                holder.setText(R.id.tvItemTitle, item.getItemTitle())
                if (item.isItemChecked()) {
                    holder.setTextColor(R.id.tvItemTitle, ContextCompat.getColor(requireContext(), R.color.colorFF3A7DE8))
                    holder.getView<View>(R.id.ivItemChecked).isVisible = true
                } else {
                    holder.setTextColor(R.id.tvItemTitle, ContextCompat.getColor(requireContext(), R.color.colorDialogContent))
                    holder.getView<View>(R.id.ivItemChecked).isVisible = false
                }
            }
        }

        setOnItemClickListener()
        binding.rvList.adapter = mAdapter
        mAdapter.setList(mList)
    }

    override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
        //圆角边的关键
        setStyle(STYLE_NO_TITLE, R.style.FunBottomSelectDialogStyle)
        return super.onCreateDialog(savedInstanceState)
    }

    private fun setOnItemClickListener() {
        mAdapter.setOnItemClickListener { _, _, position ->
            if (!mAdapter.data[position].isItemChecked()) {
                if (mSelectedPosition != -1) {
                    mAdapter.data[mSelectedPosition].setItemChecked(false)
                } else {
                    if (mAdapter.data[0].isItemChecked()) {
                        mAdapter.data[0].setItemChecked(false)
                    }
                }
                mAdapter.data[position].setItemChecked(true)

                // 局部刷新，可参考：
                // 1. https://www.jianshu.com/p/45a43a117365 文中的 RecyclerView 部分
                // 2. https://github.com/CymChad/BaseRecyclerViewAdapterHelper/issues/867
                if (mSelectedPosition != -1) {
                    mAdapter.notifyItemChanged(mSelectedPosition, Any())
                }
                mAdapter.notifyItemChanged(position, Any())
                mSelectedPosition = position
            }
            dismiss()
            mOnItemSelectListener!!.onItemSelect(mSelectedPosition)
        }
    }

    fun setOnItemSelectListener(onItemSelectListener: OnItemSelectListener?) {
        mOnItemSelectListener = onItemSelectListener
    }

    fun setOnCloseClickListener(onClickListener: OnClickListener) {
        mOnCloseClickListener = onClickListener
    }

    fun show(manager: FragmentManager) {
        show(manager, "dialog")
    }

    override fun show(manager: FragmentManager, tag: String?) {
        try {
            val ft: FragmentTransaction = manager.beginTransaction()
            ft.add(this, tag)
            ft.commitAllowingStateLoss()
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        }
    }

    fun isShowing(): Boolean {
        return dialog?.isShowing ?: false
    }
}