/**
 * This file was auto-generated by Fern from our API Definition.
 */
package com.intercom.api.core;

import java.io.IOException;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Optional;
import java.util.Random;
import okhttp3.Interceptor;
import okhttp3.Response;

public class RetryInterceptor implements Interceptor {

    private static final Duration INITIAL_RETRY_DELAY = Duration.ofMillis(1000);
    private static final Duration MAX_RETRY_DELAY = Duration.ofMillis(60000);
    private static final double JITTER_FACTOR = 0.2;

    private final ExponentialBackoff backoff;
    private final Random random = new Random();

    public RetryInterceptor(int maxRetries) {
        this.backoff = new ExponentialBackoff(maxRetries);
    }

    @Override
    public Response intercept(Chain chain) throws IOException {
        Response response = chain.proceed(chain.request());

        if (shouldRetry(response.code())) {
            return retryChain(response, chain);
        }

        return response;
    }

    private Response retryChain(Response response, Chain chain) throws IOException {
        Optional<Duration> nextBackoff = this.backoff.nextBackoff(response);
        while (nextBackoff.isPresent()) {
            try {
                Thread.sleep(nextBackoff.get().toMillis());
            } catch (InterruptedException e) {
                throw new IOException("Interrupted while trying request", e);
            }
            response.close();
            response = chain.proceed(chain.request());
            if (shouldRetry(response.code())) {
                nextBackoff = this.backoff.nextBackoff(response);
            } else {
                return response;
            }
        }

        return response;
    }

    /**
     * Calculates the retry delay from response headers, with fallback to exponential backoff.
     * Priority: Retry-After > X-RateLimit-Reset > Exponential Backoff
     */
    private Duration getRetryDelayFromHeaders(Response response, int retryAttempt) {
        // Check for Retry-After header first (RFC 7231), with no jitter
        String retryAfter = response.header("Retry-After");
        if (retryAfter != null) {
            // Parse as number of seconds...
            Optional<Duration> secondsDelay = tryParseLong(retryAfter)
                    .map(seconds -> seconds * 1000)
                    .filter(delayMs -> delayMs > 0)
                    .map(delayMs -> Math.min(delayMs, MAX_RETRY_DELAY.toMillis()))
                    .map(Duration::ofMillis);
            if (secondsDelay.isPresent()) {
                return secondsDelay.get();
            }

            // ...or as an HTTP date; both are valid
            Optional<Duration> dateDelay = tryParseHttpDate(retryAfter)
                    .map(resetTime -> resetTime.toInstant().toEpochMilli() - System.currentTimeMillis())
                    .filter(delayMs -> delayMs > 0)
                    .map(delayMs -> Math.min(delayMs, MAX_RETRY_DELAY.toMillis()))
                    .map(Duration::ofMillis);
            if (dateDelay.isPresent()) {
                return dateDelay.get();
            }
        }

        // Then check for industry-standard X-RateLimit-Reset header, with positive jitter
        String rateLimitReset = response.header("X-RateLimit-Reset");
        if (rateLimitReset != null) {
            // Assume Unix timestamp in epoch seconds
            Optional<Duration> rateLimitDelay = tryParseLong(rateLimitReset)
                    .map(resetTimeSeconds -> (resetTimeSeconds * 1000) - System.currentTimeMillis())
                    .filter(delayMs -> delayMs > 0)
                    .map(delayMs -> Math.min(delayMs, MAX_RETRY_DELAY.toMillis()))
                    .map(this::addPositiveJitter)
                    .map(Duration::ofMillis);
            if (rateLimitDelay.isPresent()) {
                return rateLimitDelay.get();
            }
        }

        // Fall back to exponential backoff, with symmetric jitter
        long baseDelay = INITIAL_RETRY_DELAY.toMillis() * (1L << retryAttempt); // 2^retryAttempt
        long cappedDelay = Math.min(baseDelay, MAX_RETRY_DELAY.toMillis());
        return Duration.ofMillis(addSymmetricJitter(cappedDelay));
    }

    /**
     * Attempts to parse a string as a long, returning empty Optional on failure.
     */
    private Optional<Long> tryParseLong(String value) {
        if (value == null) {
            return Optional.empty();
        }
        try {
            return Optional.of(Long.parseLong(value));
        } catch (NumberFormatException e) {
            return Optional.empty();
        }
    }

    /**
     * Attempts to parse a string as an HTTP date (RFC 1123), returning empty Optional on failure.
     */
    private Optional<ZonedDateTime> tryParseHttpDate(String value) {
        if (value == null) {
            return Optional.empty();
        }
        try {
            return Optional.of(ZonedDateTime.parse(value, DateTimeFormatter.RFC_1123_DATE_TIME));
        } catch (DateTimeParseException e) {
            return Optional.empty();
        }
    }

    /**
     * Adds positive jitter (100-120% of original value) to prevent thundering herd.
     * Used for X-RateLimit-Reset header delays.
     */
    private long addPositiveJitter(long delayMs) {
        double jitterMultiplier = 1.0 + (random.nextDouble() * JITTER_FACTOR);
        return (long) (delayMs * jitterMultiplier);
    }

    /**
     * Adds symmetric jitter (90-110% of original value) to prevent thundering herd.
     * Used for exponential backoff delays.
     */
    private long addSymmetricJitter(long delayMs) {
        double jitterMultiplier = 1.0 + ((random.nextDouble() - 0.5) * JITTER_FACTOR);
        return (long) (delayMs * jitterMultiplier);
    }

    private static boolean shouldRetry(int statusCode) {
        return statusCode == 408 || statusCode == 429 || statusCode >= 500;
    }

    private final class ExponentialBackoff {

        private final int maxNumRetries;

        private int retryNumber = 0;

        ExponentialBackoff(int maxNumRetries) {
            this.maxNumRetries = maxNumRetries;
        }

        public Optional<Duration> nextBackoff(Response response) {
            if (retryNumber >= maxNumRetries) {
                return Optional.empty();
            }

            Duration delay = getRetryDelayFromHeaders(response, retryNumber);
            retryNumber += 1;
            return Optional.of(delay);
        }
    }
}
