/*
 * Decompiled with CFR 0.152.
 */
package com.sendly;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.sendly.exceptions.AuthenticationException;
import com.sendly.exceptions.InsufficientCreditsException;
import com.sendly.exceptions.NetworkException;
import com.sendly.exceptions.NotFoundException;
import com.sendly.exceptions.RateLimitException;
import com.sendly.exceptions.SendlyException;
import com.sendly.exceptions.ValidationException;
import com.sendly.resources.Messages;
import java.io.IOException;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class Sendly {
    public static final String VERSION = "3.0.1";
    public static final String DEFAULT_BASE_URL = "https://sendly.live/api/v1";
    public static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(30L);
    private final String apiKey;
    private final String baseUrl;
    private final OkHttpClient httpClient;
    private final Gson gson;
    private final int maxRetries;
    private final Messages messages;

    public Sendly(String apiKey) {
        this(apiKey, new Builder());
    }

    public Sendly(String apiKey, Builder builder) {
        if (apiKey == null || apiKey.isEmpty()) {
            throw new AuthenticationException("API key is required");
        }
        this.apiKey = apiKey;
        this.baseUrl = builder.baseUrl;
        this.maxRetries = builder.maxRetries;
        this.httpClient = new OkHttpClient.Builder().connectTimeout(builder.connectTimeout.toMillis(), TimeUnit.MILLISECONDS).readTimeout(builder.readTimeout.toMillis(), TimeUnit.MILLISECONDS).writeTimeout(builder.writeTimeout.toMillis(), TimeUnit.MILLISECONDS).build();
        this.gson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").create();
        this.messages = new Messages(this);
    }

    public Messages messages() {
        return this.messages;
    }

    public JsonObject get(String path, Map<String, String> params) throws SendlyException {
        HttpUrl.Builder urlBuilder = HttpUrl.parse((String)(this.baseUrl + path)).newBuilder();
        if (params != null) {
            params.forEach((key, value) -> {
                if (value != null) {
                    urlBuilder.addQueryParameter(key, value);
                }
            });
        }
        Request request = new Request.Builder().url(urlBuilder.build()).get().addHeader("Authorization", "Bearer " + this.apiKey).addHeader("Accept", "application/json").addHeader("User-Agent", "sendly-java/3.0.1").build();
        return this.executeWithRetry(request);
    }

    public JsonObject post(String path, Object body) throws SendlyException {
        String json = this.gson.toJson(body);
        RequestBody requestBody = RequestBody.create((String)json, (MediaType)MediaType.parse((String)"application/json"));
        Request request = new Request.Builder().url(this.baseUrl + path).post(requestBody).addHeader("Authorization", "Bearer " + this.apiKey).addHeader("Content-Type", "application/json").addHeader("Accept", "application/json").addHeader("User-Agent", "sendly-java/3.0.1").build();
        return this.executeWithRetry(request);
    }

    public JsonObject delete(String path) throws SendlyException {
        Request request = new Request.Builder().url(this.baseUrl + path).delete().addHeader("Authorization", "Bearer " + this.apiKey).addHeader("Accept", "application/json").addHeader("User-Agent", "sendly-java/3.0.1").build();
        return this.executeWithRetry(request);
    }

    private JsonObject executeWithRetry(Request request) throws SendlyException {
        SendlyException lastException = null;
        for (int attempt = 0; attempt <= this.maxRetries; ++attempt) {
            if (attempt > 0) {
                try {
                    long delay = (long)Math.pow(2.0, attempt - 1) * 1000L;
                    Thread.sleep(delay);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new NetworkException("Request interrupted");
                }
            }
            try {
                return this.execute(request);
            }
            catch (AuthenticationException | InsufficientCreditsException | NotFoundException | ValidationException e) {
                throw e;
            }
            catch (RateLimitException e) {
                if (e.getRetryAfter() > 0) {
                    try {
                        Thread.sleep((long)e.getRetryAfter() * 1000L);
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                lastException = e;
                continue;
            }
            catch (SendlyException e) {
                lastException = e;
            }
        }
        throw lastException != null ? lastException : new SendlyException("Request failed after retries");
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private JsonObject execute(Request request) throws SendlyException {
        try (Response response = this.httpClient.newCall(request).execute();){
            SendlyException sendlyException;
            String body;
            String string = body = response.body() != null ? response.body().string() : "";
            if (response.isSuccessful()) {
                JsonObject jsonObject = body.isEmpty() ? new JsonObject() : (JsonObject)this.gson.fromJson(body, JsonObject.class);
                return jsonObject;
            }
            JsonObject error = body.isEmpty() ? new JsonObject() : (JsonObject)this.gson.fromJson(body, JsonObject.class);
            String message = error.has("message") ? error.get("message").getAsString() : "Unknown error";
            switch (response.code()) {
                case 401: {
                    sendlyException = new AuthenticationException(message);
                    throw sendlyException;
                }
                case 402: {
                    sendlyException = new InsufficientCreditsException(message);
                    throw sendlyException;
                }
                case 404: {
                    sendlyException = new NotFoundException(message);
                    throw sendlyException;
                }
                case 429: {
                    String retryAfter = response.header("Retry-After");
                    int seconds = retryAfter != null ? Integer.parseInt(retryAfter) : 0;
                    sendlyException = new RateLimitException(message, seconds);
                    throw sendlyException;
                }
                case 400: 
                case 422: {
                    sendlyException = new ValidationException(message);
                    throw sendlyException;
                }
            }
            sendlyException = new SendlyException(message, response.code());
            throw sendlyException;
        }
        catch (IOException e) {
            throw new NetworkException("Request failed: " + e.getMessage());
        }
    }

    public Gson getGson() {
        return this.gson;
    }

    public static class Builder {
        private String baseUrl = "https://sendly.live/api/v1";
        private Duration connectTimeout = Duration.ofSeconds(10L);
        private Duration readTimeout = DEFAULT_TIMEOUT;
        private Duration writeTimeout = DEFAULT_TIMEOUT;
        private int maxRetries = 3;

        public Builder baseUrl(String baseUrl) {
            this.baseUrl = baseUrl;
            return this;
        }

        public Builder connectTimeout(Duration timeout) {
            this.connectTimeout = timeout;
            return this;
        }

        public Builder readTimeout(Duration timeout) {
            this.readTimeout = timeout;
            return this;
        }

        public Builder writeTimeout(Duration timeout) {
            this.writeTimeout = timeout;
            return this;
        }

        public Builder timeout(Duration timeout) {
            this.readTimeout = timeout;
            this.writeTimeout = timeout;
            return this;
        }

        public Builder maxRetries(int maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }
    }
}

