package net.java.ao;

import net.java.ao.sql.LoggingInterceptor;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;

class DelegateLoggingStatement extends DelegateStatement {
    protected final LoggingInterceptor logger;
    protected final Map<Integer, String> params = new HashMap<>();

    DelegateLoggingStatement(final Statement statement, final LoggingInterceptor logger) {
        super(statement);
        this.logger = logger;
    }

    protected final StringBuilder batchQueryBuffer = new StringBuilder();

    @FunctionalInterface
    protected interface DelegateBlock<T> {
        T invoke() throws SQLException;
    }

    <T> T delegateExecute(String sql, DelegateBlock<T> block) throws SQLException {
        logger.beforeExecution();

        try {
            T result = block.invoke();

            logger.afterSuccessfulExecution(sql, params);

            return result;
        } catch (SQLException e) {
            logger.onException(sql, params, e);
            throw e;
        }
    }

    @Override
    public boolean execute(String sql) throws SQLException {
        return delegateExecute(sql, () -> statement.execute(sql));
    }

    @Override
    public void addBatch(String sql) throws SQLException {
        statement.addBatch(sql);

        if (batchQueryBuffer.length() > 0) {
            batchQueryBuffer.append("\n");
        }

        batchQueryBuffer.append(sql);
    }

    @Override
    public void clearBatch() throws SQLException {
        statement.clearBatch();

        batchQueryBuffer.setLength(0);
    }

    @Override
    public int[] executeBatch() throws SQLException {
        logger.beforeExecution();

        try {
            int[] result = statement.executeBatch();

            logger.afterSuccessfulExecution(batchQueryBuffer.toString());

            return result;
        } catch (SQLException e) {
            logger.onException(batchQueryBuffer.toString(), e);
            throw e;
        }
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        return delegateExecute(sql, () -> statement.execute(sql, autoGeneratedKeys));
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        return delegateExecute(sql, () -> statement.execute(sql, columnIndexes));
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        return delegateExecute(sql, () -> statement.execute(sql, columnNames));
    }

    @Override
    public long executeLargeUpdate(String sql) throws SQLException {
        return delegateExecute(sql, () -> statement.executeLargeUpdate(sql));
    }

    @Override
    public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        return delegateExecute(sql, () -> statement.executeLargeUpdate(sql, autoGeneratedKeys));
    }

    @Override
    public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
        return delegateExecute(sql, () -> statement.executeLargeUpdate(sql, columnIndexes));
    }

    @Override
    public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
        return delegateExecute(sql, () -> statement.executeLargeUpdate(sql, columnNames));
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        return delegateExecute(sql, () -> statement.executeQuery(sql));
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        return delegateExecute(sql, () -> statement.executeUpdate(sql));
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        return delegateExecute(sql, () -> statement.executeUpdate(sql, autoGeneratedKeys));
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        return delegateExecute(sql, () -> statement.executeUpdate(sql, columnIndexes));
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        return delegateExecute(sql, () -> statement.executeUpdate(sql, columnNames));
    }

}
