/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apithefire.sql.h2;

import org.apithefire.util.lang.Objects;

/**
 * A compatibility mode to other DBMS.
 * 
 * @see <a
 *      href="http://www.h2database.com/html/features.html#compatibility">Compatibility</a>
 * @author <a href="mailto:bachew@gmail.com">Chew Boon Aik</a>
 */
public enum CompatibilityMode {

	/**
	 * Concatenation of a NULL with another value results in NULL. Usually, the
	 * NULL is treated as an empty string if only one of the operators is NULL,
	 * and NULL is only returned if both values are NULL.
	 */
	POSTGRESQL("POSTGRESQL"),
	
	/**
	 * When inserting data, if a column is defined to be NOT NULL and NULL is
	 * inserted, then a 0 (or empty string, or the current timestamp for
	 * timestamp columns) value is used. Usually, this operation is not allowed
	 * and an exception is thrown.
	 */
	MYSQL("MYSQL"),
	
	/**
	 * When converting the scale of decimal data, the number is only converted
	 * if the new scale is smaller then current scale. Usually, the scale is
	 * converted and 0s are added if required.
	 */
	HSQLDB("HSQLDB");
	
	private String parameterValue;
	
	private CompatibilityMode(String parameterValue) {
		this.parameterValue = Objects.nonNull(parameterValue);
	}
	
	protected String getParameterValue() {
		return parameterValue;
	}
	
}
