/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apithefire.sql.h2;

import org.apithefire.util.lang.Objects;
import org.apithefire.util.lang.Path;
import org.h2.jdbcx.JdbcDataSource;

class H2DataSourceBuilder {

	private String url;
	private Cipher cipher;
	private FileLock fileLock;
	private Boolean openIfExists;
	private Boolean closeDatabaseOnExit;
	private Integer databaseCloseDelay;
	private boolean logIndexChanges;
	private TraceLevel fileTraceLevel;
	private TraceLevel consoleTraceLevel;
	private AccessMode logAccessMode;
	private AccessMode dataAccessMode;
	private Boolean ignoreCase;
	private CompatibilityMode compatibilityMode;
	// TODO: TRACE_MAX_FILE_SIZE
	private String username;
	private String password;

	public JdbcDataSource build() {
		JdbcDataSource dataSource = new JdbcDataSource();
		
		dataSource.setURL(buildFullUrl());
		
		if (username != null) {
			dataSource.setUser(username);
		}
		if (password != null) {
			dataSource.setPassword(password);
		}
		return dataSource;
	}
	
	private String buildFullUrl() {
		StringBuilder fullUrl = new StringBuilder();
		
		fullUrl.append("jdbc:h2:");
		
		if (url != null) {
			fullUrl.append(url);
		}
		if (cipher != null) {
			fullUrl.append(";CIPHER=");
			fullUrl.append(cipher.getParameterValue());
		}
		if (fileLock != null) {
			fullUrl.append(";FILE_LOCK=");
			fullUrl.append(fileLock.getParameterValue());
		}
		if (openIfExists != null) {
			fullUrl.append(";IFEXISTS=");
			fullUrl.append(openIfExists.toString().toUpperCase());
		}
		if (closeDatabaseOnExit != null) {
			fullUrl.append(";DB_CLOSE_ON_EXIT=");
			fullUrl.append(closeDatabaseOnExit.toString().toUpperCase());
		}
		if (databaseCloseDelay != null) {
			fullUrl.append(";DB_CLOSE_DELAY=");
			fullUrl.append(databaseCloseDelay);
		}
		if (fileTraceLevel != null) {
			fullUrl.append(";TRACE_LEVEL_FILE=");
			fullUrl.append(fileTraceLevel.getParameterValue());
		}
		if (consoleTraceLevel != null) {
			fullUrl.append(";TRACE_LEVEL_SYSTEM_OUT=");
			fullUrl.append(consoleTraceLevel.getParameterValue());
		}
		if (logIndexChanges) {
			fullUrl.append(";LOG=2");
		}
		if (logAccessMode != null) {
			fullUrl.append(";ACCESS_MODE_LOG=");
			fullUrl.append(logAccessMode.getParameterValue());
		}
		if (dataAccessMode != null) {
			fullUrl.append(";ACCESS_MODE_DATA=");
			fullUrl.append(dataAccessMode.getParameterValue());
		}
		if (ignoreCase != null) {
			fullUrl.append(";IGNORECASE=");
			fullUrl.append(ignoreCase.toString().toUpperCase());
		}
		if (compatibilityMode != null) {
			fullUrl.append(";MODE=");
			fullUrl.append(compatibilityMode.getParameterValue());
		}
		return fullUrl.toString();
	}
	
	public H2DataSourceBuilder setMemoryUrl(String databaseName) {
		this.url = "mem:" + Objects.nonNull(databaseName);
		return this;
	}
	
	public H2DataSourceBuilder setFileUrl(Path dirPath, String databaseName) {
		// TODO: dirPath might contains semicolon
		this.url = "file:" + dirPath.trimRight() + "/"
				+ Objects.nonNull(databaseName);
		return this;
	}
	
	public H2DataSourceBuilder setZipUrl(Path zipFilePath, Path entryPath,
			String databaseName) {
		this.url = "zip:" + zipFilePath.trimRight() + "!/" + entryPath.trim()
				+ "/" + databaseName;
		return this;
	}
	
	public H2DataSourceBuilder setTcpUrl(String host, Integer port, String databaseName) {
		return setRemoteUrl("tcp", host, port, databaseName);
	}

	public H2DataSourceBuilder setSslUrl(String host, Integer port, String databaseName) {
		return setRemoteUrl("ssl", host, port, databaseName);
	}
	
	private H2DataSourceBuilder setRemoteUrl(String protocol, String host,
			Integer port, String databaseName) {
		StringBuilder urlBuilder = new StringBuilder();
		
		urlBuilder.append(Objects.nonNull(protocol));
		urlBuilder.append("://");
		urlBuilder.append(Objects.nonNull(host));
		
		if (port != null) {
			urlBuilder.append(':');
			urlBuilder.append(port);
		}
		urlBuilder.append('/');
		urlBuilder.append(Objects.nonNull(databaseName));
		this.url = urlBuilder.toString();
		return this;
	}
	
	public H2DataSourceBuilder setCipher(Cipher cipher) {
		this.cipher = cipher;
		return this;
	}
	
	public H2DataSourceBuilder setFileLock(FileLock fileLock) {
		this.fileLock = fileLock;
		return this;
	}
	
	public H2DataSourceBuilder setOpenIfExists(Boolean openIfExists) {
		this.openIfExists = openIfExists;
		return this;
	}
	
	public H2DataSourceBuilder setCloseDatabaseOnExit(Boolean closeDatabaseOnExit) {
		this.closeDatabaseOnExit = closeDatabaseOnExit;
		return this;
	}

	public H2DataSourceBuilder setDatabaseCloseDelay(Integer delay) {
		this.databaseCloseDelay = delay;
		return this;
	}
	
	public H2DataSourceBuilder setLogIndexChanges(boolean logIndexChanges) {
		this.logIndexChanges = logIndexChanges;
		return this;
	}
	
	public H2DataSourceBuilder setFileTraceLevel(TraceLevel traceLevel) {
		this.fileTraceLevel = traceLevel;
		return this;
	}
	
	public H2DataSourceBuilder setConsoleTraceLevel(TraceLevel traceLevel) {
		this.consoleTraceLevel = traceLevel;
		return this;
	}
	
	public H2DataSourceBuilder setLogAccessMode(AccessMode accessMode) {
		this.logAccessMode = accessMode;
		return this;
	}
	
	public H2DataSourceBuilder setDataAccessMode(AccessMode accessMode) {
		this.dataAccessMode = accessMode;
		return this;
	}

	public H2DataSourceBuilder setIgnoreCase(Boolean ignoreCase) {
		this.ignoreCase = ignoreCase;
		return this;
	}

	public H2DataSourceBuilder setCompatibilityMode(
			CompatibilityMode compatibilityMode) {
		this.compatibilityMode = compatibilityMode;
		return this;
	}

	public H2DataSourceBuilder setUsername(String username) {
		this.username = username;
		return this;
	}

	public H2DataSourceBuilder setPassword(String password) {
		this.password = password;
		return this;
	}
	
}
