/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apithefire.sql.h2;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import org.apithefire.sql.RuntimeSqlException;
import org.apithefire.sql.connect.ConnectionProvider;
import org.apithefire.util.lang.BuilderFactory;
import org.h2.jdbcx.JdbcDataSource;

/**
 * <a href="http://www.h2database.com/">H2</a> memory database.
 * 
 * @see <a
 *      href="http://www.h2database.com/html/features.html#memory_only_databases">
 *      Memory-Only Databases</a>
 * @author <a href="mailto:bachew@gmail.com">Chew Boon Aik</a>
 */
// TODO: write unit test
public class H2MemoryDatabase implements ConnectionProvider {
	
	/**
	 * A {@link H2MemoryDatabase} builder.
	 * 
	 * @author <a href="mailto:bachew@gmail.com">Chew Boon Aik</a>
	 */
	public static class H2MemoryDatabaseBuilder {
		
		private final H2DataSourceBuilder dataSourceBuilder;
		
		private H2MemoryDatabaseBuilder() {
			dataSourceBuilder = new H2DataSourceBuilder();
		}
		
		// TODO: more properties
		
		/**
		 * Builds a new {@link H2MemoryDatabase}.
		 * 
		 * @return the built {@link H2MemoryDatabase}.
		 */
		public H2MemoryDatabase build() {
			H2MemoryDatabase db = new H2MemoryDatabase();
			
			dataSourceBuilder.setMemoryUrl(db.generateUniqueName());
			dataSourceBuilder.setFileTraceLevel(TraceLevel.SLF4J);
			dataSourceBuilder.setConsoleTraceLevel(TraceLevel.SLF4J);
			dataSourceBuilder.setDatabaseCloseDelay(-1);
			db.dataSource = dataSourceBuilder.build();
			return db;
		}
		
	}
	
	private JdbcDataSource dataSource;
	
	private H2MemoryDatabase() {
	}
	
	/**
	 * Returns a new {@link H2MemoryDatabaseBuilder}.
	 */
	@BuilderFactory
	public static H2MemoryDatabaseBuilder newBuilder() {
		return new H2MemoryDatabaseBuilder();
	}
	
	private String generateUniqueName() {
		return getClass().getSimpleName() + super.hashCode();
	}

	/**
	 * Gets a new connection from this database.
	 */
	public synchronized Connection getConnection() {
		if (dataSource == null) {
			throw new RuntimeSqlException("Database is closed.");
		}
		try {
			return dataSource.getConnection();
		} catch (SQLException e) {
			throw new RuntimeSqlException(e);
		}
	}
	
	private void closeThrows() throws SQLException {
		if (dataSource == null) {
			return;
		}
		
		Connection connection = null;
		Statement statement = null;
		
		try {
			connection = dataSource.getConnection();
			statement = connection.createStatement();
			statement.executeUpdate("SHUTDOWN");
			dataSource = null;
		} finally {
			if (statement != null) {
				statement.close();
			}
			if (connection != null) {
				connection.close();
			}
		}
	}

	/**
	 * Closes this database by executing a {@code SHUTDOWN} SQL statement.
	 */
	public synchronized void close() {
		try {
			closeThrows();
		} catch (SQLException e) {
			throw new RuntimeSqlException(e);
		}
	}

}
