/*
 * Decompiled with CFR 0.152.
 */
package org.biojava.nbio.structure.asa;

import java.util.ArrayList;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.vecmath.Point3d;
import org.biojava.nbio.structure.AminoAcid;
import org.biojava.nbio.structure.Atom;
import org.biojava.nbio.structure.Calc;
import org.biojava.nbio.structure.Element;
import org.biojava.nbio.structure.Group;
import org.biojava.nbio.structure.GroupType;
import org.biojava.nbio.structure.NucleotideImpl;
import org.biojava.nbio.structure.ResidueNumber;
import org.biojava.nbio.structure.Structure;
import org.biojava.nbio.structure.StructureTools;
import org.biojava.nbio.structure.asa.GroupAsa;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsaCalculator {
    private static final Logger logger = LoggerFactory.getLogger(AsaCalculator.class);
    public static final int DEFAULT_N_SPHERE_POINTS = 960;
    public static final double DEFAULT_PROBE_SIZE = 1.4;
    public static final int DEFAULT_NTHREADS = 1;
    public static final double TRIGONAL_CARBON_VDW = 1.76;
    public static final double TETRAHEDRAL_CARBON_VDW = 1.87;
    public static final double TRIGONAL_NITROGEN_VDW = 1.65;
    public static final double TETRAHEDRAL_NITROGEN_VDW = 1.5;
    public static final double SULFUR_VDW = 1.85;
    public static final double OXIGEN_VDW = 1.4;
    public static final double NUC_CARBON_VDW = 1.8;
    public static final double NUC_NITROGEN_VDW = 1.6;
    public static final double PHOSPHOROUS_VDW = 1.9;
    private Point3d[] atomCoords;
    private Atom[] atoms;
    private double[] radii;
    private double probe;
    private int nThreads;
    private Point3d[] spherePoints;
    private double cons;

    public AsaCalculator(Structure structure, double probe, int nSpherePoints, int nThreads, boolean hetAtoms) {
        this.atoms = StructureTools.getAllNonHAtomArray(structure, hetAtoms);
        this.atomCoords = Calc.atomsToPoints(this.atoms);
        this.probe = probe;
        this.nThreads = nThreads;
        this.radii = new double[this.atomCoords.length];
        for (int i = 0; i < this.atomCoords.length; ++i) {
            this.radii[i] = AsaCalculator.getRadius(this.atoms[i]);
        }
        this.spherePoints = this.generateSpherePoints(nSpherePoints);
        this.cons = Math.PI * 4 / (double)nSpherePoints;
    }

    public AsaCalculator(Atom[] atoms, double probe, int nSpherePoints, int nThreads) {
        this.atoms = atoms;
        this.atomCoords = Calc.atomsToPoints(atoms);
        this.probe = probe;
        this.nThreads = nThreads;
        for (Atom atom : atoms) {
            if (atom.getElement() != Element.H) continue;
            throw new IllegalArgumentException("Can't calculate ASA for an array that contains Hydrogen atoms ");
        }
        this.radii = new double[atoms.length];
        for (int i = 0; i < atoms.length; ++i) {
            this.radii[i] = AsaCalculator.getRadius(atoms[i]);
        }
        this.spherePoints = this.generateSpherePoints(nSpherePoints);
        this.cons = Math.PI * 4 / (double)nSpherePoints;
    }

    public AsaCalculator(Point3d[] atomCoords, double probe, int nSpherePoints, int nThreads, double radius) {
        this.atoms = null;
        this.atomCoords = atomCoords;
        this.probe = probe;
        this.nThreads = nThreads;
        this.radii = new double[atomCoords.length];
        for (int i = 0; i < atomCoords.length; ++i) {
            this.radii[i] = radius;
        }
        this.spherePoints = this.generateSpherePoints(nSpherePoints);
        this.cons = Math.PI * 4 / (double)nSpherePoints;
    }

    public GroupAsa[] getGroupAsas() {
        TreeMap<ResidueNumber, GroupAsa> asas = new TreeMap<ResidueNumber, GroupAsa>();
        double[] asasPerAtom = this.calculateAsas();
        for (int i = 0; i < this.atomCoords.length; ++i) {
            GroupAsa groupAsa;
            Group g = this.atoms[i].getGroup();
            if (!asas.containsKey(g.getResidueNumber())) {
                groupAsa = new GroupAsa(g);
                groupAsa.addAtomAsaU(asasPerAtom[i]);
                asas.put(g.getResidueNumber(), groupAsa);
                continue;
            }
            groupAsa = (GroupAsa)asas.get(g.getResidueNumber());
            groupAsa.addAtomAsaU(asasPerAtom[i]);
        }
        return asas.values().toArray(new GroupAsa[asas.size()]);
    }

    public double[] calculateAsas() {
        double[] asas = new double[this.atomCoords.length];
        if (this.nThreads <= 1) {
            for (int i = 0; i < this.atomCoords.length; ++i) {
                asas[i] = this.calcSingleAsa(i);
            }
        } else {
            ExecutorService threadPool = Executors.newFixedThreadPool(this.nThreads);
            for (int i = 0; i < this.atomCoords.length; ++i) {
                threadPool.submit(new AsaCalcWorker(i, asas));
            }
            threadPool.shutdown();
            while (!threadPool.isTerminated()) {
            }
        }
        return asas;
    }

    private Point3d[] generateSpherePoints(int nSpherePoints) {
        Point3d[] points = new Point3d[nSpherePoints];
        double inc = Math.PI * (3.0 - Math.sqrt(5.0));
        double offset = 2.0 / (double)nSpherePoints;
        for (int k = 0; k < nSpherePoints; ++k) {
            double y = (double)k * offset - 1.0 + offset / 2.0;
            double r = Math.sqrt(1.0 - y * y);
            double phi = (double)k * inc;
            points[k] = new Point3d(Math.cos(phi) * r, y, Math.sin(phi) * r);
        }
        return points;
    }

    private ArrayList<Integer> findNeighborIndices(int k) {
        ArrayList<Integer> neighbor_indices = new ArrayList<Integer>(40);
        double radius = this.radii[k] + this.probe + this.probe;
        for (int i = 0; i < this.atomCoords.length; ++i) {
            if (i == k) continue;
            double dist = 0.0;
            dist = this.atomCoords[i].distance(this.atomCoords[k]);
            if (!(dist < radius + this.radii[i])) continue;
            neighbor_indices.add(i);
        }
        return neighbor_indices;
    }

    private double calcSingleAsa(int i) {
        Point3d atom_i = this.atomCoords[i];
        ArrayList<Integer> neighbor_indices = this.findNeighborIndices(i);
        int n_neighbor = neighbor_indices.size();
        int j_closest_neighbor = 0;
        double radius = this.probe + this.radii[i];
        int n_accessible_point = 0;
        for (Point3d point : this.spherePoints) {
            boolean is_accessible = true;
            Point3d test_point = new Point3d(point.x * radius + atom_i.x, point.y * radius + atom_i.y, point.z * radius + atom_i.z);
            int[] cycled_indices = new int[n_neighbor];
            int arind = 0;
            int ind = j_closest_neighbor;
            while (ind < n_neighbor) {
                cycled_indices[arind] = ind++;
                ++arind;
            }
            ind = 0;
            while (ind < j_closest_neighbor) {
                cycled_indices[arind] = ind++;
                ++arind;
            }
            for (int j : cycled_indices) {
                Point3d atom_j = this.atomCoords[neighbor_indices.get(j)];
                double r = this.radii[neighbor_indices.get(j)] + this.probe;
                double diff_sq = test_point.distanceSquared(atom_j);
                if (!(diff_sq < r * r)) continue;
                j_closest_neighbor = j;
                is_accessible = false;
                break;
            }
            if (!is_accessible) continue;
            ++n_accessible_point;
        }
        return this.cons * (double)n_accessible_point * radius * radius;
    }

    private static double getRadiusForAmino(AminoAcid amino, Atom atom) {
        if (atom.getElement().equals((Object)Element.H)) {
            return Element.H.getVDWRadius();
        }
        if (atom.getElement().equals((Object)Element.D)) {
            return Element.D.getVDWRadius();
        }
        String atomCode = atom.getName();
        char aa = amino.getAminoType().charValue();
        if (atom.getElement() == Element.O) {
            return 1.4;
        }
        if (atom.getElement() == Element.S) {
            return 1.85;
        }
        if (atom.getElement() == Element.N) {
            if (atomCode.equals("NZ")) {
                return 1.5;
            }
            return 1.65;
        }
        if (atom.getElement() == Element.C) {
            if (atomCode.equals("C") || atomCode.equals("CE1") || atomCode.equals("CE2") || atomCode.equals("CE3") || atomCode.equals("CH2") || atomCode.equals("CZ") || atomCode.equals("CZ2") || atomCode.equals("CZ3")) {
                return 1.76;
            }
            if (atomCode.equals("CA") || atomCode.equals("CB") || atomCode.equals("CE") || atomCode.equals("CG1") || atomCode.equals("CG2")) {
                return 1.87;
            }
            switch (aa) {
                case 'D': 
                case 'F': 
                case 'H': 
                case 'N': 
                case 'W': 
                case 'Y': {
                    return 1.76;
                }
                case 'I': 
                case 'K': 
                case 'L': 
                case 'M': 
                case 'P': 
                case 'R': {
                    return 1.87;
                }
                case 'E': 
                case 'Q': {
                    if (atomCode.equals("CD")) {
                        return 1.76;
                    }
                    if (!atomCode.equals("CG")) break;
                    return 1.87;
                }
            }
            logger.info("Unexpected carbon atom " + atomCode + " for aminoacid " + aa + ", assigning its standard vdw radius");
            return Element.C.getVDWRadius();
        }
        logger.info("Unexpected atom " + atomCode + " for aminoacid " + aa + " (" + amino.getPDBName() + "), assigning its standard vdw radius");
        return atom.getElement().getVDWRadius();
    }

    private static double getRadiusForNucl(NucleotideImpl nuc, Atom atom) {
        if (atom.getElement().equals((Object)Element.H)) {
            return Element.H.getVDWRadius();
        }
        if (atom.getElement().equals((Object)Element.D)) {
            return Element.D.getVDWRadius();
        }
        if (atom.getElement() == Element.C) {
            return 1.8;
        }
        if (atom.getElement() == Element.N) {
            return 1.6;
        }
        if (atom.getElement() == Element.P) {
            return 1.9;
        }
        if (atom.getElement() == Element.O) {
            return 1.4;
        }
        logger.info("Unexpected atom " + atom.getName() + " for nucleotide " + nuc.getPDBName() + ", assigning its standard vdw radius");
        return atom.getElement().getVDWRadius();
    }

    public static double getRadius(Atom atom) {
        if (atom.getElement() == null) {
            logger.warn("Unrecognised atom " + atom.getName() + " with serial " + atom.getPDBserial() + ", assigning the default vdw radius (Nitrogen vdw radius).");
            return Element.N.getVDWRadius();
        }
        Group res = atom.getGroup();
        if (res == null) {
            logger.warn("Unknown parent residue for atom " + atom.getName() + " with serial " + atom.getPDBserial() + ", assigning its default vdw radius");
            return atom.getElement().getVDWRadius();
        }
        GroupType type = res.getType();
        if (type == GroupType.AMINOACID) {
            return AsaCalculator.getRadiusForAmino((AminoAcid)res, atom);
        }
        if (type == GroupType.NUCLEOTIDE) {
            return AsaCalculator.getRadiusForNucl((NucleotideImpl)res, atom);
        }
        return atom.getElement().getVDWRadius();
    }

    private class AsaCalcWorker
    implements Runnable {
        private int i;
        private double[] asas;

        public AsaCalcWorker(int i, double[] asas) {
            this.i = i;
            this.asas = asas;
        }

        @Override
        public void run() {
            this.asas[this.i] = AsaCalculator.this.calcSingleAsa(this.i);
        }
    }
}

