/*
 * Decompiled with CFR 0.152.
 */
package org.expath.pkg.repo;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import org.expath.pkg.repo.Extension;
import org.expath.pkg.repo.FileHelper;
import org.expath.pkg.repo.FileSystemStorage;
import org.expath.pkg.repo.Package;
import org.expath.pkg.repo.PackageException;
import org.expath.pkg.repo.Packages;
import org.expath.pkg.repo.Storage;
import org.expath.pkg.repo.URISpace;
import org.expath.pkg.repo.Universe;
import org.expath.pkg.repo.UserInteractionStrategy;
import org.expath.pkg.repo.XarSource;
import org.expath.pkg.repo.XarUriSource;
import org.expath.pkg.repo.ZipHelper;
import org.expath.pkg.repo.parser.DescriptorParser;
import org.expath.pkg.repo.tools.Logger;

public class Repository
implements Universe {
    private final Storage myStorage;
    private Map<String, Packages> myPackages;
    private Map<String, Extension> myExtensions;
    private static final Logger LOG = Logger.getLogger(Repository.class);

    public Repository(Storage storage) throws PackageException {
        LOG.info("Create a new repository with storage: {0}", storage);
        this.myStorage = storage;
        this.myPackages = new HashMap<String, Packages>();
        this.myExtensions = new HashMap<String, Extension>();
        ServiceLoader<Extension> loader = ServiceLoader.load(Extension.class);
        for (Extension e : loader) {
            this.registerExtension(e);
        }
        this.parsePublicUris();
    }

    public static Repository makeDefaultRepo() throws PackageException {
        return Repository.makeDefaultRepo(null);
    }

    public static Repository makeDefaultRepo(String dir) throws PackageException {
        if (dir == null) {
            dir = System.getProperty("expath.repo");
        }
        if (dir == null) {
            dir = System.getenv("EXPATH_REPO");
        }
        if (dir != null) {
            Path f = Paths.get(dir, new String[0]);
            if (!Files.exists(f, new LinkOption[0])) {
                throw new PackageException("Repo directory does not exist: " + dir);
            }
            if (!Files.isDirectory(f, new LinkOption[0])) {
                throw new PackageException("Repo is not a directory: " + dir);
            }
            try {
                FileSystemStorage storage = new FileSystemStorage(f);
                return new Repository(storage);
            }
            catch (PackageException ex) {
                throw new PackageException("Error setting the repo (" + dir + ")", ex);
            }
        }
        return null;
    }

    public final void registerExtension(Extension ext) throws PackageException {
        if (!this.myExtensions.containsKey(ext.getName())) {
            this.myExtensions.put(ext.getName(), ext);
            ext.init(this, this.myPackages);
        }
    }

    public synchronized void reload() throws PackageException {
        this.myPackages = new HashMap<String, Packages>();
        this.parsePublicUris();
    }

    public Storage getStorage() {
        return this.myStorage;
    }

    public Collection<Packages> listPackages() {
        return this.myPackages.values();
    }

    public Packages getPackages(String name) {
        return this.myPackages.get(name);
    }

    public static Repository createRepository(Path dir) throws PackageException {
        if (Files.exists(dir, new LinkOption[0])) {
            if (!Files.isDirectory(dir, new LinkOption[0])) {
                throw new PackageException("File exists and is not a directory (" + dir + ")");
            }
            if (!FileHelper.isEmpty(dir)) {
                throw new PackageException("Directory exists and is not empty (" + dir + ")");
            }
        } else {
            try {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new PackageException("Error creating the directory (" + dir + ")", e);
            }
        }
        Path priv_dir = dir.resolve(".expath-pkg");
        try {
            Files.createDirectories(priv_dir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new PackageException("Error creating the private directory (" + priv_dir + ")", e);
        }
        return new Repository(new FileSystemStorage(dir));
    }

    public Package installPackage(URI pkg, boolean force, UserInteractionStrategy interact) throws PackageException {
        if (this.myStorage.isReadOnly()) {
            throw new PackageException("The storage is read-only, package install not supported");
        }
        return this.installPackage(new XarUriSource(pkg), force, interact);
    }

    public Package installPackage(XarSource xarSource, boolean force, UserInteractionStrategy interact) throws PackageException {
        Package p2;
        Package pkg;
        if (!xarSource.isValid()) {
            throw new PackageException("Package file does not exist (" + xarSource.getURI() + ")");
        }
        this.myStorage.beforeInstall(force, interact);
        Path tmp_dir = this.myStorage.makeTempDir("install");
        try {
            ZipHelper zip = new ZipHelper(xarSource);
            zip.unzip(tmp_dir);
        }
        catch (IOException ex) {
            throw new PackageException("Error unziping the package", ex);
        }
        interact.logInfo("Package unziped to " + tmp_dir);
        Path desc_f = tmp_dir.resolve("expath-pkg.xml");
        if (!Files.exists(desc_f, new LinkOption[0])) {
            throw new PackageException("Package descriptor does NOT exist in: " + tmp_dir);
        }
        try (InputStream is = Files.newInputStream(desc_f, new OpenOption[0]);){
            StreamSource desc = new StreamSource(is);
            DescriptorParser parser = new DescriptorParser();
            pkg = parser.parse(desc, null, this.myStorage, this);
        }
        catch (IOException e) {
            throw new PackageException(e.getMessage(), e);
        }
        String name = pkg.getName();
        String version = pkg.getVersion();
        Packages pp = this.myPackages.get(name);
        if (pp != null && (p2 = pp.version(version)) != null) {
            if (force || interact.ask("Force override " + name + " - " + version + "?", false)) {
                this.myStorage.remove(p2);
                pp.remove(p2);
                if (pp.latest() == null) {
                    this.myPackages.remove(name);
                }
            } else {
                throw new AlreadyInstalledException(name, version);
            }
        }
        String key = pkg.getAbbrev() + "-" + version;
        for (int i = 1; this.myStorage.packageKeyExists(key) && i < 100; ++i) {
            key = pkg.getAbbrev() + "-" + version + "__" + i;
        }
        if (this.myStorage.packageKeyExists(key)) {
            String msg = "Impossible to find a non-existing package key in the repo, stopped at: ";
            throw new PackageException(msg + key);
        }
        this.myStorage.storeInstallDir(tmp_dir, key, pkg);
        if (pp == null) {
            pp = new Packages(name);
            this.myPackages.put(name, pp);
        }
        pp.add(pkg);
        this.myStorage.updatePackageLists(pkg);
        for (Extension ext : this.myExtensions.values()) {
            ext.install(this, pkg);
        }
        return pkg;
    }

    public boolean removePackage(String pkg, boolean force, UserInteractionStrategy interact) throws PackageException {
        if (!interact.ask("Remove package " + pkg + "?", true)) {
            return false;
        }
        Packages pp = this.myPackages.get(pkg);
        if (pp == null) {
            if (force) {
                return false;
            }
            throw new PackageException("The package does not exist: " + pkg);
        }
        if (pp.packages().size() != 1) {
            throw new PackageException("The package has several versions installed: " + pkg);
        }
        Package p = pp.latest();
        this.myStorage.remove(p);
        pp.remove(p);
        this.myPackages.remove(pkg);
        return true;
    }

    public boolean removePackage(String pkg, String version, boolean force, UserInteractionStrategy interact) throws PackageException {
        if (!interact.ask("Remove package " + pkg + ", version " + version + "?", true)) {
            return false;
        }
        Packages pp = this.myPackages.get(pkg);
        if (pp == null) {
            if (force) {
                return false;
            }
            throw new PackageException("The package does not exist: " + pkg);
        }
        Package p = pp.version(version);
        if (p == null) {
            if (force) {
                return false;
            }
            throw new PackageException("The version " + version + " does not exist for the package: " + pkg);
        }
        this.myStorage.remove(p);
        pp.remove(p);
        if (pp.latest() == null) {
            this.myPackages.remove(pkg);
        }
        return true;
    }

    @Override
    public Source resolve(String href, URISpace space) throws PackageException {
        LOG.debug("Repository, resolve in {0}: ''{1}''", new Object[]{space, href});
        for (Packages pp : this.myPackages.values()) {
            Package p = pp.latest();
            Source src = p.resolve(href, space);
            if (src == null) continue;
            return src;
        }
        return null;
    }

    @Override
    public Source resolve(String href, URISpace space, boolean transitive) throws PackageException {
        return this.resolve(href, space);
    }

    private synchronized void parsePublicUris() throws PackageException {
        Set<String> packages = this.myStorage.listPackageDirectories();
        DescriptorParser parser = new DescriptorParser();
        for (String p : packages) {
            Source desc;
            Storage.PackageResolver res = this.myStorage.makePackageResolver(p, null);
            try {
                desc = res.resolveResource("expath-pkg.xml");
            }
            catch (Storage.NotExistException ex) {
                throw new PackageException("Package descriptor does NOT exist in: " + p, ex);
            }
            try {
                Package pkg = parser.parse(desc, p, this.myStorage, this);
                this.addPackage(pkg);
                for (Extension ext : this.myExtensions.values()) {
                    ext.init(this, pkg);
                }
            }
            catch (PackageException packageException) {
            }
        }
    }

    void addPackage(Package pkg) {
        String name = pkg.getName();
        Packages pp = this.myPackages.get(name);
        if (pp == null) {
            pp = new Packages(name);
            this.myPackages.put(name, pp);
        }
        pp.add(pkg);
    }

    Repository() {
        this.myStorage = null;
        this.myPackages = new HashMap<String, Packages>();
        this.myExtensions = new HashMap<String, Extension>();
    }

    public static class HttpException
    extends OnlineException {
        private final int myCode;
        private final String myStatus;

        public HttpException(URI url, int code, String status) {
            super(url, "HTTP error at URL: " + url + ", code: " + code + ", status: " + status);
            this.myCode = code;
            this.myStatus = status;
        }

        public int getCode() {
            return this.myCode;
        }

        public String getStatus() {
            return this.myStatus;
        }
    }

    public static class NotFoundException
    extends OnlineException {
        public NotFoundException(URI url) {
            super(url, "Package not found at URL: " + url);
        }
    }

    public static class OnlineException
    extends PackageException {
        private final URI myUrl;

        public OnlineException(URI url) {
            super("Error downloading the package at URL: " + url);
            this.myUrl = url;
        }

        public OnlineException(URI url, String msg) {
            super(msg);
            this.myUrl = url;
        }

        public OnlineException(URI url, Exception cause) {
            super("Error downloading the package at URL: " + url, cause);
            this.myUrl = url;
        }

        public URI getUrl() {
            return this.myUrl;
        }
    }

    public static class AlreadyInstalledException
    extends PackageException {
        private final String myName;
        private final String myVersion;

        public AlreadyInstalledException(String name, String version) {
            super("Same version of the package is already installed: " + name + ", " + version);
            this.myName = name;
            this.myVersion = version;
        }

        public String getName() {
            return this.myName;
        }

        public String getVersion() {
            return this.myVersion;
        }
    }
}

