// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.codec;

import static org.junit.jupiter.api.Assertions.*;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.Random;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.refcodes.textual.RandomTextGenerartor;
import org.refcodes.textual.RandomTextMode;

/**
 * The Class BaseOutputStreamEncoderTest.
 */
public class OutputStreamBaseEncoderTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TESTS = Boolean.getBoolean( "log.tests" );
	private static final int WEAK_INTENSITY_LOOPS = 4096;
	private static final int MAX_DATA_LENGTH = 4096;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test all random text.
	 *
	 * @throws IOException the open exception
	 * @throws IOException the close exception
	 */
	@Test
	public void testAllRandomText() throws IOException {
		for ( BaseMetrics eCodec : BaseMetricsConfig.values() ) {
			runRandomTextTest( eCodec );
		}
	}

	/**
	 * Test all random bytes.
	 *
	 * @throws IOException the open exception
	 * @throws IOException the close exception
	 */
	@Test
	public void testAllRandomBytes() throws IOException {
		for ( BaseMetrics eCodec : BaseMetricsConfig.values() ) {
			runRandomBytesTest( eCodec );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// DEBUG:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Debug base encode output stream 1.
	 *
	 * @throws IOException the open exception
	 * @throws IOException the close exception
	 */
	@Disabled("Just for debugging purposes")
	@Test
	public void debugBaseEncodeOutputStream1() throws IOException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		BaseMetricsConfig theCodecMetrics = BaseMetricsConfig.BASE64;
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( theCodecMetrics );
		byte[] theDecodedBytes = "Hello world!".getBytes();
		theBaseCodeBuilder.withDecodedData( theDecodedBytes );
		String theEncodedText = theBaseCodeBuilder.getEncodedText();
		BaseEncoder theEncodeSender = new BaseEncoder( theOutputStream ).withBaseMetrics( theCodecMetrics );
		theEncodeSender.transmitAllBytes( theDecodedBytes );
		theEncodeSender.close();
		String theResult = new String( theOutputStream.toByteArray() );
		if ( IS_LOG_TESTS ) System.out.println( "Expecting <" + theResult + "> to be <" + theEncodedText + "> ..." );
		assertEquals( theEncodedText, theResult );
	}

	/**
	 * Debug base encode output stream 2.
	 *
	 * @throws IOException the open exception
	 * @throws IOException the close exception
	 */
	@Disabled("Just for debugging purposes")
	@Test
	public void debugBaseEncodeOutputStream2() throws IOException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		BaseMetricsConfig theCodecMetrics = BaseMetricsConfig.BASE64;
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( theCodecMetrics );
		byte[] theDecodedBytes = "Hello world!!".getBytes();
		theBaseCodeBuilder.withDecodedData( theDecodedBytes );
		String theEncodedText = theBaseCodeBuilder.getEncodedText();
		BaseEncoder theEncodeSender = new BaseEncoder( theOutputStream ).withBaseMetrics( theCodecMetrics );
		theEncodeSender.transmitAllBytes( theDecodedBytes );
		theEncodeSender.close();
		String theResult = new String( theOutputStream.toByteArray() );
		if ( IS_LOG_TESTS ) System.out.println( "Expecting <" + theResult + "> to be <" + theEncodedText + "> ..." );
		assertEquals( theEncodedText, theResult );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private String toRandomText( int aLength ) {
		return RandomTextGenerartor.asString( aLength, RandomTextMode.ASCII );
	}

	/**
	 * Run random text test.
	 *
	 * @param aBaseMetrics the base metrics
	 * 
	 * @throws IOException the open exception
	 * @throws IOException the close exception
	 */
	private void runRandomTextTest( BaseMetrics aBaseMetrics ) throws IOException {
		if ( IS_LOG_TESTS ) System.out.println( "Running random text tests for base <" + aBaseMetrics.getNumberBase() + "> ..." );
		ByteArrayOutputStream eOutputStream;
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( aBaseMetrics );
		String eReferenceEncodedText, eEncodedText;
		byte[] eInputBytes, eDecodedBytes;
		String eRndText;
		BaseEncoder eBaseEncodeOutputStream;
		for ( int i = 0; i < WEAK_INTENSITY_LOOPS; i++ ) {
			eOutputStream = new ByteArrayOutputStream();
			eBaseEncodeOutputStream = new BaseEncoder( eOutputStream ).withBaseMetrics( aBaseMetrics );
			eRndText = toRandomText( i % MAX_DATA_LENGTH );
			eInputBytes = eRndText.getBytes();
			eEncodedText = theBaseCodeBuilder.withDecodedData( eInputBytes ).getEncodedText();

			eBaseEncodeOutputStream.transmitAllBytes( eInputBytes );
			eBaseEncodeOutputStream.close();
			String theResult = new String( eOutputStream.toByteArray() );
			assertEquals( eEncodedText, theResult );

			// if (IS_LOG_TESTS) System.out.println( theRndText + ": Reference := " + eReferenceEncoding
			// + " --> REFCODES := " + eEncodedText );
			eDecodedBytes = theBaseCodeBuilder.withEncodedText( eEncodedText ).getDecodedData();
			assertArrayEquals( eInputBytes, eDecodedBytes );
			if ( aBaseMetrics == BaseMetricsConfig.BASE64 ) {
				eReferenceEncodedText = Base64.getEncoder().encodeToString( eInputBytes );
				assertEquals( eReferenceEncodedText, eEncodedText );
			}
		}
	}

	/**
	 * Run random bytes test.
	 *
	 * @param aBaseMetrics the base metrics
	 * 
	 * @throws IOException the open exception
	 * @throws IOException the close exception
	 */
	private void runRandomBytesTest( BaseMetrics aBaseMetrics ) throws IOException {
		if ( IS_LOG_TESTS ) System.out.println( "Running random bytes tests for base <" + aBaseMetrics.getNumberBase() + "> ..." );
		ByteArrayOutputStream eOutputStream;
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( aBaseMetrics );
		String eReferenceEncodedText, eEncodedText;
		int eMod;
		byte[] eInputBytes, eDecodedBytes;
		BaseEncoder eBaseEncodeOutputStream;
		for ( int i = 0; i < WEAK_INTENSITY_LOOPS; i++ ) {
			eOutputStream = new ByteArrayOutputStream();
			eBaseEncodeOutputStream = new BaseEncoder( eOutputStream ).withBaseMetrics( aBaseMetrics );
			eMod = i % MAX_DATA_LENGTH;
			eInputBytes = new byte[eMod];
			new Random().nextBytes( eInputBytes );
			eEncodedText = theBaseCodeBuilder.withDecodedData( eInputBytes ).getEncodedText();

			eBaseEncodeOutputStream.transmitAllBytes( eInputBytes );
			eBaseEncodeOutputStream.close();
			String theResult = new String( eOutputStream.toByteArray() );
			assertEquals( eEncodedText, theResult );

			// if (IS_LOG_TESTS) System.out.println( new VerboseTextBuilderImpl().withElements( eBytes )
			// + ": Reference := " + eReferenceEncoding + " --> REFCODES := " +
			// eEncodedText );
			eDecodedBytes = theBaseCodeBuilder.withEncodedText( eEncodedText ).getDecodedData();
			assertArrayEquals( eInputBytes, eDecodedBytes );
			if ( aBaseMetrics == BaseMetricsConfig.BASE64 ) {
				eReferenceEncodedText = Base64.getEncoder().encodeToString( eInputBytes );
				assertEquals( eReferenceEncodedText, eEncodedText );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
