// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.codec;

import static org.junit.jupiter.api.Assertions.*;

import java.util.Base64;
import java.util.Random;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.refcodes.data.CharSet;
import org.refcodes.textual.RandomTextGenerartor;
import org.refcodes.textual.RandomTextMode;

/**
 * The Class BaseBuilderTest.
 */
public class BaseBuilderTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int WEAK_INTENSITY_LOOPS = 10240;
	private static final int STRONG_INTENSITY_LOOPS = 150000;
	private static final int MAX_DATA_LENGTH = 8192;
	private static char[] CHAR_SET = CharSet.ALPHANUMERIC.getCharSet();
	private static Random RND = new Random();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	//	@Test
	//	public void testDiv() {
	//		for ( int i = 0; i < 1024; i++ ) {
	//			//			float a = i / 1024F;
	//			//			int b = (int) (a * 100F);
	//			//			float c = b / 100F;
	//			//			System.out.println( i + "= " + c );
	//			float result = ((int) (i / 1024F * 100F)) / 100F;
	//			System.out.println( i + "= " + result );
	//		}
	//
	//	}

	/**
	 * Test all random text.
	 */
	@Test
	public void testAllRandomText() {
		boolean isTestSideEffectFreeCall = true;
		for ( BaseMetrics eCodec : BaseMetricsConfig.values() ) {
			runRandomTextTest( eCodec, isTestSideEffectFreeCall );
			isTestSideEffectFreeCall = !isTestSideEffectFreeCall;
		}
	}

	/**
	 * Test all random bytes.
	 */
	@Test
	public void testAllRandomBytes() {
		boolean isTestSideEffectFreeCall = true;
		for ( BaseMetrics eCodec : BaseMetricsConfig.values() ) {
			runRandomBytesTest( eCodec, isTestSideEffectFreeCall );
			isTestSideEffectFreeCall = !isTestSideEffectFreeCall;
		}
	}

	@Test
	public void testBlockSize() {
		// BaseMetrics eCodec = BaseMetricsConfig.BASE64;
		for ( BaseMetrics eCodec : BaseMetricsConfig.values() ) {
			if ( IS_LOG_TEST_ENABLED ) System.out.println( eCodec.toString() );
			BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( eCodec );
			String eEncoded;
			int ePaddingNumber;
			for ( int i = 1; i < 64; i++ ) {
				eEncoded = theBaseCodeBuilder.toEncodedText( RandomTextGenerartor.asString( i, RandomTextMode.ALPHANUMERIC ) );
				int eCount = count( eEncoded, eCodec.getPaddingChar() );
				ePaddingNumber = eCodec.toPaddingCharsNumber( i );
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "[" + (eCount != ePaddingNumber ? "X" : "✓") + "] " + ePaddingNumber + " ? " + eCount + ": " + eEncoded + " (" + i + ")" );
				}
				assertEquals( eCount, ePaddingNumber );

			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// DEBUG:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Debuge edge case.
	 */
	@Disabled("Just for debugging purposes")
	@Test
	public void debugeEdgeCase() {
		BaseMetrics theMetrics = BaseMetricsConfig.ENCODED_AS_NUMBER;
		if ( IS_LOG_TEST_ENABLED ) System.out.println( theMetrics );
		BaseBuilder theBuilder = new BaseBuilder().withBaseMetrics( theMetrics );
		String theEncoded = theBuilder.toEncodedText( "Hallo welt! 1234567890 ABC".getBytes() );
		byte[] theDecoded = theBuilder.toDecodedData( theEncoded );
		if ( IS_LOG_TEST_ENABLED ) System.out.println( theEncoded );
		if ( IS_LOG_TEST_ENABLED ) System.out.println( new String( theDecoded ) );
	}

	/**
	 * Debug encoded as number codec.
	 */
	@Test
	public void debugEncodedAsNumberCodec() {
		runRandomTextTest( BaseMetricsConfig.ENCODED_AS_NUMBER, false );
	}

	/**
	 * Debug base 64 codec.
	 */
	@Test
	public void debugBase64Codec() {
		BaseMetrics theBase64 = new BaseMetricsImpl( 64, CharSet.BASE64.getCharSet() );
		runRandomTextTest( theBase64, true );
	}

	/**
	 * Debug get bytes from text.
	 */
	@Disabled("Just for debugging purposes")
	@Test
	public void debugGetBytesFromText() {
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( BaseMetricsConfig.BASE64 );
		theBaseCodeBuilder.withDecodedData( "Hello world!".getBytes() );
		if ( IS_LOG_TEST_ENABLED ) System.out.println( theBaseCodeBuilder.getEncodedText() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// THROUGHPUT:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test encode refcodes.
	 */
	@Disabled("Just for throughput measurement")
	@Test
	public void testEncodeRefcodes() {
		byte[] eBytes;
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( BaseMetricsConfig.BASE64 );
		for ( int i = 0; i < STRONG_INTENSITY_LOOPS; i++ ) {
			String theRndText = toRandom( i % MAX_DATA_LENGTH );
			eBytes = theRndText.getBytes();
			theBaseCodeBuilder.withDecodedData( eBytes ).getEncodedText();
		}
	}

	/**
	 * Test encode java.
	 */
	@Disabled("Just for throughput measurement")
	@Test
	public void testEncodeJava() {
		byte[] eBytes;
		for ( int i = 0; i < STRONG_INTENSITY_LOOPS; i++ ) {
			String theRndText = toRandom( i % MAX_DATA_LENGTH );
			eBytes = theRndText.getBytes();
			Base64.getEncoder().encodeToString( eBytes );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Run random text test.
	 *
	 * @param aBaseMetrics the base metrics
	 * @param isWithSideEffectFreeShortcut the is with side effect free shortcut
	 */
	private void runRandomTextTest( BaseMetrics aBaseMetrics, boolean isWithSideEffectFreeShortcut ) {
		if ( IS_LOG_TEST_ENABLED ) System.out.println( "Running random text tests for base <" + (aBaseMetrics.getNumberBase() < 10 ? "0" : "") + aBaseMetrics.getNumberBase() + "> (" + (isWithSideEffectFreeShortcut ? "*" : "?") + ") ..." );
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( aBaseMetrics );
		String eReferenceEncodedText, eEncodedText;
		byte[] eInputBytes, eDecodedBytes;
		String eRndText;
		for ( int i = 0; i < WEAK_INTENSITY_LOOPS; i++ ) {
			eRndText = toRandom( i % MAX_DATA_LENGTH );
			eInputBytes = eRndText.getBytes();
			if ( !isWithSideEffectFreeShortcut ) {
				eEncodedText = theBaseCodeBuilder.withDecodedData( eInputBytes ).getEncodedText();
			}
			else {
				eEncodedText = theBaseCodeBuilder.toEncodedText( eInputBytes );
			}
			// if (IS_LOG_TEST_ENABLED) System.out.println( theRndText + ": Reference := " + eReferenceEncoding
			// + " --> REFCODES := " + eEncodedText );
			eDecodedBytes = theBaseCodeBuilder.withEncodedText( eEncodedText ).getDecodedData();
			assertArrayEquals( eInputBytes, eDecodedBytes );
			if ( aBaseMetrics == BaseMetricsConfig.BASE64 ) {
				eReferenceEncodedText = Base64.getEncoder().encodeToString( eInputBytes );
				assertEquals( eReferenceEncodedText, eEncodedText );
			}
		}
	}

	/**
	 * Run random bytes test.
	 *
	 * @param aBaseMetrics the base metrics
	 * @param isWithSideEffectFreeShortcut the is with side effect free shortcut
	 */
	private void runRandomBytesTest( BaseMetrics aBaseMetrics, boolean isWithSideEffectFreeShortcut ) {
		if ( IS_LOG_TEST_ENABLED ) System.out.println( "Running random bytes tests for base <" + (aBaseMetrics.getNumberBase() < 10 ? "0" : "") + aBaseMetrics.getNumberBase() + "> (" + (isWithSideEffectFreeShortcut ? "*" : "?") + ") ..." );
		BaseBuilder theBaseCodeBuilder = new BaseBuilder().withBaseMetrics( aBaseMetrics );
		String eReferenceEncodedText, eEncodedText;
		int eMod;
		byte[] eInputBytes, eDecodedBytes;
		for ( int i = 0; i < WEAK_INTENSITY_LOOPS; i++ ) {
			eMod = i % MAX_DATA_LENGTH;
			eInputBytes = new byte[eMod];
			new Random().nextBytes( eInputBytes );
			if ( !isWithSideEffectFreeShortcut ) {
				eEncodedText = theBaseCodeBuilder.withDecodedData( eInputBytes ).getEncodedText();
			}
			else {
				eEncodedText = theBaseCodeBuilder.toEncodedText( eInputBytes );
			}
			// if (IS_LOG_TEST_ENABLED) System.out.println( new VerboseTextBuilderImpl().withElements( eBytes )
			// + ": Reference := " + eReferenceEncoding + " --> REFCODES := " +
			// eEncodedText );
			eDecodedBytes = theBaseCodeBuilder.withEncodedText( eEncodedText ).getDecodedData();
			assertArrayEquals( eInputBytes, eDecodedBytes );
			if ( aBaseMetrics == BaseMetricsConfig.BASE64 ) {
				eReferenceEncodedText = Base64.getEncoder().encodeToString( eInputBytes );
				assertEquals( eReferenceEncodedText, eEncodedText );
			}
		}
	}

	/**
	 * To random.
	 *
	 * @param aColumnWidth the column width
	 * 
	 * @return the string
	 */
	private String toRandom( int aColumnWidth ) {
		int theBound = CHAR_SET.length;
		char[] theRandom = new char[aColumnWidth];
		for ( int i = 0; i < aColumnWidth; i++ ) {
			theRandom[i] = CHAR_SET[RND.nextInt( theBound )];
		}
		return new String( theRandom );
	}

	private int count( String aString, char aChar ) {
		int theCount, eIndexOf;
		theCount = 0;
		eIndexOf = aString.indexOf( aChar );
		while ( eIndexOf != -1 ) {
			aString = aString.substring( eIndexOf + 1 );
			theCount++;
			eIndexOf = aString.indexOf( aChar );
		}
		return theCount;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
