package org.sqlproc.engine.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A META SQL sub-element. It represents a dynamic input value.
 * 
 * <p>
 * Schematically:
 * 
 * <pre>
 * SqlMetaIdent
 *     +-SqlMetaIdentItem.SqlMetaIdentItem...^SqlMyType^value
 * </pre>
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
class SqlMetaIdent implements SqlMetaSimple, SqlMetaLogOperand {

    /**
     * The internal slf4j logger.
     */
    final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * Which conversion should be done on input value.
     */
    private SqlInputValue.Case caseConversion;
    /**
     * An indicator, which is used to control, how the input value is added to the final ANSI SQL. A standard behavior
     * is to add an input value only in the case it's not empty. The definition of the emptiness depends on the type of
     * the input value.
     */
    private boolean not;
    /**
     * The list of sub-elements. Every sub-element represents the name of an attribute in the input class (the static
     * parameters class). In case there're more names, the input classes are embedded one in other.
     */
    private List<SqlMetaIdentItem> elements;
    /**
     * The type of this input value. It can be Hibernate or an internal type.
     */
    private SqlType sqlType;

    /**
     * Values for a special identifier handling, for example a sequence for an identity.
     */
    Map<String, String> values = new HashMap<String, String>();

    /**
     * Creates a new instance of this entity. Used from inside ANTLR parser.
     * 
     * @param caseConversion
     *            which conversion should be done on inputValue
     */
    SqlMetaIdent(SqlInputValue.Case caseConversion) {
        this(caseConversion, false);
    }

    /**
     * Creates a new instance of this entity. Used from inside ANTLR parser.
     * 
     * @param caseConversion
     *            which conversion should be done on inputValue
     * @param not
     *            an indicator, which is used to control, how the input value is added to the final ANSI SQL
     */
    SqlMetaIdent(SqlInputValue.Case caseConversion, boolean not) {
        this(caseConversion, false, new SqlType());
    }

    /**
     * Creates a new instance of this entity. Used from inside ANTLR parser.
     * 
     * @param caseConversion
     *            which conversion should be done on inputValue
     * @param not
     *            an indicator, which is used to control, how the input value is added to the final ANSI SQL
     * @param type
     *            the type of this input value, which can be Hibernate or an internal type
     */
    SqlMetaIdent(SqlInputValue.Case caseConversion, boolean not, SqlType type) {
        this.elements = new ArrayList<SqlMetaIdentItem>();
        this.caseConversion = caseConversion;
        this.not = not;
        this.sqlType = type;
    }

    /**
     * Adds a new name. This is the name of an attribute in the input class (the static parameters class). In case
     * there're more names, the input classes are embedded one in other.
     * 
     * @param name
     *            the next name in the list of names
     */
    void addIdent(String name) {
        int size = elements.size();
        SqlMetaIdentItem lastItem = (SqlMetaIdentItem) (size > 0 ? elements.get(size - 1) : null);
        if (lastItem != null)
            lastItem.setType(SqlMetaIdentItem.Type.REF);
        elements.add(new SqlMetaIdentItem(name, SqlMetaIdentItem.Type.VAL));
    }

    /**
     * Adds a new name. This is the name of an attribute in the input class (the static parameters class). In case
     * there're more names, the input classes are embedded one in other.
     * 
     * @param element
     *            an object representation of the next name in the list of names
     */
    void addIdent(SqlMetaIdentItem element) {
        elements.add(element);
    }

    /**
     * Sets the internal type of this input value.
     * 
     * @param sMetaType
     *            a String representation of the internal type
     */
    void setMetaType(String sMetaType) {
        if (sMetaType.startsWith("h_"))
            this.sqlType = new SqlType(sMetaType.substring(2));
        else
            this.sqlType = new SqlType(sMetaType, this.sqlType);
    }

    /**
     * Sets Hibernate type of this input value.
     * 
     * @param sHibernateType
     *            a String representation of Hibernate type
     */
    void setHibernateType(String sHibernateType) {
        this.sqlType = new SqlType(sHibernateType);
    }

    /**
     * Sets the values for a special identifier handling, for example a sequence for an identity. Or it can used be for
     * the special handling of the enumeration type of the input value. The logical evaluation of the input value is
     * based on the comparison to this value.
     * 
     * @param value
     *            the value for a special treatment, might be an identifier of value2 or a value for a comparison
     * @param value2
     *            the value for a special treatment, might be a sequence name for an identity column
     */
    public void setValues(String value, String value2) {
        if (value2 == null)
            sqlType.setValue(value);
        else
            values.put(value.toUpperCase(), value2);
    }

    /**
     * Returns the type of this input value. It can be Hibernate or an internal type.
     * 
     * @return the type of this input value
     */
    SqlType getSqlType() {
        return this.sqlType;
    }

    /**
     * Sets the indicator, which is used to control, how the input value is added to the final ANSI SQL. A standard
     * behavior is to add an input value only in the case it's not empty. The definition of the emptiness depends on the
     * type of the input value.
     * 
     * @param not
     *            a new indicator value
     */
    void setNot(boolean not) {
        this.not = not;
    }

    /**
     * Returns the indicator, which is used to control, how the input value is added to the final ANSI SQL. A standard
     * behavior is to add an input value only in the case it's not empty. The definition of the emptiness depends on the
     * type of the input value.
     * 
     * @return the indicator value
     */
    boolean isNot() {
        return not;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public SqlProcessResult process(SqlProcessContext ctx) {

        SqlProcessResult result = new SqlProcessResult();
        boolean first = true;
        Object obj = ctx.dynamicInputValues;
        StringBuilder s = new StringBuilder(elements.size() * 32);
        s.append(IDENT_PREFIX);
        if (logger.isDebugEnabled())
            logger.debug("SqlMetaIdent process " + ((obj != null) ? obj.getClass() : null) + " " + obj + " "
                    + this.sqlType);

        int size = this.elements.size();
        int count = 1;
        String sequenceName = values.get(SEQUENCE);
        String attributeName = null;
        Class<?> attributeType = (obj != null) ? obj.getClass() : null;

        for (SqlMetaIdentItem item : this.elements) {
            attributeName = item.getName();
            if (attributeType != null) {
                attributeType = SqlUtils.getFieldType(attributeType, attributeName);
            }
            if (count > 1)
                s.append(IDENT_SEPARATOR);
            s.append(attributeName);
            if (sequenceName != null && count == size)
                break;
            if (obj != null) {
                obj = SqlUtils.invokeMethod(obj, item.get(), item.is());
            }
            count++;
        }

        if (sequenceName != null) {
            String sequence = SqlProcessContext.getFeature(sequenceName);
            if (sequence == null) {
                throw new RuntimeException("Missing sequence " + sequenceName);
            }
            result.add(true);
            SqlInputValue identityInputValue = new SqlInputValue(obj, attributeType, sequence, this.sqlType);
            result.addInputValue(s.substring(lIDENT_PREFIX), identityInputValue);
            result.addIdentity(attributeName, identityInputValue);
        } else {
            try {
                result.add(SqlUtils.isEmpty(obj, sqlType, ctx.inSqlSetOrInsert));
            } catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("Input value " + attributeName + ", failed reason" + e.getMessage());
            }
            result.addInputValue(s.substring(lIDENT_PREFIX), new SqlInputValue(obj, attributeType, caseConversion,
                    sqlType));
        }

        result.setSql(s);
        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean processExpression(SqlProcessContext ctx) {
        Object obj = ctx.dynamicInputValues;

        if (logger.isDebugEnabled())
            logger.debug("SqlMetaIdent process expression " + obj.getClass() + " " + obj + " " + this.sqlType);

        for (SqlMetaIdentItem item : this.elements) {
            if (obj != null) {
                obj = SqlUtils.invokeMethod(obj, item.get(), item.is());
            }
        }

        boolean result = SqlUtils.isTrue(obj, sqlType);
        return (this.not ? !result : result);
    }
}
