package com.checkhim;

import com.checkhim.exceptions.CheckHimException;
import com.checkhim.http.HttpClient;
import com.checkhim.models.VerificationRequest;
import com.checkhim.models.VerificationResponse;

/**
 * Main client class for interacting with the CheckHim phone number verification API.
 * 
 * This class provides a simple interface for verifying phone numbers using the
 * CheckHim service. It handles authentication, request formatting, and error handling.
 * 
 * Example usage:
 * <pre>
 * CheckHim client = new CheckHim("ch_test_your_api_key");
 * VerificationResponse result = client.verify(
 *     VerificationRequest.builder()
 *         .number("+5511984339000")
 *         .build()
 * );
 * System.out.println("Valid: " + result.isValid());
 * </pre>
 */
public class CheckHim {

    private static final String DEFAULT_BASE_URL = "https://api.checkhim.tech";
    private static final String API_VERSION = "v1";
    
    private final HttpClient httpClient;
    private final String apiKey;
    private final String baseUrl;

    /**
     * Creates a new CheckHim client with the specified API key.
     *
     * @param apiKey Your CheckHim API key (e.g., "ch_test_your_api_key")
     * @throws IllegalArgumentException if apiKey is null or empty
     */
    public CheckHim(String apiKey) {
        this(apiKey, DEFAULT_BASE_URL);
    }

    /**
     * Creates a new CheckHim client with the specified API key and custom base URL.
     *
     * @param apiKey Your CheckHim API key
     * @param baseUrl Custom base URL for the API (useful for testing or enterprise environments)
     * @throws IllegalArgumentException if apiKey is null or empty
     */
    public CheckHim(String apiKey, String baseUrl) {
        if (apiKey == null || apiKey.trim().isEmpty()) {
            throw new IllegalArgumentException("API key cannot be null or empty");
        }
        if (baseUrl == null || baseUrl.trim().isEmpty()) {
            throw new IllegalArgumentException("Base URL cannot be null or empty");
        }

        this.apiKey = apiKey.trim();
        this.baseUrl = baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl.length() - 1) : baseUrl;
        this.httpClient = new HttpClient(this.apiKey, this.baseUrl);
    }

    /**
     * Verifies a phone number using the CheckHim API.
     *
     * @param request The verification request containing the phone number to verify
     * @return VerificationResponse containing validation results and carrier information
     * @throws CheckHimException if the API request fails or returns an error
     * @throws IllegalArgumentException if request is null
     */
    public VerificationResponse verify(VerificationRequest request) throws CheckHimException {
        if (request == null) {
            throw new IllegalArgumentException("VerificationRequest cannot be null");
        }

        String endpoint = String.format("/api/%s/verify", API_VERSION);
        return httpClient.post(endpoint, request, VerificationResponse.class);
    }

    /**
     * Convenience method to verify a phone number using just the number string.
     *
     * @param phoneNumber The phone number to verify in international format (e.g., "+5511984339000")
     * @return VerificationResponse containing validation results and carrier information
     * @throws CheckHimException if the API request fails or returns an error
     * @throws IllegalArgumentException if phoneNumber is null or empty
     */
    public VerificationResponse verify(String phoneNumber) throws CheckHimException {
        if (phoneNumber == null || phoneNumber.trim().isEmpty()) {
            throw new IllegalArgumentException("Phone number cannot be null or empty");
        }

        VerificationRequest request = VerificationRequest.builder()
                .number(phoneNumber.trim())
                .build();

        return verify(request);
    }

    /**
     * Gets the API key being used by this client.
     *
     * @return The API key (masked for security)
     */
    public String getApiKey() {
        if (apiKey.length() <= 8) {
            return "*".repeat(apiKey.length());
        }
        return apiKey.substring(0, 4) + "*".repeat(apiKey.length() - 8) + apiKey.substring(apiKey.length() - 4);
    }

    /**
     * Gets the base URL being used by this client.
     *
     * @return The base URL
     */
    public String getBaseUrl() {
        return baseUrl;
    }

    /**
     * Closes the HTTP client and releases any resources.
     * Call this method when you're done using the client to ensure proper cleanup.
     */
    public void close() {
        if (httpClient != null) {
            httpClient.close();
        }
    }
}