package com.checkhim.exceptions;

/**
 * Base exception class for all CheckHim SDK related errors.
 * 
 * This exception is thrown when any error occurs during API communication
 * or processing of verification requests.
 */
public class CheckHimException extends Exception {

    private final String errorCode;
    private final int httpStatusCode;

    /**
     * Creates a new CheckHimException with the specified message.
     *
     * @param message The error message
     */
    public CheckHimException(String message) {
        super(message);
        this.errorCode = null;
        this.httpStatusCode = 0;
    }

    /**
     * Creates a new CheckHimException with the specified message and cause.
     *
     * @param message The error message
     * @param cause The underlying cause
     */
    public CheckHimException(String message, Throwable cause) {
        super(message, cause);
        this.errorCode = null;
        this.httpStatusCode = 0;
    }

    /**
     * Creates a new CheckHimException with detailed error information.
     *
     * @param message The error message
     * @param errorCode The CheckHim error code
     * @param httpStatusCode The HTTP status code from the API response
     */
    public CheckHimException(String message, String errorCode, int httpStatusCode) {
        super(message);
        this.errorCode = errorCode;
        this.httpStatusCode = httpStatusCode;
    }

    /**
     * Creates a new CheckHimException with detailed error information and cause.
     *
     * @param message The error message
     * @param errorCode The CheckHim error code
     * @param httpStatusCode The HTTP status code from the API response
     * @param cause The underlying cause
     */
    public CheckHimException(String message, String errorCode, int httpStatusCode, Throwable cause) {
        super(message, cause);
        this.errorCode = errorCode;
        this.httpStatusCode = httpStatusCode;
    }

    /**
     * Gets the CheckHim-specific error code.
     *
     * @return The error code (e.g., "REJECTED_NETWORK"), or null if not available
     */
    public String getErrorCode() {
        return errorCode;
    }

    /**
     * Gets the HTTP status code from the API response.
     *
     * @return The HTTP status code, or 0 if not available
     */
    public int getHttpStatusCode() {
        return httpStatusCode;
    }

    /**
     * Checks if this exception represents a client error (4xx status code).
     *
     * @return true if the HTTP status code is in the 4xx range
     */
    public boolean isClientError() {
        return httpStatusCode >= 400 && httpStatusCode < 500;
    }

    /**
     * Checks if this exception represents a server error (5xx status code).
     *
     * @return true if the HTTP status code is in the 5xx range
     */
    public boolean isServerError() {
        return httpStatusCode >= 500 && httpStatusCode < 600;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName()).append(": ").append(getMessage());
        
        if (errorCode != null) {
            sb.append(" [").append(errorCode).append("]");
        }
        
        if (httpStatusCode > 0) {
            sb.append(" (HTTP ").append(httpStatusCode).append(")");
        }
        
        return sb.toString();
    }
}