package com.checkhim.http;

import com.checkhim.exceptions.*;
import com.checkhim.models.ErrorResponse;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import okhttp3.*;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

/**
 * HTTP client wrapper for making requests to the CheckHim API.
 * 
 * This class handles authentication, request formatting, response parsing,
 * and error handling for all API communications.
 */
public class HttpClient {

    private static final String USER_AGENT = "CheckHim-Java-SDK/1.0.0";
    private static final MediaType JSON_MEDIA_TYPE = MediaType.get("application/json; charset=utf-8");
    
    private final OkHttpClient client;
    private final ObjectMapper objectMapper;
    private final String apiKey;
    private final String baseUrl;

    /**
     * Creates a new HTTP client with the specified API key and base URL.
     *
     * @param apiKey The CheckHim API key
     * @param baseUrl The base URL for the API
     */
    public HttpClient(String apiKey, String baseUrl) {
        this.apiKey = apiKey;
        this.baseUrl = baseUrl;
        
        this.client = new OkHttpClient.Builder()
                .connectTimeout(30, TimeUnit.SECONDS)
                .readTimeout(60, TimeUnit.SECONDS)
                .writeTimeout(60, TimeUnit.SECONDS)
                .addInterceptor(this::addAuthenticationHeaders)
                .build();

        this.objectMapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * Makes a POST request to the specified endpoint.
     *
     * @param endpoint The API endpoint (e.g., "/api/v1/verify")
     * @param requestBody The request body object
     * @param responseType The expected response type class
     * @param <T> The type of the request body
     * @param <R> The type of the response
     * @return The parsed response object
     * @throws CheckHimException if the request fails or returns an error
     */
    public <T, R> R post(String endpoint, T requestBody, Class<R> responseType) throws CheckHimException {
        try {
            String jsonBody = objectMapper.writeValueAsString(requestBody);
            RequestBody body = RequestBody.create(jsonBody, JSON_MEDIA_TYPE);
            
            Request request = new Request.Builder()
                    .url(baseUrl + endpoint)
                    .post(body)
                    .build();

            try (Response response = client.newCall(request).execute()) {
                return handleResponse(response, responseType);
            }

        } catch (JsonProcessingException e) {
            throw new CheckHimException("Failed to serialize request body", e);
        } catch (IOException e) {
            throw new CheckHimException("Network error occurred while making request", e);
        }
    }

    /**
     * Handles the HTTP response and parses it appropriately.
     *
     * @param response The HTTP response
     * @param responseType The expected response type class
     * @param <R> The type of the response
     * @return The parsed response object
     * @throws CheckHimException if the response indicates an error
     */
    private <R> R handleResponse(Response response, Class<R> responseType) throws CheckHimException {
        int statusCode = response.code();
        
        try {
            ResponseBody responseBody = response.body();
            if (responseBody == null) {
                throw new CheckHimException("Empty response body", null, statusCode);
            }

            String responseJson = responseBody.string();

            if (response.isSuccessful()) {
                return objectMapper.readValue(responseJson, responseType);
            } else {
                handleErrorResponse(responseJson, statusCode);
                // This line should never be reached, but added for completeness
                throw new CheckHimException("Unexpected error occurred", null, statusCode);
            }

        } catch (JsonProcessingException e) {
            throw new CheckHimException("Failed to parse response JSON", e);
        } catch (IOException e) {
            throw new CheckHimException("Failed to read response body", e);
        }
    }

    /**
     * Parses error responses and throws appropriate exceptions.
     *
     * @param responseJson The JSON response body
     * @param statusCode The HTTP status code
     * @throws CheckHimException The appropriate exception based on the error code
     */
    private void handleErrorResponse(String responseJson, int statusCode) throws CheckHimException {
        try {
            ErrorResponse errorResponse = objectMapper.readValue(responseJson, ErrorResponse.class);
            String errorCode = errorResponse.getCode();
            String errorMessage = errorResponse.getDisplayMessage();

            // Map specific error codes to specialized exceptions
            switch (errorCode) {
                case "REJECTED_NETWORK":
                    throw new NetworkRejectedException(errorMessage, statusCode);
                case "REJECTED_FORMAT":
                    throw new InvalidFormatException(errorMessage, statusCode);
                case "REJECTED_PREFIX_MISSING":
                    throw new PrefixMissingException(errorMessage, statusCode);
                case "REJECTED_SUBSCRIBER_ABSENT":
                    throw new SubscriberAbsentException(errorMessage, statusCode);
                case "REJECTED_UNKNOWN_SUBSCRIBER":
                    throw new UnknownSubscriberException(errorMessage, statusCode);
                case "REJECTED_UNDELIVERABLE":
                case "UNDELIVERABLE_NOT_DELIVERED":
                case "TEMPORARY_FAILURE":
                case "SERVICE_UNAVAILABLE":
                default:
                    throw new CheckHimException(errorMessage, errorCode, statusCode);
            }

        } catch (JsonProcessingException e) {
            // If we can't parse the error response, throw a generic exception
            throw new CheckHimException("API request failed with status " + statusCode, null, statusCode);
        }
    }

    /**
     * Interceptor to add authentication headers to all requests.
     *
     * @param chain The interceptor chain
     * @return The response
     * @throws IOException if the request fails
     */
    private Response addAuthenticationHeaders(Interceptor.Chain chain) throws IOException {
        Request originalRequest = chain.request();
        
        Request authenticatedRequest = originalRequest.newBuilder()
                .addHeader("Authorization", "Bearer " + apiKey)
                .addHeader("Content-Type", "application/json")
                .addHeader("User-Agent", USER_AGENT)
                .build();

        return chain.proceed(authenticatedRequest);
    }

    /**
     * Closes the HTTP client and releases resources.
     */
    public void close() {
        if (client != null) {
            client.dispatcher().executorService().shutdown();
            client.connectionPool().evictAll();
            
            // Close the cache if it exists
            Cache cache = client.cache();
            if (cache != null) {
                try {
                    cache.close();
                } catch (IOException e) {
                    // Ignore close errors
                }
            }
        }
    }
}