package com.checkhim.models;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Represents the response from a phone number verification request.
 * 
 * Contains information about the phone number's validity, carrier information,
 * and other relevant verification details.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class VerificationResponse {

    @JsonProperty("valid")
    private Boolean valid;

    @JsonProperty("carrier")
    private String carrier;

    @JsonProperty("country")
    private String country;

    @JsonProperty("countryCode")
    private String countryCode;

    @JsonProperty("countryName")
    private String countryName;

    @JsonProperty("lineType")
    private String lineType;

    @JsonProperty("networkCode")
    private String networkCode;

    @JsonProperty("status")
    private String status;

    /**
     * Default constructor for JSON deserialization.
     */
    public VerificationResponse() {
    }

    /**
     * Checks if the phone number is valid.
     *
     * @return true if the phone number is valid, false otherwise
     */
    public boolean isValid() {
        return valid != null && valid;
    }

    /**
     * Gets the validation status as a Boolean object.
     *
     * @return Boolean indicating validity, or null if not set
     */
    public Boolean getValid() {
        return valid;
    }

    /**
     * Sets the validation status.
     *
     * @param valid true if valid, false otherwise
     */
    public void setValid(Boolean valid) {
        this.valid = valid;
    }

    /**
     * Gets the carrier/network provider name.
     *
     * @return The carrier name (e.g., "UNITEL", "VIVO")
     */
    public String getCarrier() {
        return carrier;
    }

    /**
     * Sets the carrier/network provider name.
     *
     * @param carrier The carrier name
     */
    public void setCarrier(String carrier) {
        this.carrier = carrier;
    }

    /**
     * Gets the country code (ISO 3166-1 alpha-2).
     *
     * @return The two-letter country code (e.g., "BR", "US")
     */
    public String getCountry() {
        return country;
    }

    /**
     * Sets the country code.
     *
     * @param country The country code
     */
    public void setCountry(String country) {
        this.country = country;
    }

    /**
     * Gets the numeric country code.
     *
     * @return The country calling code (e.g., "55" for Brazil, "1" for US)
     */
    public String getCountryCode() {
        return countryCode;
    }

    /**
     * Sets the numeric country code.
     *
     * @param countryCode The country calling code
     */
    public void setCountryCode(String countryCode) {
        this.countryCode = countryCode;
    }

    /**
     * Gets the full country name.
     *
     * @return The country name (e.g., "Brazil", "United States")
     */
    public String getCountryName() {
        return countryName;
    }

    /**
     * Sets the full country name.
     *
     * @param countryName The country name
     */
    public void setCountryName(String countryName) {
        this.countryName = countryName;
    }

    /**
     * Gets the line type (mobile, landline, etc.).
     *
     * @return The line type
     */
    public String getLineType() {
        return lineType;
    }

    /**
     * Sets the line type.
     *
     * @param lineType The line type
     */
    public void setLineType(String lineType) {
        this.lineType = lineType;
    }

    /**
     * Gets the network code.
     *
     * @return The network code
     */
    public String getNetworkCode() {
        return networkCode;
    }

    /**
     * Sets the network code.
     *
     * @param networkCode The network code
     */
    public void setNetworkCode(String networkCode) {
        this.networkCode = networkCode;
    }

    /**
     * Gets the verification status code.
     *
     * @return The status code (e.g., "DELIVERED_TO_HANDSET", "REJECTED_NETWORK")
     */
    public String getStatus() {
        return status;
    }

    /**
     * Sets the verification status code.
     *
     * @param status The status code
     */
    public void setStatus(String status) {
        this.status = status;
    }

    @Override
    public String toString() {
        return "VerificationResponse{" +
                "valid=" + valid +
                ", carrier='" + carrier + '\'' +
                ", country='" + country + '\'' +
                ", countryCode='" + countryCode + '\'' +
                ", countryName='" + countryName + '\'' +
                ", lineType='" + lineType + '\'' +
                ", networkCode='" + networkCode + '\'' +
                ", status='" + status + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        VerificationResponse that = (VerificationResponse) o;

        if (valid != null ? !valid.equals(that.valid) : that.valid != null) return false;
        if (carrier != null ? !carrier.equals(that.carrier) : that.carrier != null) return false;
        if (country != null ? !country.equals(that.country) : that.country != null) return false;
        if (countryCode != null ? !countryCode.equals(that.countryCode) : that.countryCode != null) return false;
        if (countryName != null ? !countryName.equals(that.countryName) : that.countryName != null) return false;
        if (lineType != null ? !lineType.equals(that.lineType) : that.lineType != null) return false;
        if (networkCode != null ? !networkCode.equals(that.networkCode) : that.networkCode != null) return false;
        return status != null ? status.equals(that.status) : that.status == null;
    }

    @Override
    public int hashCode() {
        int result = valid != null ? valid.hashCode() : 0;
        result = 31 * result + (carrier != null ? carrier.hashCode() : 0);
        result = 31 * result + (country != null ? country.hashCode() : 0);
        result = 31 * result + (countryCode != null ? countryCode.hashCode() : 0);
        result = 31 * result + (countryName != null ? countryName.hashCode() : 0);
        result = 31 * result + (lineType != null ? lineType.hashCode() : 0);
        result = 31 * result + (networkCode != null ? networkCode.hashCode() : 0);
        result = 31 * result + (status != null ? status.hashCode() : 0);
        return result;
    }
}