package tech.checkhim.models;

import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Represents an error response from the CheckHim API.
 * 
 * This class is used internally to parse error responses and convert
 * them to appropriate exception types.
 */
public class ErrorResponse {

    @JsonProperty("error")
    private String error;

    @JsonProperty("code")
    private String code;

    @JsonProperty("message")
    private String message;

    /**
     * Default constructor for JSON deserialization.
     */
    public ErrorResponse() {
    }

    /**
     * Gets the error message from the API.
     *
     * @return The error message
     */
    public String getError() {
        return error;
    }

    /**
     * Sets the error message.
     *
     * @param error The error message
     */
    public void setError(String error) {
        this.error = error;
    }

    /**
     * Gets the error code from the API.
     *
     * @return The error code (e.g., "REJECTED_NETWORK")
     */
    public String getCode() {
        return code;
    }

    /**
     * Sets the error code.
     *
     * @param code The error code
     */
    public void setCode(String code) {
        this.code = code;
    }

    /**
     * Gets an additional error message if available.
     *
     * @return The additional message
     */
    public String getMessage() {
        return message;
    }

    /**
     * Sets an additional error message.
     *
     * @param message The additional message
     */
    public void setMessage(String message) {
        this.message = message;
    }

    /**
     * Gets the most appropriate error message to display.
     * 
     * @return The error message, preferring 'error' over 'message'
     */
    public String getDisplayMessage() {
        if (error != null && !error.trim().isEmpty()) {
            return error;
        }
        if (message != null && !message.trim().isEmpty()) {
            return message;
        }
        return "Unknown error occurred";
    }

    @Override
    public String toString() {
        return "ErrorResponse{" +
                "error='" + error + '\'' +
                ", code='" + code + '\'' +
                ", message='" + message + '\'' +
                '}';
    }
}