package tech.checkhim.models;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

/**
 * Represents a phone number verification request to the CheckHim API.
 * 
 * This class encapsulates the phone number to be verified and includes
 * a builder pattern for easy construction.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class VerificationRequest {

    @JsonProperty("number")
    @NotNull(message = "Phone number cannot be null")
    @NotBlank(message = "Phone number cannot be blank")
    private String number;

    @JsonProperty("type")
    private String type = "frontend";

    /**
     * Default constructor for JSON serialization.
     */
    public VerificationRequest() {
    }

    /**
     * Private constructor used by the builder.
     *
     * @param builder The builder instance
     */
    private VerificationRequest(Builder builder) {
        this.number = builder.number;
        this.type = builder.type;
    }

    /**
     * Gets the phone number to be verified.
     *
     * @return The phone number in international format (e.g., +5511984339000)
     */
    public String getNumber() {
        return number;
    }

    /**
     * Sets the phone number to be verified.
     *
     * @param number The phone number in international format
     */
    public void setNumber(String number) {
        this.number = number;
    }

    /**
     * Gets the request type.
     *
     * @return The request type (default: "frontend")
     */
    public String getType() {
        return type;
    }

    /**
     * Sets the request type.
     *
     * @param type The request type
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * Creates a new builder instance for constructing VerificationRequest objects.
     *
     * @return A new Builder instance
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder class for constructing VerificationRequest instances with a fluent API.
     */
    public static class Builder {
        private String number;
        private String type = "frontend";

        /**
         * Sets the phone number to be verified.
         *
         * @param number The phone number in international format (e.g., +5511984339000)
         * @return This builder instance for method chaining
         */
        public Builder number(String number) {
            this.number = number;
            return this;
        }

        /**
         * Sets the request type (internal use only).
         *
         * @param type The request type
         * @return This builder instance for method chaining
         */
        public Builder type(String type) {
            this.type = type;
            return this;
        }

        /**
         * Builds and returns a new VerificationRequest instance.
         *
         * @return A new VerificationRequest with the configured parameters
         * @throws IllegalArgumentException if required fields are not set
         */
        public VerificationRequest build() {
            if (number == null || number.trim().isEmpty()) {
                throw new IllegalArgumentException("Phone number is required and cannot be null or empty");
            }
            return new VerificationRequest(this);
        }
    }

    @Override
    public String toString() {
        return "VerificationRequest{" +
                "number='" + number + '\'' +
                ", type='" + type + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        VerificationRequest that = (VerificationRequest) o;

        if (!number.equals(that.number)) return false;
        return type.equals(that.type);
    }

    @Override
    public int hashCode() {
        int result = number.hashCode();
        result = 31 * result + type.hashCode();
        return result;
    }
}