/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util;

import com.cedarsoftware.util.AdjustableGZIPOutputStream;
import com.cedarsoftware.util.ByteUtilities;
import com.cedarsoftware.util.Convention;
import com.cedarsoftware.util.ExceptionUtilities;
import com.cedarsoftware.util.FastByteArrayOutputStream;
import com.cedarsoftware.util.LoggingConfig;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.Flushable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.Arrays;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

public final class IOUtilities {
    private static final int TRANSFER_BUFFER = 32768;
    private static final int DEFAULT_CONNECT_TIMEOUT = 5000;
    private static final int DEFAULT_READ_TIMEOUT = 30000;
    private static final int MIN_TIMEOUT = 1000;
    private static final int MAX_TIMEOUT = 300000;
    private static final boolean DEBUG = Boolean.parseBoolean(System.getProperty("io.debug", "false"));
    private static final Logger LOG = Logger.getLogger(IOUtilities.class.getName());

    private static void debug(String msg, Exception e) {
        if (DEBUG) {
            if (e == null) {
                LOG.fine(msg);
            } else {
                LOG.log(Level.FINE, msg, e);
            }
        }
    }

    private IOUtilities() {
    }

    private static int getValidatedTimeout(String propertyName, int defaultValue, String propertyType) {
        try {
            String propertyValue = System.getProperty(propertyName);
            if (propertyValue == null || propertyValue.trim().isEmpty()) {
                return defaultValue;
            }
            if (!propertyValue.matches("^-?\\d+$")) {
                IOUtilities.debug("Invalid " + propertyType + " format, using default", null);
                return defaultValue;
            }
            int timeout = Integer.parseInt(propertyValue.trim());
            if (timeout < 1000) {
                IOUtilities.debug("Configured " + propertyType + " too low, using minimum value", null);
                return 1000;
            }
            if (timeout > 300000) {
                IOUtilities.debug("Configured " + propertyType + " too high, using maximum value", null);
                return 300000;
            }
            return timeout;
        }
        catch (NumberFormatException e) {
            IOUtilities.debug("Invalid " + propertyType + " configuration detected, using defaults", null);
            return defaultValue;
        }
        catch (SecurityException e) {
            IOUtilities.debug("Security restriction accessing " + propertyType + " property, using defaults", null);
            return defaultValue;
        }
    }

    private static int getValidatedSizeProperty(String propertyName, int defaultValue, String propertyType) {
        try {
            String propertyValue = System.getProperty(propertyName);
            if (propertyValue == null || propertyValue.trim().isEmpty()) {
                return defaultValue;
            }
            if (!propertyValue.matches("^-?\\d+$")) {
                IOUtilities.debug("Invalid " + propertyType + " format, using default", null);
                return defaultValue;
            }
            long size = Long.parseLong(propertyValue.trim());
            if (size <= 0L) {
                IOUtilities.debug("Configured " + propertyType + " must be positive, using default", null);
                return defaultValue;
            }
            if (size > Integer.MAX_VALUE) {
                IOUtilities.debug("Configured " + propertyType + " too large, using maximum safe value", null);
                return Integer.MAX_VALUE;
            }
            return (int)size;
        }
        catch (NumberFormatException e) {
            IOUtilities.debug("Invalid " + propertyType + " configuration detected, using defaults", null);
            return defaultValue;
        }
        catch (SecurityException e) {
            IOUtilities.debug("Security restriction accessing " + propertyType + " property, using defaults", null);
            return defaultValue;
        }
    }

    private static int getDefaultMaxStreamSize() {
        return IOUtilities.getValidatedSizeProperty("io.max.stream.size", Integer.MAX_VALUE, "max stream size");
    }

    private static int getDefaultMaxDecompressionSize() {
        return IOUtilities.getValidatedSizeProperty("io.max.decompression.size", Integer.MAX_VALUE, "max decompression size");
    }

    private static void validateFilePath(File file) {
        Convention.throwIfNull(file, "File cannot be null");
        if (Boolean.parseBoolean(System.getProperty("io.path.validation.disabled", "false"))) {
            return;
        }
        String filePath = file.getPath();
        if (filePath.contains("../") || filePath.contains("..\\") || filePath.contains("/..") || filePath.contains("\\..")) {
            throw new SecurityException("Path traversal attempt detected: " + IOUtilities.sanitizePathForLogging(filePath));
        }
        if (filePath.indexOf(0) != -1) {
            throw new SecurityException("Null byte in file path: " + IOUtilities.sanitizePathForLogging(filePath));
        }
        if (filePath.contains("|") || filePath.contains(";") || filePath.contains("&") || filePath.contains("`") || filePath.contains("$")) {
            throw new SecurityException("Suspicious characters detected in file path: " + IOUtilities.sanitizePathForLogging(filePath));
        }
        IOUtilities.validateFileSystemSecurity(file, filePath);
    }

    private static void validateFileSystemSecurity(File file, String filePath) {
        try {
            String lowerPath;
            String lowerCanonical;
            String canonicalPath = file.getCanonicalPath();
            String absolutePath = file.getAbsolutePath();
            if (!canonicalPath.equals(absolutePath)) {
                if (System.getProperty("os.name", "").toLowerCase().contains("windows")) {
                    if (!canonicalPath.equalsIgnoreCase(absolutePath)) {
                        IOUtilities.debug("Potential symlink or case manipulation detected in file access", null);
                    }
                } else {
                    IOUtilities.debug("Potential symlink detected in file access", null);
                }
            }
            if ((lowerCanonical = canonicalPath.toLowerCase()).startsWith("/proc/") || lowerCanonical.startsWith("/sys/") || lowerCanonical.startsWith("/dev/") || lowerCanonical.equals("/etc/passwd") || lowerCanonical.equals("/etc/shadow") || lowerCanonical.startsWith("/etc/ssh/")) {
                throw new SecurityException("Access to system directory/file denied: " + IOUtilities.sanitizePathForLogging(canonicalPath));
            }
            if (System.getProperty("os.name", "").toLowerCase().contains("windows") && ((lowerPath = canonicalPath.toLowerCase()).contains("\\windows\\system32\\") || lowerPath.contains("\\windows\\syswow64\\") || lowerPath.endsWith("\\sam") || lowerPath.endsWith("\\system") || lowerPath.endsWith("\\security"))) {
                throw new SecurityException("Access to Windows system directory/file denied: " + IOUtilities.sanitizePathForLogging(canonicalPath));
            }
            if (canonicalPath.length() > 4096) {
                throw new SecurityException("File path too long (potential buffer overflow): " + IOUtilities.sanitizePathForLogging(canonicalPath));
            }
            IOUtilities.validatePathElements(canonicalPath);
        }
        catch (IOException e) {
            throw new SecurityException("Unable to validate file path security: " + IOUtilities.sanitizePathForLogging(filePath), e);
        }
    }

    private static void validatePathElements(String canonicalPath) {
        String[] pathElements;
        for (String element : pathElements = canonicalPath.split("[/\\\\]")) {
            if (element.isEmpty()) continue;
            if (element.startsWith(".") && (element.equals(".ssh") || element.equals(".gnupg") || element.equals(".aws") || element.equals(".docker"))) {
                throw new SecurityException("Access to sensitive hidden directory denied: " + IOUtilities.sanitizePathForLogging(element));
            }
            if (element.endsWith(".bak") || element.endsWith(".tmp") || element.endsWith(".old") || element.endsWith("~") || element.startsWith("core.")) {
                IOUtilities.debug("Accessing potentially sensitive file type detected", null);
            }
            if (!element.contains("\t") && !element.contains("\n") && !element.contains("\r")) continue;
            throw new SecurityException("Invalid characters in path element: " + IOUtilities.sanitizePathForLogging(element));
        }
    }

    private static void validateUrlProtocol(URLConnection connection) {
        if (connection == null || connection.getURL() == null) {
            return;
        }
        String protocol = connection.getURL().getProtocol();
        if (protocol == null) {
            throw new SecurityException("URL protocol cannot be null");
        }
        protocol = protocol.toLowerCase();
        if (Boolean.parseBoolean(System.getProperty("io.url.protocol.validation.disabled", "false"))) {
            IOUtilities.debug("URL protocol validation disabled via system property", null);
            return;
        }
        String allowedProtocolsProperty = System.getProperty("io.allowed.protocols", "http,https,file,jar");
        String[] allowedProtocols = allowedProtocolsProperty.toLowerCase().split(",");
        for (int i = 0; i < allowedProtocols.length; ++i) {
            allowedProtocols[i] = allowedProtocols[i].trim();
        }
        boolean isAllowed = false;
        for (String allowedProtocol : allowedProtocols) {
            if (!protocol.equals(allowedProtocol)) continue;
            isAllowed = true;
            break;
        }
        if (!isAllowed) {
            String sanitizedUrl = IOUtilities.sanitizeUrlForLogging(connection.getURL().toString());
            IOUtilities.debug("Blocked dangerous URL protocol: " + sanitizedUrl, null);
            throw new SecurityException("URL protocol '" + protocol + "' is not allowed. Allowed protocols: " + allowedProtocolsProperty);
        }
        IOUtilities.validateAgainstDangerousProtocols(protocol, allowedProtocols);
        if (protocol.equals("file") || protocol.equals("jar")) {
            IOUtilities.validateFileProtocolSafety(connection);
        }
        IOUtilities.debug("URL protocol validation passed for: " + protocol, null);
    }

    private static void validateAgainstDangerousProtocols(String protocol, String[] allowedProtocols) {
        String[] criticallyDangerousProtocols;
        for (String dangerous : criticallyDangerousProtocols = new String[]{"javascript", "data", "vbscript"}) {
            if (!protocol.equals(dangerous)) continue;
            throw new SecurityException("Critically dangerous protocol '" + protocol + "' is never allowed");
        }
        String[] potentiallyDangerousProtocols = new String[]{"netdoc", "mailto", "gopher", "ldap", "dict", "sftp", "tftp"};
        boolean explicitlyAllowed = false;
        for (String allowed : allowedProtocols) {
            if (!protocol.equals(allowed)) continue;
            explicitlyAllowed = true;
            break;
        }
        if (!explicitlyAllowed) {
            for (String dangerous : potentiallyDangerousProtocols) {
                if (!protocol.equals(dangerous)) continue;
                throw new SecurityException("Dangerous protocol '" + protocol + "' is forbidden unless explicitly allowed");
            }
        }
        if (protocol.contains(":") || protocol.contains("/") || protocol.contains("\\") || protocol.contains(" ") || protocol.contains("\t") || protocol.contains("\n") || protocol.contains("\r")) {
            throw new SecurityException("Invalid characters detected in protocol: " + protocol);
        }
    }

    private static void validateFileProtocolSafety(URLConnection connection) {
        String urlString = connection.getURL().toString();
        String protocol = connection.getURL().getProtocol();
        if (Boolean.parseBoolean(System.getProperty("io.file.protocol.validation.disabled", "false"))) {
            IOUtilities.debug("File protocol validation disabled via system property", null);
            return;
        }
        if ("jar".equals(protocol)) {
            if (urlString.contains("..") || urlString.contains("\u0000")) {
                throw new SecurityException("Dangerous path patterns detected in jar URL");
            }
            return;
        }
        if ("file".equals(protocol)) {
            String path = connection.getURL().getPath();
            if (path == null) {
                throw new SecurityException("File URL path cannot be null");
            }
            if (IOUtilities.isSystemPath(path)) {
                throw new SecurityException("File URL accesses system path: " + IOUtilities.sanitizeUrlForLogging(urlString));
            }
            if (path.contains("..") || path.contains("\u0000")) {
                throw new SecurityException("Dangerous path patterns detected in file URL");
            }
            if (IOUtilities.isSuspiciousPath(path)) {
                throw new SecurityException("Suspicious file path detected: " + IOUtilities.sanitizeUrlForLogging(urlString));
            }
            IOUtilities.debug("File protocol validation passed for resource path", null);
        }
    }

    private static boolean isSystemPath(String path) {
        if (path == null) {
            return false;
        }
        String lowerPath = path.toLowerCase();
        if (lowerPath.startsWith("/etc/") || lowerPath.startsWith("/proc/") || lowerPath.startsWith("/sys/") || lowerPath.startsWith("/dev/")) {
            return true;
        }
        return lowerPath.contains("system32") || lowerPath.contains("syswow64") || lowerPath.contains("\\windows\\") || lowerPath.contains("/windows/");
    }

    private static boolean isSuspiciousPath(String path) {
        if (path == null) {
            return false;
        }
        if (path.contains("/.ssh/") || path.contains("/.gnupg/") || path.contains("/.aws/") || path.contains("/.docker/")) {
            return true;
        }
        return path.endsWith("/passwd") || path.endsWith("/shadow") || path.contains("id_rsa") || path.contains("private");
    }

    private static String sanitizeUrlForLogging(String url) {
        if (url == null) {
            return "[null]";
        }
        boolean allowDetailedLogging = Boolean.parseBoolean(System.getProperty("io.debug.detailed.urls", "false"));
        if (!allowDetailedLogging) {
            try {
                URL urlObj = new URL(url);
                return "[" + urlObj.getProtocol() + "://...:" + url.length() + "-chars]";
            }
            catch (Exception e) {
                return "[malformed-url:" + url.length() + "-chars]";
            }
        }
        String sanitized = url.replaceAll("://[^@/]*@", "://[credentials]@");
        if (sanitized.length() > 200) {
            sanitized = sanitized.substring(0, 200) + "...[truncated]";
        }
        return sanitized;
    }

    private static String sanitizePathForLogging(String path) {
        if (path == null) {
            return "[null]";
        }
        boolean allowDetailedLogging = Boolean.parseBoolean(System.getProperty("io.debug.detailed.paths", "false"));
        if (!allowDetailedLogging) {
            if (path.contains("..")) {
                return "[path-with-traversal-pattern]";
            }
            if (path.contains("\u0000")) {
                return "[path-with-null-byte]";
            }
            if (path.toLowerCase().contains("system32") || path.toLowerCase().contains("syswow64")) {
                return "[windows-system-path]";
            }
            if (path.startsWith("/proc/") || path.startsWith("/sys/") || path.startsWith("/dev/") || path.startsWith("/etc/")) {
                return "[unix-system-path]";
            }
            if (path.contains("/.")) {
                return "[hidden-directory-path]";
            }
            return "[file-path:" + path.length() + "-chars]";
        }
        if (path.length() > 100) {
            path = path.substring(0, 100) + "...[truncated]";
        }
        return path.replaceAll("[\\x00-\\x1F\\x7F]", "?");
    }

    public static InputStream getInputStream(URLConnection c) {
        InputStream is;
        Convention.throwIfNull(c, "URLConnection cannot be null");
        IOUtilities.validateUrlProtocol(c);
        IOUtilities.optimizeConnection(c);
        String enc = c.getContentEncoding();
        try {
            is = c.getInputStream();
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
            return null;
        }
        if (enc != null) {
            if ("gzip".equalsIgnoreCase(enc) || "x-gzip".equalsIgnoreCase(enc)) {
                try {
                    is = new GZIPInputStream(is, 32768);
                }
                catch (IOException e) {
                    ExceptionUtilities.uncheckedThrow(e);
                    return null;
                }
            } else if ("deflate".equalsIgnoreCase(enc)) {
                is = new InflaterInputStream(is, new Inflater(), 32768);
            }
        }
        return new BufferedInputStream(is, 32768);
    }

    private static void optimizeConnection(URLConnection c) {
        if (c instanceof HttpURLConnection) {
            HttpURLConnection http = (HttpURLConnection)c;
            http.setInstanceFollowRedirects(true);
            http.setUseCaches(false);
            int connectTimeout = IOUtilities.getValidatedTimeout("io.connect.timeout", 5000, "connect timeout");
            int readTimeout = IOUtilities.getValidatedTimeout("io.read.timeout", 30000, "read timeout");
            http.setConnectTimeout(connectTimeout);
            http.setReadTimeout(readTimeout);
            c.setRequestProperty("Accept-Encoding", "gzip, x-gzip, deflate");
        }
    }

    public static void transfer(File f, URLConnection c, TransferCallback cb) {
        Convention.throwIfNull(f, "File cannot be null");
        Convention.throwIfNull(c, "URLConnection cannot be null");
        IOUtilities.validateFilePath(f);
        try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(f.toPath(), new OpenOption[0]));
             BufferedOutputStream out = new BufferedOutputStream(c.getOutputStream());){
            IOUtilities.transfer((InputStream)in, out, cb);
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static void transfer(URLConnection c, File f, TransferCallback cb) {
        Convention.throwIfNull(c, "URLConnection cannot be null");
        Convention.throwIfNull(f, "File cannot be null");
        IOUtilities.validateFilePath(f);
        try (InputStream in = IOUtilities.getInputStream(c);){
            IOUtilities.transfer(in, f, cb);
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static void transfer(InputStream s, File f, TransferCallback cb) {
        Convention.throwIfNull(s, "InputStream cannot be null");
        Convention.throwIfNull(f, "File cannot be null");
        IOUtilities.validateFilePath(f);
        try (BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(f.toPath(), new OpenOption[0]));){
            IOUtilities.transfer(s, out, cb);
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    private static byte[] createSafeCallbackBuffer(byte[] buffer, int count) {
        if (count <= 0) {
            return new byte[0];
        }
        byte[] callbackBuffer = new byte[count];
        System.arraycopy(buffer, 0, callbackBuffer, 0, count);
        return callbackBuffer;
    }

    public static void transfer(InputStream in, OutputStream out, TransferCallback cb) {
        Convention.throwIfNull(in, "InputStream cannot be null");
        Convention.throwIfNull(out, "OutputStream cannot be null");
        try {
            int count;
            byte[] buffer = new byte[32768];
            while ((count = in.read(buffer)) != -1) {
                out.write(buffer, 0, count);
                if (cb == null) continue;
                byte[] callbackBuffer = IOUtilities.createSafeCallbackBuffer(buffer, count);
                cb.bytesTransferred(callbackBuffer, count);
                if (!cb.isCancelled()) continue;
                break;
            }
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static void transfer(InputStream in, byte[] bytes) {
        Convention.throwIfNull(in, "InputStream cannot be null");
        Convention.throwIfNull(bytes, "byte array cannot be null");
        try {
            new DataInputStream(in).readFully(bytes);
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static void transfer(InputStream in, OutputStream out) {
        Convention.throwIfNull(in, "InputStream cannot be null");
        Convention.throwIfNull(out, "OutputStream cannot be null");
        try {
            int count;
            byte[] buffer = new byte[32768];
            while ((count = in.read(buffer)) != -1) {
                out.write(buffer, 0, count);
            }
            out.flush();
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void transfer(File file, OutputStream out) {
        Convention.throwIfNull(file, "File cannot be null");
        Convention.throwIfNull(out, "OutputStream cannot be null");
        IOUtilities.validateFilePath(file);
        try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]), 32768);){
            IOUtilities.transfer((InputStream)in, out);
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
        finally {
            IOUtilities.flush(out);
        }
    }

    public static void close(XMLStreamReader reader) {
        if (reader != null) {
            try {
                reader.close();
            }
            catch (XMLStreamException e) {
                ExceptionUtilities.uncheckedThrow(e);
            }
        }
    }

    public static void close(XMLStreamWriter writer) {
        if (writer != null) {
            try {
                writer.close();
            }
            catch (XMLStreamException e) {
                ExceptionUtilities.uncheckedThrow(e);
            }
        }
    }

    public static void close(Closeable c) {
        if (c != null) {
            try {
                c.close();
            }
            catch (IOException e) {
                ExceptionUtilities.uncheckedThrow(e);
            }
        }
    }

    public static void flush(Flushable f) {
        if (f != null) {
            try {
                f.flush();
            }
            catch (IOException e) {
                ExceptionUtilities.uncheckedThrow(e);
            }
        }
    }

    public static void flush(XMLStreamWriter writer) {
        if (writer != null) {
            try {
                writer.flush();
            }
            catch (XMLStreamException e) {
                ExceptionUtilities.uncheckedThrow(e);
            }
        }
    }

    public static byte[] inputStreamToBytes(InputStream in) {
        return IOUtilities.inputStreamToBytes(in, IOUtilities.getDefaultMaxStreamSize());
    }

    public static byte[] inputStreamToBytes(InputStream in, int maxSize) {
        Convention.throwIfNull(in, "Inputstream cannot be null");
        if (maxSize <= 0) {
            throw new IllegalArgumentException("maxSize must be > 0");
        }
        FastByteArrayOutputStream out = new FastByteArrayOutputStream(Math.min(16384, maxSize));
        try {
            int total;
            int count;
            byte[] buffer = new byte[Math.min(32768, maxSize)];
            for (total = 0; total < maxSize && (count = in.read(buffer, 0, Math.min(buffer.length, maxSize - total))) != -1; total += count) {
                if (total + count > maxSize) {
                    throw new IOException("Stream exceeds maximum allowed size: " + maxSize);
                }
                out.write(buffer, 0, count);
            }
            if (total >= maxSize && in.read() != -1) {
                throw new IOException("Stream exceeds maximum allowed size: " + maxSize);
            }
            byte[] byArray = out.toByteArray();
            out.close();
            return byArray;
        }
        catch (Throwable throwable) {
            try {
                try {
                    out.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                ExceptionUtilities.uncheckedThrow(e);
                return null;
            }
        }
    }

    public static void transfer(URLConnection c, byte[] bytes) {
        Convention.throwIfNull(c, "URLConnection cannot be null");
        Convention.throwIfNull(bytes, "byte array cannot be null");
        try (BufferedOutputStream out = new BufferedOutputStream(c.getOutputStream());){
            ((OutputStream)out).write(bytes);
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static void compressBytes(ByteArrayOutputStream original, ByteArrayOutputStream compressed) {
        Convention.throwIfNull(original, "Original ByteArrayOutputStream cannot be null");
        Convention.throwIfNull(compressed, "Compressed ByteArrayOutputStream cannot be null");
        try (AdjustableGZIPOutputStream gzipStream = new AdjustableGZIPOutputStream((OutputStream)compressed, 1);){
            original.writeTo(gzipStream);
            gzipStream.flush();
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static void compressBytes(FastByteArrayOutputStream original, FastByteArrayOutputStream compressed) {
        Convention.throwIfNull(original, "Original FastByteArrayOutputStream cannot be null");
        Convention.throwIfNull(compressed, "Compressed FastByteArrayOutputStream cannot be null");
        try (AdjustableGZIPOutputStream gzipStream = new AdjustableGZIPOutputStream((OutputStream)compressed, 1);){
            ((DeflaterOutputStream)gzipStream).write(original.toByteArray(), 0, original.size());
            gzipStream.flush();
        }
        catch (IOException e) {
            ExceptionUtilities.uncheckedThrow(e);
        }
    }

    public static byte[] compressBytes(byte[] bytes) {
        return IOUtilities.compressBytes(bytes, 0, bytes.length);
    }

    public static byte[] compressBytes(byte[] bytes, int offset, int len) {
        Convention.throwIfNull(bytes, "Byte array cannot be null");
        FastByteArrayOutputStream byteStream = new FastByteArrayOutputStream();
        try {
            try (AdjustableGZIPOutputStream gzipStream = new AdjustableGZIPOutputStream((OutputStream)byteStream, 1);){
                ((DeflaterOutputStream)gzipStream).write(bytes, offset, len);
                gzipStream.flush();
            }
            byte[] byArray = Arrays.copyOf(byteStream.toByteArray(), byteStream.size());
            byteStream.close();
            return byArray;
        }
        catch (Throwable throwable) {
            try {
                try {
                    byteStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new RuntimeException("Error compressing bytes.", e);
            }
        }
    }

    public static byte[] uncompressBytes(byte[] bytes) {
        return IOUtilities.uncompressBytes(bytes, 0, bytes.length, IOUtilities.getDefaultMaxDecompressionSize());
    }

    public static byte[] uncompressBytes(byte[] bytes, int offset, int len) {
        return IOUtilities.uncompressBytes(bytes, offset, len, IOUtilities.getDefaultMaxDecompressionSize());
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static byte[] uncompressBytes(byte[] bytes, int offset, int len, int maxSize) {
        Objects.requireNonNull(bytes, "Byte array cannot be null");
        if (maxSize <= 0) {
            throw new IllegalArgumentException("maxSize must be > 0");
        }
        if (ByteUtilities.isGzipped(bytes, offset)) {
            try (ByteArrayInputStream byteStream = new ByteArrayInputStream(bytes, offset, len);){
                byte[] byArray;
                try (GZIPInputStream gzipStream = new GZIPInputStream((InputStream)byteStream, 32768);){
                    byArray = IOUtilities.inputStreamToBytes(gzipStream, maxSize);
                }
                return byArray;
            }
            catch (IOException e) {
                throw new RuntimeException("Error uncompressing bytes", e);
            }
        }
        return Arrays.copyOfRange(bytes, offset, offset + len);
    }

    static {
        LoggingConfig.init();
    }

    @FunctionalInterface
    public static interface TransferCallback {
        public void bytesTransferred(byte[] var1, int var2);

        default public boolean isCancelled() {
            return false;
        }
    }
}

