/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util;

import com.cedarsoftware.util.ConcurrentHashMapNullSafe;
import java.lang.ref.WeakReference;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

public class TTLCache<K, V>
implements Map<K, V>,
AutoCloseable {
    private final long ttlMillis;
    private final int maxSize;
    private final ConcurrentMap<K, CacheEntry<K, V>> cacheMap;
    private final ReentrantLock lock = new ReentrantLock();
    private final Node<K, V> head;
    private final Node<K, V> tail;
    private PurgeTask purgeTask;
    private static volatile ScheduledExecutorService scheduler = TTLCache.createScheduler();

    private static ScheduledExecutorService createScheduler() {
        return Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "TTLCache-Purge-Thread");
            thread.setDaemon(true);
            return thread;
        });
    }

    private static synchronized ScheduledExecutorService ensureScheduler() {
        if (scheduler == null || scheduler.isShutdown() || scheduler.isTerminated()) {
            scheduler = TTLCache.createScheduler();
        }
        return scheduler;
    }

    public TTLCache(long ttlMillis) {
        this(ttlMillis, -1, 60000L);
    }

    public TTLCache(long ttlMillis, int maxSize) {
        this(ttlMillis, maxSize, 60000L);
    }

    public TTLCache(long ttlMillis, int maxSize, long cleanupIntervalMillis) {
        if (ttlMillis < 1L) {
            throw new IllegalArgumentException("TTL must be at least 1 millisecond.");
        }
        if (cleanupIntervalMillis < 10L) {
            throw new IllegalArgumentException("cleanupIntervalMillis must be at least 10 milliseconds.");
        }
        this.ttlMillis = ttlMillis;
        this.maxSize = maxSize;
        this.cacheMap = new ConcurrentHashMapNullSafe<K, CacheEntry<K, V>>();
        this.head = new Node<Object, Object>(null, null);
        this.tail = new Node<Object, Object>(null, null);
        this.head.next = this.tail;
        this.tail.prev = this.head;
        this.schedulePurgeTask(cleanupIntervalMillis);
    }

    private void schedulePurgeTask(long cleanupIntervalMillis) {
        WeakReference cacheRef = new WeakReference(this);
        PurgeTask task = new PurgeTask(cacheRef);
        ScheduledFuture<?> future = TTLCache.ensureScheduler().scheduleAtFixedRate(task, cleanupIntervalMillis, cleanupIntervalMillis, TimeUnit.MILLISECONDS);
        task.setFuture(future);
        this.purgeTask = task;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void purgeExpiredEntries() {
        long currentTime = System.currentTimeMillis();
        Iterator it = this.cacheMap.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = it.next();
            CacheEntry cacheEntry = (CacheEntry)entry.getValue();
            if (cacheEntry.expiryTime >= currentTime) continue;
            it.remove();
            this.lock.lock();
            try {
                this.unlink(cacheEntry.node);
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeEntry(K cacheKey) {
        CacheEntry entry = (CacheEntry)this.cacheMap.remove(cacheKey);
        if (entry != null) {
            Node node = entry.node;
            this.lock.lock();
            try {
                this.unlink(node);
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    private void unlink(Node<K, V> node) {
        node.prev.next = node.next;
        node.next.prev = node.prev;
        node.prev = null;
        node.next = null;
        node.value = null;
    }

    private void moveToTail(Node<K, V> node) {
        node.prev.next = node.next;
        node.next.prev = node.prev;
        node.prev = this.tail.prev;
        node.next = this.tail;
        this.tail.prev.next = node;
        this.tail.prev = node;
    }

    private void insertAtTail(Node<K, V> node) {
        node.prev = this.tail.prev;
        node.next = this.tail;
        this.tail.prev.next = node;
        this.tail.prev = node;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V put(K key, V value) {
        long expiryTime = System.currentTimeMillis() + this.ttlMillis;
        Node<K, V> node = new Node<K, V>(key, value);
        CacheEntry<K, V> newEntry = new CacheEntry<K, V>(node, expiryTime);
        CacheEntry<K, V> oldEntry = this.cacheMap.put(key, newEntry);
        boolean acquired = this.lock.tryLock();
        try {
            if (acquired) {
                Node lruNode;
                if (oldEntry != null) {
                    this.unlink(oldEntry.node);
                }
                this.insertAtTail(node);
                if (this.maxSize > -1 && this.cacheMap.size() > this.maxSize && (lruNode = this.head.next) != this.tail) {
                    this.removeEntry(lruNode.key);
                }
            }
        }
        finally {
            if (acquired) {
                this.lock.unlock();
            }
        }
        return oldEntry != null ? (V)oldEntry.node.value : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V get(Object key) {
        CacheEntry entry = (CacheEntry)this.cacheMap.get(key);
        if (entry == null) {
            return null;
        }
        long currentTime = System.currentTimeMillis();
        if (entry.expiryTime < currentTime) {
            this.removeEntry(key);
            return null;
        }
        Object value = entry.node.value;
        boolean acquired = this.lock.tryLock();
        try {
            if (acquired) {
                this.moveToTail(entry.node);
            }
        }
        finally {
            if (acquired) {
                this.lock.unlock();
            }
        }
        return value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V remove(Object key) {
        CacheEntry entry = (CacheEntry)this.cacheMap.remove(key);
        if (entry != null) {
            Object value = entry.node.value;
            this.lock.lock();
            try {
                this.unlink(entry.node);
            }
            finally {
                this.lock.unlock();
            }
            return value;
        }
        return null;
    }

    @Override
    public void clear() {
        this.cacheMap.clear();
        this.lock.lock();
        try {
            this.head.next = this.tail;
            this.tail.prev = this.head;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public int size() {
        return this.cacheMap.size();
    }

    @Override
    public boolean isEmpty() {
        return this.cacheMap.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        CacheEntry entry = (CacheEntry)this.cacheMap.get(key);
        if (entry == null) {
            return false;
        }
        if (entry.expiryTime < System.currentTimeMillis()) {
            this.removeEntry(key);
            return false;
        }
        return true;
    }

    @Override
    public boolean containsValue(Object value) {
        for (CacheEntry entry : this.cacheMap.values()) {
            Object entryValue = entry.node.value;
            if (!Objects.equals(entryValue, value)) continue;
            return true;
        }
        return false;
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        for (Map.Entry<K, V> e : m.entrySet()) {
            this.put(e.getKey(), e.getValue());
        }
    }

    @Override
    public Set<K> keySet() {
        HashSet keys = new HashSet();
        for (CacheEntry entry : this.cacheMap.values()) {
            Object key = entry.node.key;
            keys.add(key);
        }
        return keys;
    }

    @Override
    public Collection<V> values() {
        ArrayList values = new ArrayList();
        for (CacheEntry entry : this.cacheMap.values()) {
            Object value = entry.node.value;
            values.add(value);
        }
        return values;
    }

    @Override
    public Set<Map.Entry<K, V>> entrySet() {
        return new EntrySet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Map)) {
            return false;
        }
        Map other = (Map)o;
        this.lock.lock();
        try {
            boolean bl = this.entrySet().equals(other.entrySet());
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int hashCode() {
        this.lock.lock();
        try {
            int hash = 0;
            for (Map.Entry entry : this.cacheMap.entrySet()) {
                Object key = entry.getKey();
                Object value = ((CacheEntry)entry.getValue()).node.value;
                int keyHash = key == null ? 0 : key.hashCode();
                int valueHash = value == null ? 0 : value.hashCode();
                hash += keyHash ^ valueHash;
            }
            int n = hash;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        this.lock.lock();
        try {
            StringBuilder sb = new StringBuilder();
            sb.append('{');
            Iterator<Map.Entry<K, V>> it = this.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<K, V> entry = it.next();
                sb.append(entry.getKey()).append('=').append(entry.getValue());
                if (!it.hasNext()) continue;
                sb.append(", ");
            }
            sb.append('}');
            String string = sb.toString();
            return string;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() {
        if (this.purgeTask != null) {
            this.purgeTask.cancel();
            this.purgeTask = null;
        }
    }

    ScheduledFuture<?> getPurgeFuture() {
        return this.purgeTask == null ? null : this.purgeTask.getFuture();
    }

    public static synchronized boolean shutdown() {
        if (scheduler == null || scheduler.isShutdown()) {
            return true;
        }
        scheduler.shutdown();
        try {
            boolean terminated = scheduler.awaitTermination(5L, TimeUnit.SECONDS);
            if (!terminated) {
                scheduler.shutdownNow();
                terminated = scheduler.awaitTermination(1L, TimeUnit.SECONDS);
            }
            scheduler = null;
            return terminated;
        }
        catch (InterruptedException e) {
            scheduler.shutdownNow();
            scheduler = null;
            Thread.currentThread().interrupt();
            return false;
        }
    }

    private static class Node<K, V> {
        final K key;
        V value;
        Node<K, V> prev;
        Node<K, V> next;

        Node(K key, V value) {
            this.key = key;
            this.value = value;
        }
    }

    private static class PurgeTask
    implements Runnable {
        private final WeakReference<TTLCache<?, ?>> cacheRef;
        private volatile boolean canceled = false;
        private ScheduledFuture<?> future;

        PurgeTask(WeakReference<TTLCache<?, ?>> cacheRef) {
            this.cacheRef = cacheRef;
        }

        void setFuture(ScheduledFuture<?> future) {
            this.future = future;
        }

        ScheduledFuture<?> getFuture() {
            return this.future;
        }

        @Override
        public void run() {
            TTLCache cache = (TTLCache)this.cacheRef.get();
            if (cache == null) {
                this.cancel();
            } else {
                cache.purgeExpiredEntries();
            }
        }

        private void cancel() {
            if (!this.canceled) {
                this.canceled = true;
                if (this.future != null) {
                    this.future.cancel(false);
                }
            }
        }
    }

    private static class CacheEntry<K, V> {
        final Node<K, V> node;
        final long expiryTime;

        CacheEntry(Node<K, V> node, long expiryTime) {
            this.node = node;
            this.expiryTime = expiryTime;
        }
    }

    private class EntrySet
    extends AbstractSet<Map.Entry<K, V>> {
        private EntrySet() {
        }

        @Override
        public Iterator<Map.Entry<K, V>> iterator() {
            return new EntryIterator();
        }

        @Override
        public int size() {
            return TTLCache.this.size();
        }

        @Override
        public void clear() {
            TTLCache.this.clear();
        }
    }

    private class EntryIterator
    implements Iterator<Map.Entry<K, V>> {
        private final Iterator<Map.Entry<K, CacheEntry<K, V>>> iterator;
        private Map.Entry<K, CacheEntry<K, V>> current;

        EntryIterator() {
            this.iterator = TTLCache.this.cacheMap.entrySet().iterator();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public Map.Entry<K, V> next() {
            this.current = this.iterator.next();
            Object key = this.current.getValue().node.key;
            Object value = this.current.getValue().node.value;
            return new AbstractMap.SimpleEntry(key, value);
        }

        @Override
        public void remove() {
            if (this.current == null) {
                throw new IllegalStateException();
            }
            Object cacheKey = this.current.getKey();
            TTLCache.this.removeEntry(cacheKey);
            this.current = null;
        }
    }
}

