/*
 * Copyright 2010-2013, the original author or authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.domain.environment;

import com.cloudbees.clickstack.domain.metadata.Metadata;
import com.cloudbees.clickstack.util.Files2;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Arrays;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>See <a href="http://genapp-docs.cloudbees.com/plugins.html#setup-environment-variables">
 * genapp documentation / plugins / Setup Environment Variables</a></p>
 *
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class Environment {
    public static final String DEFAULT_JAVA_VERSION = "1.7";
    private static final Pattern VARIABLE = Pattern.compile("\\$([A-Za-z0-9_]+|\\{[A-Za-z0-9_.]+\\}|\\$)");
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-dir">app_dir</a>:
     * The application source package directory</p>
     */
    @Nonnull
    public final Path appDir;
    /**
     * directory that contains the plugin's setup script, aka {@code plugin_dir}
     */
    @Nonnull
    public final Path clickstackDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#pkg-dir">pkd_dir</a>:
     * The application source package directory</p>
     */
    @Nonnull
    public final Path packageDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-user">app_user</a>:
     * The user associated with the application being deployed</p>
     */
    @Nonnull
    public final String appUser;
    /**
     * <p>Application name</p>
     */
    @Nonnull
    public final String appName;
    /**
     * CloudBees account
     */
    @Nonnull
    public final String accountName;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-id">app_id</a>:
     * The unique ID of the application being deployed</p>
     */
    @Nonnull
    public final String appId;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-port">app_port</a>:
     * The genapp assigned port for the application
     * <p/>
     */
    public final int appPort;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#control-dir">control_dir</a>:
     * The control subdirectory located within the genapp directory</p>
     */
    @Nonnull
    public final Path controlDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#log-dir">log_dir</a>:
     * The conventional location for application logs</p>
     */
    @Nonnull
    public final Path logDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#genapp-dir"">genapp_dir</a>:
     * The genapp subdirectory located within the application directory
     * </p>
     * <p>This variable is equivalent to {@code $appDir/.genapp}.</p>
     * <p>Plugins may use this variable to reference files located in the genapp subdirectory.
     * These may be files created by other plugins that have already performed their setup operations or files that the plugin itself creates.
     * </p>
     */
    @Nonnull
    public final Path genappDir;
    /**
     * Convenient directory for application to embed custom extra files, like runtime dependencies.
     * <p/>
     * This variable is equivalent to {@code $appDir/app-extra-files}
     */
    @Nonnull
    public final Path appExtraFiles;
    /**
     * Identifier of the server on which the app is deployed, usually the EC2 Instance ID, otherwise the server hostname
     */
    @Nonnull
    public final String serverInstanceId;
    protected final Logger logger = LoggerFactory.getLogger(getClass());
    protected final FileSystem fs = FileSystems.getDefault();
    public Path javaHome;

    public Environment(@Nonnull Path appDir,
                       @Nonnull Path clickstackDir,
                       @Nonnull Path packageDir,
                       @Nonnull String appUser,
                       @Nonnull String appId,
                       int appPort,
                       @Nonnull String appName,
                       @Nonnull String accountName,
                       @Nonnull String serverInstanceId) {
        this(appDir, clickstackDir, packageDir,
                appUser, appId, appPort, appDir.resolve(".genapp/control"), appDir.resolve(".genapp/log"),
                appName, accountName, serverInstanceId);
    }

    public Environment(
            @Nonnull Path appDir,
            @Nonnull Path clickstackDir,
            @Nonnull Path packageDir,
            @Nonnull String appUser,
            @Nonnull String appId,
            int appPort,
            @Nonnull Path controlDir,
            @Nonnull Path logDir,
            @Nonnull String appName,
            @Nonnull String accountName,
            @Nonnull String serverInstanceId
    ) {
        this.appDir = Preconditions.checkNotNull(appDir, "appDir");
        this.clickstackDir = Preconditions.checkNotNull(clickstackDir, "clickstackDir");
        this.packageDir = Preconditions.checkNotNull(packageDir, "packageDir");
        this.appUser = Preconditions.checkNotNull(appUser, "appUser");
        this.appId = Preconditions.checkNotNull(appId, "appId");
        this.appPort = appPort;
        this.controlDir = Preconditions.checkNotNull(controlDir,"controlDir");
        this.logDir = Preconditions.checkNotNull(logDir, "logDir");
        this.genappDir = appDir.resolve(".genapp");
        this.appExtraFiles = appDir.resolve("app-extra-files");
        this.appName = Preconditions.checkNotNull(appName, "appName");
        this.accountName = Preconditions.checkNotNull(accountName, "accountName");
        this.serverInstanceId = Preconditions.checkNotNull(serverInstanceId, "serverInstanceId");
    }

    @Override
    public String toString() {
        return "Environment{" + ", \n" +
                " appUser='" + appUser + '\'' + ", \n" +
                " appId='" + appId + '\'' + ", \n" +
                " appName='" + appName + '\'' + ", \n" +
                " accountName='" + accountName + '\'' + ", \n" +
                " appPort=" + appPort + ", \n" +
                " appDir=" + appDir.toAbsolutePath() + ", \n" +
                " logDir=" + logDir.toAbsolutePath() + ", \n" +
                " genappDir=" + genappDir.toAbsolutePath() + ", \n" +
                " controlDir=" + controlDir.toAbsolutePath() + ", \n" +
                " clickstackDir=" + clickstackDir.toAbsolutePath() + ", \n" +
                " packageDir=" + packageDir.toAbsolutePath() + ", \n" +
                " instanceId=" + serverInstanceId + ", \n" +
                '}';
    }

    /**
     * Expand variables in string as <tt>$variable</tt> or <tt>${variable}</tt>/ Invalid variables are left as-is.
     * <p/>
     * Adapted from hudson.Util#replaceMacro
     */
    public String expand(@Nullable String s) {
        if (s == null) {
            return null;
        }

        int idx = 0;
        while (true) {
            Matcher m = VARIABLE.matcher(s);
            if (!m.find(idx)) return s;
            String key = m.group().substring(1);

            // escape the dollar sign or get the key to resolve
            String value;
            if (key.charAt(0) == '$') {
                value = "$";
            } else {
                if (key.charAt(0) == '{') key = key.substring(1, key.length() - 1);
                value = resolve(key);
            }

            if (value == null)
                idx = m.end(); // skip this
            else {
                s = s.substring(0, m.start()) + value + s.substring(m.end());
                idx = m.start() + value.length();
            }
        }
    }

    private String resolve(String key) {
        switch (key) {
            case "app_dir":
                return appDir.toAbsolutePath().toString();
            case "plugin_dir":
                return clickstackDir.toAbsolutePath().toString();
            case "pkg_dir":
                return packageDir.toAbsolutePath().toString();
            case "app_user":
                return appUser;
            case "app_id":
                return appId;
            case "app_port":
                return String.valueOf(appPort);
            case "control_dir":
                return controlDir.toAbsolutePath().toString();
            case "log_dir":
                return logDir.toAbsolutePath().toString();
            case "genapp_dir":
                return genappDir.toAbsolutePath().toString();
            case "app-extra-files":
                return appExtraFiles.toAbsolutePath().toString();
            default:
                return key;
        }
    }
}
