/*
 * Copyright 2010-2013, CloudBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.util;

import com.cloudbees.clickstack.domain.environment.Environment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.NoSuchElementException;

/**
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class CommandLineUtils {

    private final static Logger logger = LoggerFactory.getLogger(CommandLineUtils.class);

    public static Environment argumentsToEnvironment(@Nonnull String[] args) {
        Path appDir = getOptionAsPath("app_dir", args);
        Path clickstackDir = getOptionAsPath("plugin_dir", args, FileSystems.getDefault().getPath("."));
        Path packageDir = getOptionAsPath("pkg_dir", args);
        String appUser = getOption("app_user", args, System.getProperty("user.name"));
        String appId = getOption("app_id", args);
        int appPort = getOptionAsInt("app_port", args);
        Path genappDir = appDir.resolve(".genapp");
        Path controlDir = getOptionAsPath("control_dir", args, genappDir.resolve("control"));
        Path logDir = getOptionAsPath("log_dir", args, genappDir.resolve("log"));

        String instanceId;
        try {
            instanceId = getOption("instance_id", args, InetAddress.getLocalHost().getHostName());
        } catch (UnknownHostException e) {
            instanceId = "unknown";
        }

        String appFullyQualifiedName = getOption("app_fully_qualified_name", args, System.getProperty("user.name") + "/unknown");
        String appName;
        String accountName;
        if (appFullyQualifiedName.contains("/")) {
            accountName = Strings2.substringBeforeFirst(appFullyQualifiedName, '/');
            appName = Strings2.substringAfterFirst(appFullyQualifiedName, '/');
        } else {
            accountName = System.getProperty("user.name");
            appName = appFullyQualifiedName;
        }

        return new Environment(
                appDir, clickstackDir, packageDir, appUser, appId, appPort, controlDir, logDir,
                appName, accountName, instanceId
        );
    }

    @Nonnull
    public static String getOption(@Nonnull String name, @Nonnull String[] args) throws NoSuchElementException {
        return getOption(name, args, true, true);
    }

    @Nullable
    public static String getOption(@Nonnull String name, @Nonnull String[] args, @Nullable String defaultValue) throws NoSuchElementException {
        try {
            return getOption(name, args, true, true);
        } catch (NoSuchElementException e) {
            logger.info("Parameter '{}' not found, use default value {}", name, defaultValue);
            return defaultValue;
        }
    }

    @Nonnull
    public static Path getOptionAsPath(@Nonnull String name, @Nonnull String[] args) throws NoSuchElementException {
        return FileSystems.getDefault().getPath(getOption(name, args, true, true));
    }

    @Nullable
    public static Path getOptionAsPath(@Nonnull String name, @Nonnull String[] args, @Nullable Path defaultValue) throws NoSuchElementException {
        try {
            return FileSystems.getDefault().getPath(getOption(name, args, true, true));
        } catch (NoSuchElementException e) {
            logger.info("Parameter '{}' not found, use default value {}", name, defaultValue);
            return defaultValue;
        }
    }

    public static int getOptionAsInt(@Nonnull String name, @Nonnull String[] args) throws NoSuchElementException {
        String value = getOption(name, args, true, true);
        try {
            return Integer.parseInt(value);
        } catch (NumberFormatException nfe) {
            NumberFormatException e = new NumberFormatException("Invalid numeric option '" + name + "'='" + value + "'");
            e.initCause(nfe);
            throw e;
        }
    }

    @Nonnull
    public static String getOption(@Nonnull String name, @Nonnull String[] args, boolean defaultToSystemProperty, boolean defaultToEnvironmentVariable) throws NoSuchElementException {
        for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if (Strings2.beginWith(arg, "--")) {
                arg = arg.substring(1);
            }

            if (arg.equals("-" + name)) {
                if ((i + 1) < args.length) {
                    return args[i + 1];
                }
            } else if (Strings2.beginWith(arg, "-" + name + "=")) {
                return Strings2.substringAfterFirst(arg, '=');
            }
        }
        if (defaultToSystemProperty && (System.getProperty(name) != null)) {
            return System.getProperty(name);
        }
        if (defaultToEnvironmentVariable && System.getenv(name) != null) {
            return System.getenv(name);
        }

        throw new NoSuchElementException("Argument '" + name + "' not found in " + Arrays.asList(args));
    }
}
