/*
 * Copyright 2010-2013, CloudBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.plugin.java;

import com.cloudbees.clickstack.domain.environment.Environment;
import com.cloudbees.clickstack.domain.metadata.Metadata;
import com.cloudbees.clickstack.util.Files2;
import com.cloudbees.clickstack.util.HttpUtils;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.net.URL;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Arrays;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class JavaPlugin {
    public static final String DEFAULT_JAVA_VERSION = "1.7";
    private static final Pattern VARIABLE = Pattern.compile("\\$([A-Za-z0-9_]+|\\{[A-Za-z0-9_.]+\\}|\\$)");
    protected final Logger logger = LoggerFactory.getLogger(getClass());
    protected final FileSystem fs = FileSystems.getDefault();

    public JavaPluginResult setup(Metadata metadata, Environment environment) throws IOException {
        String javaVersion = metadata.getRuntimeParameter("java", "version", DEFAULT_JAVA_VERSION);
        Path javaHome;
        switch (javaVersion) {
            case "6":
            case "1.6":
                javaHome = fs.getPath("/opt/java6");
                break;
            case "7":
            case "1.7":
                javaHome = fs.getPath("/opt/java7");
                break;
            case "8":
            case "1.8":
                javaHome = fs.getPath("/opt/java8");
                break;
            case "custom":
                logger.warn("You're running with a custom JRE, AT YOUR OWN RISK");
                String path = expand(metadata.getRuntimeParameter("java", "home"), environment);
                javaHome = fs.getPath(path);
                break;
            default:
                if (javaVersion.startsWith("http://") || javaVersion.startsWith("https://")) {
                    javaHome = installJvm(javaVersion, environment);
                } else {
                    javaHome = fs.getPath("/opt/").resolve(javaVersion);
                }
                break;
        }

        Preconditions.checkState(Files.exists(javaHome), "JAVA_HOME %s does not exist for specified java version %s", javaHome, javaVersion);
        Preconditions.checkState(Files.isDirectory(javaHome), "JAVA_HOME %s is not a directory for specified java version %s", javaHome, javaVersion);

        Path javaExecutable = javaHome.resolve("bin/java");
        Preconditions.checkState(Files.exists(javaExecutable), "java executable %s does not exist for specified java version %s", javaHome, javaVersion);
        Preconditions.checkState(!Files.isDirectory(javaExecutable), "java executable %s is not a file for specified java version %s", javaHome, javaVersion);

        Set<PosixFilePermission> posixFilePermissions = Files.getPosixFilePermissions(javaExecutable);
        if (!posixFilePermissions.containsAll(Arrays.asList(PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OWNER_EXECUTE))) {
            logger.warn("Missing permissions for {} ({}), add read & execute", javaExecutable, posixFilePermissions);
            Files2.chmodAddReadExecute(javaExecutable);
        }

        return new JavaPluginResult(javaHome, javaExecutable);
    }

    @Nonnull
    protected Path installJvm(@Nonnull String jvmPackageUrl, @Nonnull Environment environment) throws IOException {

        Path tmpFile = Files.createTempFile("jvm", ".tmp");
        try {
            Path jvmFolder = environment.genappDir.resolve("jvm");
            HttpUtils.get(new URL(jvmPackageUrl), tmpFile);
            logger.debug("Install JVM {} in {}", jvmPackageUrl, jvmFolder);
            Files.createDirectories(jvmFolder);

            String lowercaseUrl = jvmPackageUrl.toLowerCase();
            if (lowercaseUrl.endsWith(".zip")) {
                Files2.unzip(tmpFile, jvmFolder);
            } else if (lowercaseUrl.endsWith(".tgz") || lowercaseUrl.endsWith(".tar.gz")) {
                Files2.untgz(tmpFile, jvmFolder);
            } else {
                throw new IllegalStateException("Unsupported JVM url, expected suffix '.zip', '.tgz' or '.tar.gz'");
            }

            if (Files.exists(jvmFolder.resolve("bin/java"))) {
                return jvmFolder;
            }

            Path childDir;
            try {
                childDir = Files2.findUniqueChildDirectory(jvmFolder);

            } catch (RuntimeException e) {
                throw new IllegalStateException("JAVA_HOME with JAVA_HOME/bin/java not found in " + jvmPackageUrl, e);
            }
            if (Files.exists(childDir.resolve("bin/java"))) {
                return childDir;
            }
            throw new IllegalStateException("JAVA_HOME with JAVA_HOME/bin/java not found in dir " + childDir.getFileName() + " of " + jvmPackageUrl);
        } finally {
            Files.delete(tmpFile);
        }
    }

    /**
     * Expand variables in string as <tt>$variable</tt> or <tt>${variable}</tt>/ Invalid variables are left as-is.
     * <p/>
     * Adapted from hudson.Util#replaceMacro
     */
    public String expand(@Nullable String s, @Nonnull Environment env) {
        if (s == null) {
            return null;
        }

        int idx = 0;
        while (true) {
            Matcher m = VARIABLE.matcher(s);
            if (!m.find(idx)) return s;
            String key = m.group().substring(1);

            // escape the dollar sign or get the key to resolve
            String value;
            if (key.charAt(0) == '$') {
                value = "$";
            } else {
                if (key.charAt(0) == '{') key = key.substring(1, key.length() - 1);
                value = resolve(key, env);
            }

            if (value == null)
                idx = m.end(); // skip this
            else {
                s = s.substring(0, m.start()) + value + s.substring(m.end());
                idx = m.start() + value.length();
            }
        }
    }

    private String resolve(String key, @Nonnull Environment env) {
        switch (key) {
            case "app_dir":
                return env.appDir.toAbsolutePath().toString();
            case "plugin_dir":
                return env.clickstackDir.toAbsolutePath().toString();
            case "pkg_dir":
                return env.packageDir.toAbsolutePath().toString();
            case "app_user":
                return env.appUser;
            case "app_id":
                return env.appId;
            case "app_port":
                return String.valueOf(env.appPort);
            case "control_dir":
                return env.controlDir.toAbsolutePath().toString();
            case "log_dir":
                return env.logDir.toAbsolutePath().toString();
            case "genapp_dir":
                return env.genappDir.toAbsolutePath().toString();
            case "app-extra-files":
                return env.appExtraFiles.toAbsolutePath().toString();
            default:
                return key;
        }
    }
}
