package com.eblly.mybatis;

import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeSpec;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import org.apache.ibatis.io.Resources;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.lang.model.element.Modifier;
import java.io.*;
import java.sql.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * mybatis代码生成器
 * 如果有报错，需要将  dao 和 mapper文件夹删除
 *
 * @author eblly
 * @since 2017/12/3
 */
public class MybatisGenerateKit {

    private static final Logger log = LoggerFactory.getLogger(MybatisGenerateKit.class);

    /**
     * @param config          配置文件
     * @param tablePre        需要生成代码表的前缀
     * @param mapperPackage
     * @param modelPackage
     * @param mapperTargt
     * @param modelTarget
     * @param mapperXmlTarget
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public void build(String config, String tablePre, String mapperPackage, String modelPackage, String mapperTargt, String modelTarget, String mapperXmlTarget) throws Exception {

        InputStream stream = Resources.getResourceAsStream(config);

        //获取配置文件
        Properties properties = new Properties();
        properties.load(stream);

        String jdbcDriver = properties.getProperty("jdbc.driver");
        String jdbcUrl = properties.getProperty("jdbc.url");
        String jdbcUserName = properties.getProperty("jdbc.username");
        String jdbcPassword = properties.getProperty("jdbc.password");

        //获取table schema
        Pattern p = Pattern.compile("/\\w+\\?");
        Matcher matcher = p.matcher(jdbcUrl);
        String tableSchema = null;

        if (matcher.find()) {
            tableSchema = matcher.group(0);
        } else {
            throw new Exception("未匹配到数据库");
        }

        //去除第一个和最后一个字符，即 "/" 和 "?"
        tableSchema = tableSchema.substring(1, tableSchema.length() - 1);

        /** connect mysql **/
        Connection conn = null;
        PreparedStatement stmt = null;

        List<Table> tableList = new ArrayList<>();

        try {
            Class.forName(jdbcDriver);
            conn = DriverManager.getConnection(jdbcUrl, jdbcUserName, jdbcPassword);

            /** 获取表 **/
            String tablesSql = " SELECT * FROM information_schema.TABLES WHERE TABLE_SCHEMA=? AND TABLE_NAME LIKE ?";

            stmt = conn.prepareStatement(tablesSql);
            stmt.setString(1, tableSchema);
            stmt.setString(2, tablePre + "%");

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {
                String tableName = rs.getString("TABLE_NAME");
                String tableComment = rs.getString("TABLE_COMMENT");

                Table table = new Table();
                table.setName(tableName);
                table.setComment(tableComment);

                tableList.add(table);
            }

            //循环
            for (Table table : tableList) {
                String sql = " SELECT * FROM information_schema.COLUMNS WHERE TABLE_SCHEMA=? AND TABLE_NAME=? order by ORDINAL_POSITION";

                stmt = conn.prepareStatement(sql);
                stmt.setString(1, tableSchema);
                stmt.setString(2, table.getName());

                rs = stmt.executeQuery();
                ResultSetMetaData rsmd = rs.getMetaData();

                List<Column> columnList = new ArrayList<>();

                while (rs.next()) {
                    String columnName = rs.getString("column_name");
                    String type = rs.getString("data_type");
                    String comment = rs.getString("column_comment");

                    Column column = new Column();
                    column.setName(columnName);
                    column.setType(type);
                    column.setComment(comment);

                    columnList.add(column);
                }

                table.setColumnList(columnList);
            }

            rs.close();
        } catch (Exception e) {
            log.error("==>error", e);
        } finally {
            if (stmt != null)
                stmt.close();
            if (conn != null)
                conn.close();
        }

        this.generate(tableList, mapperPackage, modelPackage, mapperTargt, modelTarget, mapperXmlTarget);

    }

    /**
     * @param tableList
     * @param mapperPackage
     * @param modelPackage
     * @param mapperXmlTarget
     */
    private void generate(List<Table> tableList, String mapperPackage, String modelPackage, String mapperTargt, String
            modelTarget, String mapperXmlTarget) throws Exception {

        String javaMysqlProperties = "javaMysqlType.properties";
        InputStream stream = Resources.getResourceAsStream(javaMysqlProperties);

        //获取配置文件
        Properties properties = new Properties();
        properties.load(stream);

        tableList.forEach(table -> {

            String tableName = table.getName();                  // goods_order
            String modelName = removeSplitChar(tableName);       // goodsOrder
            String modelNameFU = firstCharUpCase(modelName);     // GoodsOrder

            List<FieldSpec> fieldSpecList = new ArrayList<>();
            List<MethodSpec> methodSpecList = new ArrayList<>();

            Map<String, String> resultMap = new LinkedHashMap<>();


            table.getColumnList().forEach(column -> {
                String columnName = column.getName();            // goods_catalog_id
                String fieldName = removeSplitChar(columnName);  // goodsCatalogId
                String fieldNameFU = firstCharUpCase(fieldName); // GoodsCatalogId


                if (!"id".equals(columnName)) {
                    resultMap.put(columnName, fieldName);
                }

                String type = column.getType();
                String comment = column.getComment();
                if (comment == null || "".equals(comment)) {
                    comment = "no comment";
                }

                Class fieldType = null;
                try {
                    fieldType = Class.forName(properties.getProperty(type.toLowerCase()));
                } catch (NullPointerException e) {
                    log.error("==>{} 未找到该properties", type, e);
                } catch (ClassNotFoundException e) {
                    log.error("==>error", e);
                }

                //=================model
                FieldSpec field = FieldSpec.builder(fieldType, fieldName)
                                           .addModifiers(Modifier.PRIVATE)
                                           .addJavadoc(comment + "\n")
                                           .build();
                fieldSpecList.add(field);

                MethodSpec getter = MethodSpec.methodBuilder("get" + fieldNameFU)
                                              .addModifiers(Modifier.PUBLIC)
                                              .returns(fieldType)
                                              .addStatement("return " + fieldName)
                                              .build();
                methodSpecList.add(getter);

                MethodSpec setter = MethodSpec.methodBuilder("set" + fieldNameFU)
                                              .addModifiers(Modifier.PUBLIC)
                                              .addParameter(fieldType, fieldName)
                                              .addStatement("this.$N = $N", fieldName, fieldName)
                                              .build();
                methodSpecList.add(setter);

            });

            String basePath = new File("src/main/java").getPath();

            Configuration configuration = new Configuration();
            configuration.setDefaultEncoding("UTF-8");
            configuration.setClassForTemplateLoading(this.getClass(), "/ftl");  //FTL文件所存在的位置

            //生成model
            generateModel(table, modelNameFU, modelPackage, fieldSpecList, methodSpecList, basePath);


            //生成mapper接口
            try {
                generateMapperInterface(table, modelName, modelNameFU, modelPackage, mapperPackage, basePath,
                        configuration);
            } catch (IOException e) {
                e.printStackTrace();
            } catch (TemplateException e) {
                e.printStackTrace();
            }

            //生成mapperXml
            try {
                generateMapperXml(table, modelNameFU, modelPackage, mapperPackage, resultMap, configuration);
            } catch (Exception e) {
                e.printStackTrace();
            }

        });


    }


    /**
     * 生成model
     */
    private void generateModel(Table table, String modelNameFU, String modelPackage, List<FieldSpec> fieldSpecList, List<MethodSpec> methodSpecList, String basePath) {

        String javadoc = table.getComment();
        if (javadoc == null || "".equals(javadoc)) {
            javadoc = "no comment";
        }

        TypeSpec typeSpec = TypeSpec.classBuilder(modelNameFU)
                                    .addModifiers(Modifier.PUBLIC)
                                    .addFields(fieldSpecList)
                                    .addMethods(methodSpecList)
                                    .addJavadoc(javadoc + "\n")
                                    .build();

        JavaFile model = JavaFile.builder(modelPackage, typeSpec)
                                 .build();

        try {
            //生成文件
            model.writeTo(new File(basePath));
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    /**
     * 生成mapper接口
     *
     * @param table
     * @param modelName
     * @param modelNameFU
     * @param modelPackage
     * @param mapperPackage
     * @param basePath
     * @param configuration
     * @throws IOException
     * @throws TemplateException
     */
    private void generateMapperInterface(Table table, String modelName, String modelNameFU, String modelPackage, String
            mapperPackage, String basePath, Configuration configuration) throws IOException, TemplateException {

//        String mapperInterfacePath = basePath + File.separator + mapperPackage.replaceAll("\\.", File.separator);
        String mapperInterfacePath = basePath + "/" + mapperPackage.replace("\\.", File.separator);
        log.debug("==>modelMapperPath: {}", mapperInterfacePath);

        File mapperInterfaceDir = new File(mapperInterfacePath);
        File mapperInterface = new File(mapperInterfacePath + File.separator + modelNameFU + "Mapper.java");

        //判断路径是否存在
        if (!mapperInterfaceDir.exists()) {
            mapperInterfaceDir.mkdirs();
            log.debug("==>create modelMapperDir");
        }

        if (!mapperInterface.exists()) {
            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("modelPackage", mapperPackage);
            dataMap.put("model", modelPackage + "." + modelNameFU);
            dataMap.put("modelNameFU", modelNameFU);
            dataMap.put("modelName", modelName);
            dataMap.put("comment", table.getComment());

            Template t = configuration.getTemplate("Mapper.ftl"); //文件名

            Writer out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(mapperInterface)));
            t.process(dataMap, out);
        }

    }

    /**
     * 生成mapper xml 文件
     *
     * @param table
     * @param modelNameFU
     * @param modelPackage
     * @param mapperPackage
     * @param resultMap
     * @param configuration
     * @throws Exception
     */
    private void generateMapperXml(Table table, String modelNameFU, String modelPackage, String
            mapperPackage, Map<String, String> resultMap, Configuration configuration) throws Exception {

        String resultMapXMLPath = "src/main/resources/mapper/resultMap";
        String mapperXMLPath = "src/main/resources/mapper";

        //==========生成xxxResultMap.xml
        Map<String, Object> map2 = new HashMap<>();
        map2.put("resultMap", resultMap);
        map2.put("resultMapModel", modelPackage + "." + modelNameFU);
        map2.put("namespaceResultMap", mapperPackage + ".resultMap." + modelNameFU + "ResultMap");

        Template resutlMapFtl = configuration.getTemplate("ResultMapXML.ftl"); //文件名

        File mapperResources = new File(resultMapXMLPath);
        if (!mapperResources.exists()) {
            mapperResources.mkdirs();
            log.debug("==>create mapperResources:{}", mapperResources);
        }

        File resultMapXMLFile = new File(resultMapXMLPath + File.separator + modelNameFU + "ResultMap.xml");  //生成文件

        //不管resultMap 是否存在，都重新覆盖
        Writer out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(resultMapXMLFile)));
        resutlMapFtl.process(map2, out);


        //==========生成xxxMapper.xml
        Map<String, Object> dataMap = new HashMap<>();
        dataMap.put("namespace", mapperPackage + "." + modelNameFU + "Mapper");
        dataMap.put("selectResultMap", mapperPackage + ".resultMap." + modelNameFU + "ResultMap" + ".resultMap");
        dataMap.put("table", table.getName());
        dataMap.put("resultMap", resultMap);
        dataMap.put("parameterType", modelNameFU);

        Template mapperFtl = configuration.getTemplate("MapperXML.ftl"); //文件名

        File mapperXMLFile = new File(mapperXMLPath + File.separator + modelNameFU + "Mapper.xml");  //生成文件的路径

        if (!mapperXMLFile.exists()) {
            Writer out2 = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(mapperXMLFile)));
            mapperFtl.process(dataMap, out2);
        }

    }


    /**
     * 去除分割符"_" , 驼峰命名
     * 例如: user_name -> useName
     */
    public static String removeSplitChar(String str) {
        String[] str2 = str.split("_");

        String removeStr = str2[0];
        for (int i = 1; i < str2.length; i++) {
            String str3 = str2[i].substring(0, 1).toUpperCase() + str2[i].substring(1);
            removeStr += str3;
        }
        return removeStr;
    }

    /**
     * 首写字母大写
     * userName -> UserName
     * @return
     */
    public static String firstCharUpCase(String word) {
        return word.substring(0, 1).toUpperCase() + word.substring(1);
    }

}
