package com.github.azbh111.utils.java.io.file;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.StringJoiner;

/**
 * @author pyz
 * @date 2018/8/11 上午11:50
 */
public class FileUtils {

    public static String getFileNameSuffix(String path) {
        if (path == null || path.isEmpty()) {
            return path;
        }
//        必须先提取名字,不然目录有.但是文件名没有.就不对了
        path = getFileName(path);
        int dot = path.lastIndexOf('.');
        if (dot >= 0) {
            return path.substring(dot + 1);
        }
//        没有.
        return "";
    }

    public static String getFileName(String path) {
        if (path == null || path.isEmpty()) {
            return path;
        }
        int dot = path.lastIndexOf('/');
        if (dot < 0) {
            dot = path.lastIndexOf('\\');
        }
        if (dot < 0) {
            dot = path.lastIndexOf(':');
        }
        if (dot < 0) {
            return path;
        }
        path = path.substring(dot + 1);
        return path;
    }

    public static String readString(Path path) throws IOException {
        return readString(path.toFile(), StandardCharsets.UTF_8);
    }

    public static String readString(Path file, Charset charset) throws IOException {
        return readString(file, charset);
    }

    public static String readString(String path) throws IOException {
        return readString(new File(path), StandardCharsets.UTF_8);
    }

    public static String readString(String file, Charset charset) throws IOException {
        return readString(new File(file), charset);
    }

    public static String readString(File file) throws IOException {
        return readString(file, StandardCharsets.UTF_8);
    }

    public static String readString(File file, Charset charset) throws IOException {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), charset))) {
            StringJoiner sj = new StringJoiner(System.lineSeparator());
            String s;
            while ((s = reader.readLine()) != null) {
                sj.add(s);
            }
            return sj.toString();
        }
    }

    public static void writeString(File file, String content) throws IOException {
        writeString(file, content, StandardCharsets.UTF_8);
    }

    public static void writeString(File file, String content, Charset charset) throws IOException {
        createFileIfNotExist(file);
        Files.write(file.toPath(), Collections.singleton(content), charset);

    }

    public static byte[] readBytes(File file) throws IOException {
        return Files.readAllBytes(file.toPath());
    }

    public static void writeBytes(File file, byte[] content) throws IOException {
        createFileIfNotExist(file);
        Files.write(file.toPath(), content);
    }

    public static File getOrCreateFile(String file) throws IOException {
        File f = new File(file);
        createFileIfNotExist(f);
        return f;
    }

    public static void createFileIfNotExist(File file) throws IOException {
        if (!file.exists()) {
            File p = file.getParentFile();
            createDirIfNotExist(p);
            file.createNewFile();
        }
    }

    public static File getOrCreateDir(String dir) {
        File f = new File(dir);
        createDirIfNotExist(f);
        return f;
    }

    public static void createDirIfNotExist(File file) {
        if (!file.exists()) {
            file.mkdirs();
        }
    }

    public static void createDirByFile(File file) {
        if (!file.exists()) {
            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs();
            }
        }
    }

    public static void createDirByFile(Path file) {
        createDirByFile(file.toFile());
    }

}
