package com.github.azbh111.utils.java.reflect;

import com.github.azbh111.utils.java.exception.ExceptionUtils;
import com.github.azbh111.utils.java.exception.UnreachableException;
import com.github.azbh111.utils.java.unsafe.UnsafeUtils;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * @author pyz
 * @date 2019/4/22 9:29 AM
 */
public class ClassUtils {
    /**
     * Maps primitive {@code Class}es to their corresponding wrapper {@code Class}.
     */
    private static final Map<Class<?>, Class<?>> primitiveWrapperMap;

    static {
        Map<Class<?>, Class<?>> _primitiveWrapperMap = new HashMap<>();
        _primitiveWrapperMap.put(Boolean.TYPE, Boolean.class);
        _primitiveWrapperMap.put(Byte.TYPE, Byte.class);
        _primitiveWrapperMap.put(Character.TYPE, Character.class);
        _primitiveWrapperMap.put(Short.TYPE, Short.class);
        _primitiveWrapperMap.put(Integer.TYPE, Integer.class);
        _primitiveWrapperMap.put(Long.TYPE, Long.class);
        _primitiveWrapperMap.put(Double.TYPE, Double.class);
        _primitiveWrapperMap.put(Float.TYPE, Float.class);
        _primitiveWrapperMap.put(Void.TYPE, Void.TYPE);
        primitiveWrapperMap = Collections.unmodifiableMap(_primitiveWrapperMap);
    }

    private static final Map<Class<?>, Class<?>> wrapperPrimitiveMap;

    static {
        Map<Class<?>, Class<?>> _wrapperPrimitiveMap = new HashMap<>();
        for (final Map.Entry<Class<?>, Class<?>> entry : primitiveWrapperMap.entrySet()) {
            final Class<?> primitiveClass = entry.getKey();
            final Class<?> wrapperClass = entry.getValue();
            if (!primitiveClass.equals(wrapperClass)) {
                _wrapperPrimitiveMap.put(wrapperClass, primitiveClass);
            }
        }
        wrapperPrimitiveMap = Collections.unmodifiableMap(_wrapperPrimitiveMap);

    }

    /**
     * Maps a primitive class name to its corresponding abbreviation used in array class names.
     */
    private static final Map<String, String> abbreviationMap;

    /**
     * Maps an abbreviation used in array class names to corresponding primitive class name.
     */
    private static final Map<String, String> reverseAbbreviationMap;

    static {
        final Map<String, String> m = new HashMap<>();
        m.put("int", "I");
        m.put("boolean", "Z");
        m.put("float", "F");
        m.put("long", "J");
        m.put("short", "S");
        m.put("byte", "B");
        m.put("double", "D");
        m.put("char", "C");
        final Map<String, String> r = new HashMap<>();
        for (final Map.Entry<String, String> e : m.entrySet()) {
            r.put(e.getValue(), e.getKey());
        }
        abbreviationMap = Collections.unmodifiableMap(m);
        reverseAbbreviationMap = Collections.unmodifiableMap(r);
    }

    /**
     * 返回包装类型对应的基本数据类型
     * 若cls不是包装类型,返回null
     *
     * @param cls
     * @return
     */
    public static Class<?> unwrapPrimitive(final Class<?> cls) {
        return wrapperPrimitiveMap.get(cls);
    }

    /**
     * 返回基本数据类型的包装类型
     * 若cls不是基本数据类型,返回null
     *
     * @param cls
     * @return
     */
    public static Class<?> wrapPrimitive(final Class<?> cls) {
        return primitiveWrapperMap.get(cls);
    }

    public static boolean isClassPresent(String className) {
        try {
            Class.forName(className);
            return true;
        } catch (ClassNotFoundException e) {
            return false;
        }
    }

    public static <T> T newInstance(Class<T> type) {
        try {
            Constructor<T> c = type.getDeclaredConstructor();
            c.setAccessible(true);
            return c.newInstance();
        } catch (NoSuchMethodException | InstantiationException | IllegalAccessException | InvocationTargetException e) {
            // ignore
        }
        try {
            return UnsafeUtils.allocateInstance(type);
        } catch (InstantiationException e) {
            ExceptionUtils.throwException(e);
        }
        throw new UnreachableException();
    }
}
