/*
 * Copyright (c) 2014-2019, Deepspring Healthcare Inc. All rights reserved.
 */

package com.github.azbh111.utils.java.web.model;

import com.github.azbh111.utils.java.model.Res;
import com.github.azbh111.utils.java.model.ResStatusEnum;
import lombok.ToString;

import java.io.Serializable;


@ToString
public class JsonResponse<T> implements Serializable {

    private static final long serialVersionUID = -8439068266525433175L;
    protected int status = ResStatusEnum.OK.getStatus();
    /**
     * 错误提示信息
     */
    protected String errorMsg;
    /**
     * 数据信息
     */
    protected T data;

    protected JsonResponse() {
    }

    public static <T> JsonResponse<T> from(Res<T> res) {
        JsonResponse<T> jr = JsonResponse.of(res.getStatus().getStatus(), res.getData());
        if (res.getErrorMsg() != null) {
            jr.setErrorMsg(res.getErrorMsg());
        } else if (res.getStatus().getDisplayMsg() != null) {
            jr.setErrorMsg(res.getStatus().getDisplayMsg());
        } else {
            jr.setErrorMsg(res.getStatus().getDesc());
        }
        return jr;
    }

    public static JsonResponse success() {
        return new JsonResponse();
    }

    public static <T> JsonResponse<T> success(T data) {
        return of(ResStatusEnum.OK.getStatus(), data, null);
    }

    public static JsonResponse fail(String errorMsg) {
        return of(ResStatusEnum.BAD_REQUEST.getStatus(), null, errorMsg);
    }

    public static JsonResponse of(int status, String errorMsg) {
        return of(status, null, errorMsg);
    }

    public static <T> JsonResponse<T> of(int status, T data) {
        return of(status, data, null);
    }

    public static <T> JsonResponse<T> of(int status, T data, String errorMsg) {
        return JsonResponse.<T>success()
                .setStatus(status)
                .setData(data)
                .setErrorMsg(errorMsg);
    }

    public JsonResponse<T> apply(Res<T> res) {
        this.setStatus(res.getStatus().getStatus());
        this.setData(res.getData());
        if (res.getErrorMsg() != null) {
            this.setErrorMsg(res.getErrorMsg());
        } else if (res.getStatus().getDisplayMsg() != null) {
            this.setErrorMsg(res.getStatus().getDisplayMsg());
        } else {
            this.setErrorMsg(res.getStatus().getDesc());
        }
        return this;
    }

    public int getStatus() {
        return status;
    }

    public JsonResponse<T> setStatus(int status) {
        this.status = status;
        return this;
    }

    public String getErrorMsg() {
        return errorMsg;
    }

    public JsonResponse<T> setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public T getData() {
        return data;
    }

    public JsonResponse<T> setData(T data) {
        this.data = data;
        return this;
    }

}