/*
 * Copyright (c) 2014-2019, Deepspring Healthcare Inc. All rights reserved.
 */

package com.github.azbh111.utils.java.map;


import com.github.azbh111.utils.java.annotation.Nonnull;
import com.github.azbh111.utils.java.annotation.Nullable;
import com.github.azbh111.utils.java.iterable.IterableUtils;

import java.util.*;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.BinaryOperator;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @author pyz
 * @date 2018/8/27 下午6:29
 */
public class MapUtils {


    /**
     * 遍历src,若target中不含遍历的key,则把对应的kv放入target中
     *
     * @param src
     * @param target
     * @param <K>
     * @param <V>
     */
    public static <K, V> void putAllIfAbsent(Map<K, V> src, Map<K, V> target) {
        src.forEach((k, v) -> {
            target.putIfAbsent(k, v);
        });
    }

    /**
     * 遍历src,若target中不含遍历的key,则使用generator生成v,并把kv放入target中
     *
     * @param src
     * @param target
     * @param generator
     * @param <K>
     * @param <V>
     * @param <T>
     */
    public static <K, V, T> void putOrCreateAllIfAbsent(Map<K, T> src, Map<K, V> target, BiFunction<K, T, V> generator) {
        src.forEach((k, v) -> {
            target.computeIfAbsent(k, key -> generator.apply(k, v));
        });
    }


    /**
     * 按给定映射,将map分成多个map
     *
     * @param map
     * @param groupMapper
     * @param <K>
     * @param <V>
     * @param <GroupKey>
     * @return
     */
    public static <K, V, GroupKey> Map<GroupKey, Map<K, V>> groupBy(Map<K, V> map, BiFunction<K, V, GroupKey> groupMapper) {
        Map<GroupKey, Map<K, V>> resultMap = new HashMap<>();
        map.forEach((k, v) -> {
            GroupKey key = groupMapper.apply(k, v);
            MapUtils.getOrCreate(resultMap, key, HashMap::new)
                    .put(k, v);
        });
        return resultMap;
    }

    public static <K, V, T> List<T> toList(Map<K, V> map, BiFunction<K, V, T> mapper) {
        return map.entrySet()
                .stream()
                .map(entry -> mapper.apply(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }

    public static <K, V> Map<K, V> extract(Map<K, V> map, Iterable<K> keys) {
        return IterableUtils.toStream(keys)
                .filter(map::containsKey)
                .collect(Collectors.toMap(k -> k, k -> map.get(k)));
    }

    /**
     * 用指定keys从map中提取value,返回values
     *
     * @param map
     * @param keys
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> List<V> extractValueList(Map<K, V> map, Iterable<K> keys) {
        return IterableUtils.toStream(keys)
                .filter(map::containsKey)
                .map(map::get)
                .collect(Collectors.toList());
    }

    /**
     * 用指定keys从map中提取value,返回values
     *
     * @param map
     * @param keys
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> Set<V> extractValueSet(Map<K, V> map, Iterable<K> keys) {
        return IterableUtils.toStream(keys)
                .filter(map::containsKey)
                .map(map::get)
                .collect(Collectors.toSet());
    }

    public static <K, V> V getOrCreate(Map<K, V> map, K key, Supplier<V> valueCreator) {
        V value = map.get(key);
        if (value == null) {
            value = valueCreator.get();
            map.put(key, value);
        }
        return value;
    }

    public static <K, V, T> Map<T, V> computeKey(Map<K, V> src, BiFunction<K, V, T> keyMapper) {
        return compute(src, keyMapper, (k, v) -> v);
    }

    public static <K, V, R> Map<K, R> computeValue(Map<K, V> src, BiFunction<K, V, R> valueMapper) {
        return compute(src, (k, v) -> k, valueMapper);
    }

    public static <K, V, T, R> Map<T, R> compute(Map<K, V> src, BiFunction<K, V, T> keyMapper, BiFunction<K, V, R> valueMapper) {
        Map<T, R> result = new HashMap<>();
        if (src == null) {
            return result;
        }
        src.forEach((k, v) -> {
            R replace = result.put(keyMapper.apply(k, v), valueMapper.apply(k, v));
            if (replace != null) {
                // 计算后的key不能重复
                throw new IllegalStateException(String.format("Duplicate key %s", k));
            }
        });
        return result;
    }

    public static <K, V> void removeIf(Map<K, V> map, BiPredicate<K, V> test) {
        Iterator<Map.Entry<K, V>> it = map.entrySet().iterator();
        Map.Entry<K, V> en;
        while (it.hasNext()) {
            en = it.next();
            if (test.test(en.getKey(), en.getValue())) {
                it.remove();
            }
        }
    }

    public static <K, V> Map<K, V> filter(Map<K, V> map, BiPredicate<K, V> test) {
        Map<K, V> result = new HashMap<>();
        map.forEach((k, v) -> {
            if (test.test(k, v)) {
                result.put(k, v);
            }
        });
        return result;
    }

    /**
     * 取一个value
     *
     * @param statusMap
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> V anyValue(Map<K, V> statusMap) {
        Map.Entry<K, V> entry = anyEntry(statusMap);
        return entry == null ? null : entry.getValue();
    }

    /**
     * 取一个key
     *
     * @param statusMap
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> K anyKey(Map<K, V> statusMap) {
        Map.Entry<K, V> entry = anyEntry(statusMap);
        return entry == null ? null : entry.getKey();
    }

    public static <K, V> Map.Entry<K, V> anyEntry(Map<K, V> statusMap) {
        if (statusMap == null) {
            return null;
        }
        Iterator<Map.Entry<K, V>> it = statusMap.entrySet().iterator();
        if (!it.hasNext()) {
            return null;
        }
        return it.next();
    }

    public static <K, V> boolean isNullOrEmpty(Map<K, V> map) {
        return map == null || map.size() == 0;
    }

    /**
     * 翻转map 如果翻转后key有冲突，会抛异常
     *
     * @param map
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> Map<V, K> reverse(Map<K, V> map) {
        return reverse(map, null);
    }

    /**
     * 翻转map
     *
     * @param map
     * @param mergeFunction 如果不传，且key有冲突，会抛异常
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> Map<V, K> reverse(@Nonnull Map<K, V> map, @Nullable BinaryOperator<K> mergeFunction) {
        Map<V, K> reverseMap = new HashMap<>();
        for (Map.Entry<K, V> kvEntry : map.entrySet()) {
            K replaceK = reverseMap.put(kvEntry.getValue(), kvEntry.getKey());
            if (replaceK == null) continue;
            if (mergeFunction == null) throw new IllegalStateException(String.format("Duplicate key %s", kvEntry.getValue()));
            K finalK = mergeFunction.apply(replaceK, kvEntry.getKey());
            if (finalK != replaceK) reverseMap.put(kvEntry.getValue(), finalK);
        }
        return reverseMap;
    }
}