package com.github.azbh111.utils.java.rsa;

import com.github.azbh111.utils.java.annotation.Nonnull;
import com.github.azbh111.utils.java.annotation.Nullable;
import com.github.azbh111.utils.java.code.HexUtils;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;

/**
 * @author: zyp
 * @date: 2020/11/13 16:08
 */
public class RSAPair {
    private static final String KEY_ALGORITHM = "RSA";

    private final RSASignMethodEnum signMethod;
    private final byte[] publicKeyBytes;
    private final byte[] privateKeyBytes;
    private final PublicKey publicKey;
    private final PrivateKey privateKey;

    private RSAPair(RSASignMethodEnum signMethod, PublicKey publicKey, PrivateKey privateKey) {
        this.signMethod = signMethod;
        this.publicKey = publicKey;
        this.privateKey = privateKey;
        publicKeyBytes = publicKey.getEncoded();
        privateKeyBytes = privateKey.getEncoded();
    }

    /**
     * 为null时无法进行签名
     *
     * @param signMethod
     * @return
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, int keySize) throws NoSuchAlgorithmException {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(KEY_ALGORITHM);
        keyPairGenerator.initialize(keySize);
        KeyPair keyPair = keyPairGenerator.generateKeyPair();
        RSAPublicKey rsaPublicKey = (RSAPublicKey) keyPair.getPublic();
        RSAPrivateKey rsaPrivateKey = (RSAPrivateKey) keyPair.getPrivate();
        return new RSAPair(signMethod, rsaPublicKey, rsaPrivateKey);
    }

    /**
     * 为null时无法进行签名
     *
     * @param signMethod
     * @param publicKeyHex
     * @param privateKeyHex
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, @Nonnull String publicKeyHex, @Nonnull String privateKeyHex) throws InvalidKeySpecException, NoSuchAlgorithmException {
        return new RSAPair(signMethod, RSAUtils.createPublicKey(HexUtils.hexDecode(publicKeyHex)), RSAUtils.createPrivateKey(HexUtils.hexDecode(privateKeyHex)));
    }

    /**
     * 为null时无法进行签名
     *
     * @param signMethod
     * @param publicKeyHex
     * @param privateKeyHex
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, @Nonnull byte[] publicKeyBytes, @Nonnull byte[] privateKeyBytes) throws InvalidKeySpecException, NoSuchAlgorithmException {
        return new RSAPair(signMethod, RSAUtils.createPublicKey(publicKeyBytes), RSAUtils.createPrivateKey(privateKeyBytes));
    }

    /**
     * @param signMethod 为null时，无法进行签名
     * @param publicKey
     * @param privateKey
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    public static RSAPair create(@Nullable RSASignMethodEnum signMethod, @Nonnull PublicKey publicKey, @Nonnull PrivateKey privateKey) {
        return new RSAPair(signMethod, publicKey, privateKey);
    }

    public byte[] getPublicKeyBytes() {
        return publicKeyBytes;
    }

    public byte[] getPrivateKeyBytes() {
        return privateKeyBytes;
    }

    public PublicKey getPublicKey() {
        return publicKey;
    }

    public PrivateKey getPrivateKey() {
        return privateKey;
    }

    /**
     * 公钥加密
     *
     * @param data
     * @return
     */
    public byte[] publicEncrypt(byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return RSAUtils.encrypt(publicKey, data);
    }

    /**
     * 私钥加密
     *
     * @param data
     * @return
     */
    public byte[] privateEncrypt(byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return RSAUtils.encrypt(privateKey, data);
    }

    /**
     * 公钥解密
     *
     * @param data
     * @return
     */
    public byte[] publicDencrypt(byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return RSAUtils.dencrypt(publicKey, data);
    }

    /**
     * 私钥解密
     *
     * @param data
     * @return
     */
    public byte[] privateDencrypt(byte[] data) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return RSAUtils.dencrypt(privateKey, data);
    }

    /**
     * 私钥签名
     *
     * @param data
     * @return
     */
    public byte[] sign(byte[] data) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
        if (signMethod == null) {
            throw new NoSuchAlgorithmException("signMethod not set");
        }
        return RSAUtils.sign(signMethod, privateKey, data);
    }

    /**
     * 公钥验证签名
     *
     * @param data 数据
     * @param sign 签名
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws SignatureException
     */
    public boolean verifySign(byte[] data, byte[] sign) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
        return RSAUtils.verifySign(signMethod, publicKey, data, sign);
    }

}
