package com.github.azbh111.utils.java.jvm;

import javax.management.openmbean.TabularData;
import java.lang.management.*;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 *
 * @author pyz
 * @date 2019/6/11 3:50 PM
 */
public class JVM {
    private static JVM instance = new JVM();

    private JVM() {
    }

    public static JVM getInstance() {
        return instance;
    }

    private final RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
    private final ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();
    private final ClassLoadingMXBean classLoadingMXBean = ManagementFactory.getClassLoadingMXBean();
    private final OperatingSystemMXBean operatingSystemMXBean = ManagementFactory.getOperatingSystemMXBean();
    private final MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
    //    这里会返回老年代，新生代等内存区的使用情况，按需自取就好
    private final List<MemoryPoolMXBean> memoryPoolMXBeans = Collections.unmodifiableList(ManagementFactory.getMemoryPoolMXBeans());
    //    gc
    private final List<GarbageCollectorMXBean> garbageCollectorMXBeans = Collections.unmodifiableList(ManagementFactory.getGarbageCollectorMXBeans());

    public List<GarbageCollectorMXBean> getGarbageCollectorMXBeans() {
        return garbageCollectorMXBeans;
    }

    public List<MemoryPoolMXBean> getMemoryPoolMXBeans() {
        return memoryPoolMXBeans;
    }

    /**
     * Returns the approximate number of objects for which
     * finalization is pending.
     *
     * @return the approximate number objects for which finalization
     * is pending.
     */
    public int getObjectPendingFinalizationCount() {
        return memoryMXBean.getObjectPendingFinalizationCount();
    }

    /**
     * Returns the current memory usage of the heap that
     * is used for object allocation.  The heap consists
     * of one or more memory pools.  The <tt>used</tt>
     * and <tt>committed</tt> size of the returned memory
     * usage is the sum of those values of all heap memory pools
     * whereas the <tt>init</tt> and <tt>max</tt> size of the
     * returned memory usage represents the setting of the heap
     * memory which may not be the sum of those of all heap
     * memory pools.
     * <p>
     * The amount of used memory in the returned memory usage
     * is the amount of memory occupied by both live objects
     * and garbage objects that have not been collected, if any.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>MemoryUsage</tt> is
     * <tt>CompositeData</tt> with attributes as specified in
     * {@link MemoryUsage#from MemoryUsage}.
     *
     * @return a {@link MemoryUsage} object representing
     * the heap memory usage.
     */
    public MemoryUsage getHeapMemoryUsage() {
        return memoryMXBean.getHeapMemoryUsage();
    }

    /**
     * Returns the current memory usage of non-heap memory that
     * is used by the Java virtual machine.
     * The non-heap memory consists of one or more memory pools.
     * The <tt>used</tt> and <tt>committed</tt> size of the
     * returned memory usage is the sum of those values of
     * all non-heap memory pools whereas the <tt>init</tt>
     * and <tt>max</tt> size of the returned memory usage
     * represents the setting of the non-heap
     * memory which may not be the sum of those of all non-heap
     * memory pools.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>MemoryUsage</tt> is
     * <tt>CompositeData</tt> with attributes as specified in
     * {@link MemoryUsage#from MemoryUsage}.
     *
     * @return a {@link MemoryUsage} object representing
     * the non-heap memory usage.
     */
    public MemoryUsage getNonHeapMemoryUsage() {
        return memoryMXBean.getNonHeapMemoryUsage();
    }

    /**
     * Runs the garbage collector.
     * The call <code>gc()</code> is effectively equivalent to the
     * call:
     * <blockquote><pre>
     * System.gc()
     * </pre></blockquote>
     *
     * @see     System#gc()
     */
    public void gc() {
        memoryMXBean.gc();
    }

    /**
     * Returns the operating system architecture.
     * This method is equivalent to <tt>System.getProperty("os.arch")</tt>.
     *
     * @return the operating system architecture.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getArch() {
        return operatingSystemMXBean.getArch();
    }

    /**
     * Returns the operating system version.
     * This method is equivalent to <tt>System.getProperty("os.version")</tt>.
     *
     * @return the operating system version.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getVersion() {
        return operatingSystemMXBean.getVersion();
    }

    /**
     * Returns the number of processors available to the Java virtual machine.
     * This method is equivalent to the {@link Runtime#availableProcessors()}
     * method.
     * <p> This value may change during a particular invocation of
     * the virtual machine.
     *
     * @return the number of processors available to the virtual
     *          machine; never smaller than one.
     */
    public int getAvailableProcessors() {
        return operatingSystemMXBean.getAvailableProcessors();
    }

    /**
     * 返回最后一分钟内系统加载平均值
     *
     * Returns the system load average for the last minute.
     * The system load average is the sum of the number of runnable entities
     * queued to the {@linkplain #getAvailableProcessors available processors}
     * and the number of runnable entities running on the available processors
     * averaged over a period of time.
     * The way in which the load average is calculated is operating system
     * specific but is typically a damped time-dependent average.
     * <p>
     * If the load average is not available, a negative value is returned.
     * <p>
     * This method is designed to provide a hint about the system load
     * and may be queried frequently.
     * The load average may be unavailable on some platform where it is
     * expensive to implement this method.
     *
     * @return the system load average; or a negative value if not available.
     *
     * @since 1.6
     */
    public double getSystemLoadAverage() {
        return operatingSystemMXBean.getSystemLoadAverage();
    }

    /**
     * 返回自 Java 虚拟机开始执行到目前已经加载的类的总数
     *
     * Returns the total number of classes that have been loaded since
     * the Java virtual machine has started execution.
     *
     * @return the total number of classes loaded.
     *
     */
    public long getTotalLoadedClassCount() {
        return classLoadingMXBean.getTotalLoadedClassCount();
    }

    /**
     * 返回当前加载到 Java 虚拟机中的类的数量
     *
     * Returns the number of classes that are currently loaded in the
     * Java virtual machine.
     *
     * @return the number of currently loaded classes.
     */
    public int getLoadedClassCount() {
        return classLoadingMXBean.getLoadedClassCount();
    }

    /**
     * 返回自 Java 虚拟机开始执行到目前已经卸载的类的总数
     *
     * Returns the total number of classes unloaded since the Java virtual machine
     * has started execution.
     *
     * @return the total number of unloaded classes.
     */
    public long getUnloadedClassCount() {
        return classLoadingMXBean.getUnloadedClassCount();
    }

    /**
     * 返回活动线程的当前数目，包括守护线程和非守护线程
     *
     * Returns the current number of live threads including both
     * daemon and non-daemon threads.
     *
     * @return the current number of live threads.
     */
    public int getThreadCount() {
        return threadMXBean.getThreadCount();
    }

    /**
     * 返回自从 Java 虚拟机启动或峰值重置以来峰值活动线程计数
     *
     * Returns the peak live thread count since the Java virtual machine
     * started or peak was reset.
     *
     * @return the peak live thread count.
     */
    public int getPeakThreadCount() {
        return threadMXBean.getPeakThreadCount();
    }

    /**
     *  返回自从 Java 虚拟机启动以来创建和启动的线程总数目
     *
     * Returns the total number of threads created and also started
     * since the Java virtual machine started.
     *
     * @return the total number of threads started.
     */
    public long getTotalStartedThreadCount() {
        return threadMXBean.getTotalStartedThreadCount();
    }

    /**
     * Returns the current number of live daemon threads.
     *
     * @return the current number of live daemon threads.
     */
    public int getDaemonThreadCount() {
        return threadMXBean.getDaemonThreadCount();
    }

    /**
     * Returns all live thread IDs.
     * Some threads included in the returned array
     * may have been terminated when this method returns.
     *
     * @return an array of <tt>long</tt>, each is a thread ID.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     */
    public long[] getAllThreadIds() {
        return threadMXBean.getAllThreadIds();
    }

    /**
     * Returns the thread info for a thread of the specified
     * <tt>id</tt> with no stack trace.
     * This method is equivalent to calling:
     * <blockquote>
     *   {@link #getThreadInfo(long, int) getThreadInfo(id, 0);}
     * </blockquote>
     *
     * <p>
     * This method returns a <tt>ThreadInfo</tt> object representing
     * the thread information for the thread of the specified ID.
     * The stack trace, locked monitors, and locked synchronizers
     * in the returned <tt>ThreadInfo</tt> object will
     * be empty.
     *
     * If a thread of the given ID is not alive or does not exist,
     * this method will return <tt>null</tt>.  A thread is alive if
     * it has been started and has not yet died.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>ThreadInfo</tt> is
     * <tt>CompositeData</tt> with attributes as specified in the
     * {@link ThreadInfo#from ThreadInfo.from} method.
     *
     * @param id the thread ID of the thread. Must be positive.
     *
     * @return a {@link ThreadInfo} object for the thread of the given ID
     * with no stack trace, no locked monitor and no synchronizer info;
     * <tt>null</tt> if the thread of the given ID is not alive or
     * it does not exist.
     *
     * @throws IllegalArgumentException if {@code id <= 0}.
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     */
    public ThreadInfo getThreadInfo(long id) {
        return threadMXBean.getThreadInfo(id);
    }

    /**
     * Returns the thread info for each thread
     * whose ID is in the input array <tt>ids</tt> with no stack trace.
     * This method is equivalent to calling:
     * <blockquote><pre>
     *   {@link #getThreadInfo(long[], int) getThreadInfo}(ids, 0);
     * </pre></blockquote>
     *
     * <p>
     * This method returns an array of the <tt>ThreadInfo</tt> objects.
     * The stack trace, locked monitors, and locked synchronizers
     * in each <tt>ThreadInfo</tt> object will be empty.
     *
     * If a thread of a given ID is not alive or does not exist,
     * the corresponding element in the returned array will
     * contain <tt>null</tt>.  A thread is alive if
     * it has been started and has not yet died.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>ThreadInfo</tt> is
     * <tt>CompositeData</tt> with attributes as specified in the
     * {@link ThreadInfo#from ThreadInfo.from} method.
     *
     * @param ids an array of thread IDs.
     * @return an array of the {@link ThreadInfo} objects, each containing
     * information about a thread whose ID is in the corresponding
     * element of the input array of IDs
     * with no stack trace, no locked monitor and no synchronizer info.
     *
     * @throws IllegalArgumentException if any element in the input array
     *         <tt>ids</tt> is {@code <= 0}.
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     */
    public ThreadInfo[] getThreadInfo(long[] ids) {
        return threadMXBean.getThreadInfo(ids);
    }

    /**
     * Returns a thread info for a thread of the specified <tt>id</tt>,
     * with stack trace of a specified number of stack trace elements.
     * The <tt>maxDepth</tt> parameter indicates the maximum number of
     * {@link StackTraceElement} to be retrieved from the stack trace.
     * If <tt>maxDepth == Integer.MAX_VALUE</tt>, the entire stack trace of
     * the thread will be dumped.
     * If <tt>maxDepth == 0</tt>, no stack trace of the thread
     * will be dumped.
     * This method does not obtain the locked monitors and locked
     * synchronizers of the thread.
     * <p>
     * When the Java virtual machine has no stack trace information
     * about a thread or <tt>maxDepth == 0</tt>,
     * the stack trace in the
     * <tt>ThreadInfo</tt> object will be an empty array of
     * <tt>StackTraceElement</tt>.
     *
     * <p>
     * If a thread of the given ID is not alive or does not exist,
     * this method will return <tt>null</tt>.  A thread is alive if
     * it has been started and has not yet died.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>ThreadInfo</tt> is
     * <tt>CompositeData</tt> with attributes as specified in the
     * {@link ThreadInfo#from ThreadInfo.from} method.
     *
     * @param id the thread ID of the thread. Must be positive.
     * @param maxDepth the maximum number of entries in the stack trace
     * to be dumped. <tt>Integer.MAX_VALUE</tt> could be used to request
     * the entire stack to be dumped.
     *
     * @return a {@link ThreadInfo} of the thread of the given ID
     * with no locked monitor and synchronizer info.
     * <tt>null</tt> if the thread of the given ID is not alive or
     * it does not exist.
     *
     * @throws IllegalArgumentException if {@code id <= 0}.
     * @throws IllegalArgumentException if <tt>maxDepth is negative</tt>.
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     *
     */
    public ThreadInfo getThreadInfo(long id, int maxDepth) {
        return threadMXBean.getThreadInfo(id, maxDepth);
    }

    /**
     * Returns the thread info for each thread
     * whose ID is in the input array <tt>ids</tt>,
     * with stack trace of a specified number of stack trace elements.
     * The <tt>maxDepth</tt> parameter indicates the maximum number of
     * {@link StackTraceElement} to be retrieved from the stack trace.
     * If <tt>maxDepth == Integer.MAX_VALUE</tt>, the entire stack trace of
     * the thread will be dumped.
     * If <tt>maxDepth == 0</tt>, no stack trace of the thread
     * will be dumped.
     * This method does not obtain the locked monitors and locked
     * synchronizers of the threads.
     * <p>
     * When the Java virtual machine has no stack trace information
     * about a thread or <tt>maxDepth == 0</tt>,
     * the stack trace in the
     * <tt>ThreadInfo</tt> object will be an empty array of
     * <tt>StackTraceElement</tt>.
     * <p>
     * This method returns an array of the <tt>ThreadInfo</tt> objects,
     * each is the thread information about the thread with the same index
     * as in the <tt>ids</tt> array.
     * If a thread of the given ID is not alive or does not exist,
     * <tt>null</tt> will be set in the corresponding element
     * in the returned array.  A thread is alive if
     * it has been started and has not yet died.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>ThreadInfo</tt> is
     * <tt>CompositeData</tt> with attributes as specified in the
     * {@link ThreadInfo#from ThreadInfo.from} method.
     *
     * @param ids an array of thread IDs
     * @param maxDepth the maximum number of entries in the stack trace
     * to be dumped. <tt>Integer.MAX_VALUE</tt> could be used to request
     * the entire stack to be dumped.
     *
     * @return an array of the {@link ThreadInfo} objects, each containing
     * information about a thread whose ID is in the corresponding
     * element of the input array of IDs with no locked monitor and
     * synchronizer info.
     *
     * @throws IllegalArgumentException if <tt>maxDepth is negative</tt>.
     * @throws IllegalArgumentException if any element in the input array
     *      <tt>ids</tt> is {@code <= 0}.
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     *
     */
    public ThreadInfo[] getThreadInfo(long[] ids, int maxDepth) {
        return threadMXBean.getThreadInfo(ids, maxDepth);
    }

    /**
     * Tests if the Java virtual machine supports thread contention monitoring.
     *
     * @return
     *   <tt>true</tt>
     *     if the Java virtual machine supports thread contention monitoring;
     *   <tt>false</tt> otherwise.
     */
    public boolean isThreadContentionMonitoringSupported() {
        return threadMXBean.isThreadContentionMonitoringSupported();
    }

    /**
     * Tests if thread contention monitoring is enabled.
     *
     * @return <tt>true</tt> if thread contention monitoring is enabled;
     *         <tt>false</tt> otherwise.
     *
     * @throws UnsupportedOperationException if the Java virtual
     * machine does not support thread contention monitoring.
     *
     * @see #isThreadContentionMonitoringSupported
     */
    public boolean isThreadContentionMonitoringEnabled() {
        return threadMXBean.isThreadContentionMonitoringEnabled();
    }

    /**
     * Enables or disables thread contention monitoring.
     * Thread contention monitoring is disabled by default.
     *
     * @param enable <tt>true</tt> to enable;
     *               <tt>false</tt> to disable.
     *
     * @throws UnsupportedOperationException if the Java
     * virtual machine does not support thread contention monitoring.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("control").
     *
     * @see #isThreadContentionMonitoringSupported
     */
    public void setThreadContentionMonitoringEnabled(boolean enable) {
        threadMXBean.setThreadContentionMonitoringEnabled(enable);
    }

    /**
     * Returns the total CPU time for the current thread in nanoseconds.
     * The returned value is of nanoseconds precision but
     * not necessarily nanoseconds accuracy.
     * If the implementation distinguishes between user mode time and system
     * mode time, the returned CPU time is the amount of time that
     * the current thread has executed in user mode or system mode.
     *
     * <p>
     * This is a convenient method for local management use and is
     * equivalent to calling:
     * <blockquote><pre>
     *   {@link #getThreadCpuTime getThreadCpuTime}(Thread.currentThread().getId());
     * </pre></blockquote>
     *
     * @return the total CPU time for the current thread if CPU time
     * measurement is enabled; <tt>-1</tt> otherwise.
     *
     * @throws UnsupportedOperationException if the Java
     * virtual machine does not support CPU time measurement for
     * the current thread.
     *
     * @see #getCurrentThreadUserTime
     * @see #isCurrentThreadCpuTimeSupported
     * @see #isThreadCpuTimeEnabled
     * @see #setThreadCpuTimeEnabled
     */
    public long getCurrentThreadCpuTime() {
        return threadMXBean.getCurrentThreadCpuTime();
    }

    /**
     * Returns the CPU time that the current thread has executed
     * in user mode in nanoseconds.
     * The returned value is of nanoseconds precision but
     * not necessarily nanoseconds accuracy.
     *
     * <p>
     * This is a convenient method for local management use and is
     * equivalent to calling:
     * <blockquote><pre>
     *   {@link #getThreadUserTime getThreadUserTime}(Thread.currentThread().getId());
     * </pre></blockquote>
     *
     * @return the user-level CPU time for the current thread if CPU time
     * measurement is enabled; <tt>-1</tt> otherwise.
     *
     * @throws UnsupportedOperationException if the Java
     * virtual machine does not support CPU time measurement for
     * the current thread.
     *
     * @see #getCurrentThreadCpuTime
     * @see #isCurrentThreadCpuTimeSupported
     * @see #isThreadCpuTimeEnabled
     * @see #setThreadCpuTimeEnabled
     */
    public long getCurrentThreadUserTime() {
        return threadMXBean.getCurrentThreadUserTime();
    }

    /**
     * Returns the total CPU time for a thread of the specified ID in nanoseconds.
     * The returned value is of nanoseconds precision but
     * not necessarily nanoseconds accuracy.
     * If the implementation distinguishes between user mode time and system
     * mode time, the returned CPU time is the amount of time that
     * the thread has executed in user mode or system mode.
     *
     * <p>
     * If the thread of the specified ID is not alive or does not exist,
     * this method returns <tt>-1</tt>. If CPU time measurement
     * is disabled, this method returns <tt>-1</tt>.
     * A thread is alive if it has been started and has not yet died.
     * <p>
     * If CPU time measurement is enabled after the thread has started,
     * the Java virtual machine implementation may choose any time up to
     * and including the time that the capability is enabled as the point
     * where CPU time measurement starts.
     *
     * @param id the thread ID of a thread
     * @return the total CPU time for a thread of the specified ID
     * if the thread of the specified ID exists, the thread is alive,
     * and CPU time measurement is enabled;
     * <tt>-1</tt> otherwise.
     *
     * @throws IllegalArgumentException if {@code id <= 0}.
     * @throws UnsupportedOperationException if the Java
     * virtual machine does not support CPU time measurement for
     * other threads.
     *
     * @see #getThreadUserTime
     * @see #isThreadCpuTimeSupported
     * @see #isThreadCpuTimeEnabled
     * @see #setThreadCpuTimeEnabled
     */
    public long getThreadCpuTime(long id) {
        return threadMXBean.getThreadCpuTime(id);
    }

    /**
     * Returns the CPU time that a thread of the specified ID
     * has executed in user mode in nanoseconds.
     * The returned value is of nanoseconds precision but
     * not necessarily nanoseconds accuracy.
     *
     * <p>
     * If the thread of the specified ID is not alive or does not exist,
     * this method returns <tt>-1</tt>. If CPU time measurement
     * is disabled, this method returns <tt>-1</tt>.
     * A thread is alive if it has been started and has not yet died.
     * <p>
     * If CPU time measurement is enabled after the thread has started,
     * the Java virtual machine implementation may choose any time up to
     * and including the time that the capability is enabled as the point
     * where CPU time measurement starts.
     *
     * @param id the thread ID of a thread
     * @return the user-level CPU time for a thread of the specified ID
     * if the thread of the specified ID exists, the thread is alive,
     * and CPU time measurement is enabled;
     * <tt>-1</tt> otherwise.
     *
     * @throws IllegalArgumentException if {@code id <= 0}.
     * @throws UnsupportedOperationException if the Java
     * virtual machine does not support CPU time measurement for
     * other threads.
     *
     * @see #getThreadCpuTime
     * @see #isThreadCpuTimeSupported
     * @see #isThreadCpuTimeEnabled
     * @see #setThreadCpuTimeEnabled
     */
    public long getThreadUserTime(long id) {
        return threadMXBean.getThreadUserTime(id);
    }

    /**
     * Tests if the Java virtual machine implementation supports CPU time
     * measurement for any thread.
     * A Java virtual machine implementation that supports CPU time
     * measurement for any thread will also support CPU time
     * measurement for the current thread.
     *
     * @return
     *   <tt>true</tt>
     *     if the Java virtual machine supports CPU time
     *     measurement for any thread;
     *   <tt>false</tt> otherwise.
     */
    public boolean isThreadCpuTimeSupported() {
        return threadMXBean.isThreadCpuTimeSupported();
    }

    /**
     * Tests if the Java virtual machine supports CPU time
     * measurement for the current thread.
     * This method returns <tt>true</tt> if {@link #isThreadCpuTimeSupported}
     * returns <tt>true</tt>.
     *
     * @return
     *   <tt>true</tt>
     *     if the Java virtual machine supports CPU time
     *     measurement for current thread;
     *   <tt>false</tt> otherwise.
     */
    public boolean isCurrentThreadCpuTimeSupported() {
        return threadMXBean.isCurrentThreadCpuTimeSupported();
    }

    /**
     * Tests if thread CPU time measurement is enabled.
     *
     * @return <tt>true</tt> if thread CPU time measurement is enabled;
     *         <tt>false</tt> otherwise.
     *
     * @throws UnsupportedOperationException if the Java virtual
     * machine does not support CPU time measurement for other threads
     * nor for the current thread.
     *
     * @see #isThreadCpuTimeSupported
     * @see #isCurrentThreadCpuTimeSupported
     */
    public boolean isThreadCpuTimeEnabled() {
        return threadMXBean.isThreadCpuTimeEnabled();
    }

    /**
     * Enables or disables thread CPU time measurement.  The default
     * is platform dependent.
     *
     * @param enable <tt>true</tt> to enable;
     *               <tt>false</tt> to disable.
     *
     * @throws UnsupportedOperationException if the Java
     * virtual machine does not support CPU time measurement for
     * any threads nor for the current thread.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("control").
     *
     * @see #isThreadCpuTimeSupported
     * @see #isCurrentThreadCpuTimeSupported
     */
    public void setThreadCpuTimeEnabled(boolean enable) {
        threadMXBean.setThreadCpuTimeEnabled(enable);
    }

    /**
     * Finds cycles of threads that are in deadlock waiting to acquire
     * object monitors. That is, threads that are blocked waiting to enter a
     * synchronization block or waiting to reenter a synchronization block
     * after an {@link Object#wait Object.wait} call,
     * where each thread owns one monitor while
     * trying to obtain another monitor already held by another thread
     * in a cycle.
     * <p>
     * More formally, a thread is <em>monitor deadlocked</em> if it is
     * part of a cycle in the relation "is waiting for an object monitor
     * owned by".  In the simplest case, thread A is blocked waiting
     * for a monitor owned by thread B, and thread B is blocked waiting
     * for a monitor owned by thread A.
     * <p>
     * This method is designed for troubleshooting use, but not for
     * synchronization control.  It might be an expensive operation.
     * <p>
     * This method finds deadlocks involving only object monitors.
     * To find deadlocks involving both object monitors and
     * <a href="LockInfo.html#OwnableSynchronizer">ownable synchronizers</a>,
     * the {@link #findDeadlockedThreads findDeadlockedThreads} method
     * should be used.
     *
     * @return an array of IDs of the threads that are monitor
     * deadlocked, if any; <tt>null</tt> otherwise.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     *
     * @see #findDeadlockedThreads
     */
    public long[] findMonitorDeadlockedThreads() {
        return threadMXBean.findMonitorDeadlockedThreads();
    }

    /**
     * Resets the peak thread count to the current number of
     * live threads.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("control").
     *
     * @see #getPeakThreadCount
     * @see #getThreadCount
     */
    public void resetPeakThreadCount() {
        threadMXBean.resetPeakThreadCount();
    }

    /**
     * Finds cycles of threads that are in deadlock waiting to acquire
     * object monitors or
     * <a href="LockInfo.html#OwnableSynchronizer">ownable synchronizers</a>.
     *
     * Threads are <em>deadlocked</em> in a cycle waiting for a lock of
     * these two types if each thread owns one lock while
     * trying to acquire another lock already held
     * by another thread in the cycle.
     * <p>
     * This method is designed for troubleshooting use, but not for
     * synchronization control.  It might be an expensive operation.
     *
     * @return an array of IDs of the threads that are
     * deadlocked waiting for object monitors or ownable synchronizers, if any;
     * <tt>null</tt> otherwise.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     * @throws UnsupportedOperationException if the Java virtual
     * machine does not support monitoring of ownable synchronizer usage.
     *
     * @see #isSynchronizerUsageSupported
     * @see #findMonitorDeadlockedThreads
     * @since 1.6
     */
    public long[] findDeadlockedThreads() {
        return threadMXBean.findDeadlockedThreads();
    }

    /**
     * Tests if the Java virtual machine supports monitoring of
     * object monitor usage.
     *
     * @return
     *   <tt>true</tt>
     *     if the Java virtual machine supports monitoring of
     *     object monitor usage;
     *   <tt>false</tt> otherwise.
     *
     * @see #dumpAllThreads
     * @since 1.6
     */
    public boolean isObjectMonitorUsageSupported() {
        return threadMXBean.isObjectMonitorUsageSupported();
    }

    /**
     * Tests if the Java virtual machine supports monitoring of
     * <a href="LockInfo.html#OwnableSynchronizer">
     * ownable synchronizer</a> usage.
     *
     * @return
     *   <tt>true</tt>
     *     if the Java virtual machine supports monitoring of ownable
     *     synchronizer usage;
     *   <tt>false</tt> otherwise.
     *
     * @see #dumpAllThreads
     * @since 1.6
     */
    public boolean isSynchronizerUsageSupported() {
        return threadMXBean.isSynchronizerUsageSupported();
    }

    /**
     * Returns the thread info for each thread
     * whose ID is in the input array <tt>ids</tt>, with stack trace
     * and synchronization information.
     *
     * <p>
     * This method obtains a snapshot of the thread information
     * for each thread including:
     * <ul>
     *    <li>the entire stack trace,</li>
     *    <li>the object monitors currently locked by the thread
     *        if <tt>lockedMonitors</tt> is <tt>true</tt>, and</li>
     *    <li>the <a href="LockInfo.html#OwnableSynchronizer">
     *        ownable synchronizers</a> currently locked by the thread
     *        if <tt>lockedSynchronizers</tt> is <tt>true</tt>.</li>
     * </ul>
     * <p>
     * This method returns an array of the <tt>ThreadInfo</tt> objects,
     * each is the thread information about the thread with the same index
     * as in the <tt>ids</tt> array.
     * If a thread of the given ID is not alive or does not exist,
     * <tt>null</tt> will be set in the corresponding element
     * in the returned array.  A thread is alive if
     * it has been started and has not yet died.
     * <p>
     * If a thread does not lock any object monitor or <tt>lockedMonitors</tt>
     * is <tt>false</tt>, the returned <tt>ThreadInfo</tt> object will have an
     * empty <tt>MonitorInfo</tt> array.  Similarly, if a thread does not
     * lock any synchronizer or <tt>lockedSynchronizers</tt> is <tt>false</tt>,
     * the returned <tt>ThreadInfo</tt> object
     * will have an empty <tt>LockInfo</tt> array.
     *
     * <p>
     * When both <tt>lockedMonitors</tt> and <tt>lockedSynchronizers</tt>
     * parameters are <tt>false</tt>, it is equivalent to calling:
     * <blockquote><pre>
     *     {@link #getThreadInfo(long[], int)  getThreadInfo(ids, Integer.MAX_VALUE)}
     * </pre></blockquote>
     *
     * <p>
     * This method is designed for troubleshooting use, but not for
     * synchronization control.  It might be an expensive operation.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of <tt>ThreadInfo</tt> is
     * <tt>CompositeData</tt> with attributes as specified in the
     * {@link ThreadInfo#from ThreadInfo.from} method.
     *
     * @param  ids an array of thread IDs.
     * @param  lockedMonitors if <tt>true</tt>, retrieves all locked monitors.
     * @param  lockedSynchronizers if <tt>true</tt>, retrieves all locked
     *             ownable synchronizers.
     *
     * @return an array of the {@link ThreadInfo} objects, each containing
     * information about a thread whose ID is in the corresponding
     * element of the input array of IDs.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     * @throws UnsupportedOperationException
     *         <ul>
     *           <li>if <tt>lockedMonitors</tt> is <tt>true</tt> but
     *               the Java virtual machine does not support monitoring
     *               of {@linkplain #isObjectMonitorUsageSupported
     *               object monitor usage}; or</li>
     *           <li>if <tt>lockedSynchronizers</tt> is <tt>true</tt> but
     *               the Java virtual machine does not support monitoring
     *               of {@linkplain #isSynchronizerUsageSupported
     *               ownable synchronizer usage}.</li>
     *         </ul>
     *
     * @see #isObjectMonitorUsageSupported
     * @see #isSynchronizerUsageSupported
     *
     * @since 1.6
     */
    public ThreadInfo[] getThreadInfo(long[] ids, boolean lockedMonitors, boolean lockedSynchronizers) {
        return threadMXBean.getThreadInfo(ids, lockedMonitors, lockedSynchronizers);
    }

    /**
     * Returns the thread info for all live threads with stack trace
     * and synchronization information.
     * Some threads included in the returned array
     * may have been terminated when this method returns.
     *
     * <p>
     * This method returns an array of {@link ThreadInfo} objects
     * as specified in the {@link #getThreadInfo(long[], boolean, boolean)}
     * method.
     *
     * @param  lockedMonitors if <tt>true</tt>, dump all locked monitors.
     * @param  lockedSynchronizers if <tt>true</tt>, dump all locked
     *             ownable synchronizers.
     *
     * @return an array of {@link ThreadInfo} for all live threads.
     *
     * @throws SecurityException if a security manager
     *         exists and the caller does not have
     *         ManagementPermission("monitor").
     * @throws UnsupportedOperationException
     *         <ul>
     *           <li>if <tt>lockedMonitors</tt> is <tt>true</tt> but
     *               the Java virtual machine does not support monitoring
     *               of {@linkplain #isObjectMonitorUsageSupported
     *               object monitor usage}; or</li>
     *           <li>if <tt>lockedSynchronizers</tt> is <tt>true</tt> but
     *               the Java virtual machine does not support monitoring
     *               of {@linkplain #isSynchronizerUsageSupported
     *               ownable synchronizer usage}.</li>
     *         </ul>
     *
     * @see #isObjectMonitorUsageSupported
     * @see #isSynchronizerUsageSupported
     *
     * @since 1.6
     */
    public ThreadInfo[] dumpAllThreads(boolean lockedMonitors, boolean lockedSynchronizers) {
        return threadMXBean.dumpAllThreads(lockedMonitors, lockedSynchronizers);
    }

    /**
     * Returns the name representing the running Java virtual machine.
     * The returned name string can be any arbitrary string and
     * a Java virtual machine implementation can choose
     * to embed platform-specific useful information in the
     * returned name string.  Each running virtual machine could have
     * a different name.
     *
     * @return the name representing the running Java virtual machine.
     */
    public String getName() {
        return runtimeMXBean.getName();
    }

    /**
     * Returns the Java virtual machine implementation name.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.vm.name")}.
     *
     * @return the Java virtual machine implementation name.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getVmName() {
        return runtimeMXBean.getVmName();
    }

    /**
     * Returns the Java virtual machine implementation vendor.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.vm.vendor")}.
     *
     * @return the Java virtual machine implementation vendor.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getVmVendor() {
        return runtimeMXBean.getVmVendor();
    }

    /**
     * Returns the Java virtual machine implementation version.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.vm.version")}.
     *
     * @return the Java virtual machine implementation version.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getVmVersion() {
        return runtimeMXBean.getVmVersion();
    }

    /**
     * Returns the Java virtual machine specification name.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.vm.specification.name")}.
     *
     * @return the Java virtual machine specification name.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getSpecName() {
        return runtimeMXBean.getSpecName();
    }

    /**
     * Returns the Java virtual machine specification vendor.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.vm.specification.vendor")}.
     *
     * @return the Java virtual machine specification vendor.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getSpecVendor() {
        return runtimeMXBean.getSpecVendor();
    }

    /**
     * Returns the Java virtual machine specification version.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.vm.specification.version")}.
     *
     * @return the Java virtual machine specification version.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getSpecVersion() {
        return runtimeMXBean.getSpecVersion();
    }

    /**
     * Returns the version of the specification for the management interface
     * implemented by the running Java virtual machine.
     *
     * @return the version of the specification for the management interface
     * implemented by the running Java virtual machine.
     */
    public String getManagementSpecVersion() {
        return runtimeMXBean.getManagementSpecVersion();
    }

    /**
     * Returns the Java class path that is used by the system class loader
     * to search for class files.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.class.path")}.
     *
     * <p> Multiple paths in the Java class path are separated by the
     * path separator character of the platform of the Java virtual machine
     * being monitored.
     *
     * @return the Java class path.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getClassPath() {
        return runtimeMXBean.getClassPath();
    }

    /**
     * Returns the Java library path.
     * This method is equivalent to {@link System#getProperty
     * System.getProperty("java.library.path")}.
     *
     * <p> Multiple paths in the Java library path are separated by the
     * path separator character of the platform of the Java virtual machine
     * being monitored.
     *
     * @return the Java library path.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to this system property.
     * @see SecurityManager#checkPropertyAccess(String)
     * @see System#getProperty
     */
    public String getLibraryPath() {
        return runtimeMXBean.getLibraryPath();
    }

    /**
     * Tests if the Java virtual machine supports the boot class path
     * mechanism used by the bootstrap class loader to search for class
     * files.
     *
     * @return <tt>true</tt> if the Java virtual machine supports the
     * class path mechanism; <tt>false</tt> otherwise.
     */
    public boolean isBootClassPathSupported() {
        return runtimeMXBean.isBootClassPathSupported();
    }

    /**
     * Returns the boot class path that is used by the bootstrap class loader
     * to search for class files.
     *
     * <p> Multiple paths in the boot class path are separated by the
     * path separator character of the platform on which the Java
     * virtual machine is running.
     *
     * <p>A Java virtual machine implementation may not support
     * the boot class path mechanism for the bootstrap class loader
     * to search for class files.
     * The {@link #isBootClassPathSupported} method can be used
     * to determine if the Java virtual machine supports this method.
     *
     * @return the boot class path.
     *
     * @throws UnsupportedOperationException
     *     if the Java virtual machine does not support this operation.
     *
     * @throws SecurityException
     *     if a security manager exists and the caller does not have
     *     ManagementPermission("monitor").
     */
    public String getBootClassPath() {
        return runtimeMXBean.getBootClassPath();
    }

    /**
     * Returns the input arguments passed to the Java virtual machine
     * which does not include the arguments to the <tt>main</tt> method.
     * This method returns an empty list if there is no input argument
     * to the Java virtual machine.
     * <p>
     * Some Java virtual machine implementations may take input arguments
     * from multiple different sources: for examples, arguments passed from
     * the application that launches the Java virtual machine such as
     * the 'java' command, environment variables, configuration files, etc.
     * <p>
     * Typically, not all command-line options to the 'java' command
     * are passed to the Java virtual machine.
     * Thus, the returned input arguments may not
     * include all command-line options.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of {@code List<String>} is <tt>String[]</tt>.
     *
     * @return a list of <tt>String</tt> objects; each element
     * is an argument passed to the Java virtual machine.
     *
     * @throws SecurityException
     *     if a security manager exists and the caller does not have
     *     ManagementPermission("monitor").
     */
    public List<String> getInputArguments() {
        return runtimeMXBean.getInputArguments();
    }

    /**
     * 返回 Java 虚拟机的正常运行时间（以毫秒为单位）
     *
     * Returns the uptime of the Java virtual machine in milliseconds.
     *
     * @return uptime of the Java virtual machine in milliseconds.
     */
    public long getUptime() {
        return runtimeMXBean.getUptime();
    }

    /**
     * 返回 Java 虚拟机的启动时间（以毫秒为单位）
     *
     * Returns the start time of the Java virtual machine in milliseconds.
     * This method returns the approximate time when the Java virtual
     * machine started.
     *
     * @return start time of the Java virtual machine in milliseconds.
     *
     */
    public long getStartTime() {
        return runtimeMXBean.getStartTime();
    }

    /**
     * Returns a map of names and values of all system properties.
     * This method calls {@link System#getProperties} to get all
     * system properties.  Properties whose name or value is not
     * a <tt>String</tt> are omitted.
     *
     * <p>
     * <b>MBeanServer access</b>:<br>
     * The mapped type of {@code Map<String,String>} is
     * {@link TabularData TabularData}
     * with two items in each row as follows:
     * <blockquote>
     * <table border summary="Name and Type for each item">
     * <tr>
     *   <th>Item Name</th>
     *   <th>Item Type</th>
     *   </tr>
     * <tr>
     *   <td><tt>key</tt></td>
     *   <td><tt>String</tt></td>
     *   </tr>
     * <tr>
     *   <td><tt>value</tt></td>
     *   <td><tt>String</tt></td>
     *   </tr>
     * </table>
     * </blockquote>
     *
     * @return a map of names and values of all system properties.
     *
     * @throws SecurityException
     *     if a security manager exists and its
     *     <code>checkPropertiesAccess</code> method doesn't allow access
     *     to the system properties.
     */
    public Map<String, String> getSystemProperties() {
        return runtimeMXBean.getSystemProperties();
    }

}
