package com.github.shoothzj.javatool.test.http.server;

import com.github.shoothzj.javatool.http.HttpConstant;
import com.github.shoothzj.javatool.test.http.server.verifier.HttpResponseProvider;
import com.github.shoothzj.javatool.util.ExceptionUtil;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.codec.http.*;
import io.netty.handler.stream.ChunkedWriteHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

/**
 * Http proxy server提供这么几种能力用来mock
 * 它的请求响应形式
 * 1.符合头部要求,符合ip要求,符合url要求,给予固定的回应
 * 2.符合头部要求,符合ip要求,符合url要求,给予回应,单次
 * 3.其他请求,判断是否有callback,没有则返回默认
 *
 */
public class TestProxyHttpServer {

    private static final Logger log = LoggerFactory.getLogger(TestProxyHttpServer.class);

    private final Executor executor = Executors.newSingleThreadExecutor();

    private volatile HttpResponseProvider httpResponseProvider;

    private final Object object = new Object();

    private volatile FullHttpResponse response = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.OK);

    public TestProxyHttpServer() throws InterruptedException {
        this(null);
    }

    public TestProxyHttpServer(HttpResponseProvider httpResponseProvider) throws InterruptedException {
        this.httpResponseProvider = httpResponseProvider;
        EventLoopGroup bossGroup = new NioEventLoopGroup(1);
        EventLoopGroup workerGroup = new NioEventLoopGroup(1);
        ServerBootstrap b = new ServerBootstrap();
        b.group(bossGroup, workerGroup)
                .channel(NioServerSocketChannel.class)
                .childHandler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch) throws Exception {
                        ch.pipeline().addLast("http-decoder", new HttpRequestDecoder());
                        ch.pipeline().addLast("http-aggregator", new HttpObjectAggregator(65536));
                        ch.pipeline().addLast("http-encoder", new HttpResponseEncoder());
                        ch.pipeline().addLast("http-chunked", new ChunkedWriteHandler());
                        ch.pipeline().addLast("business", new TestProxyHttpHandler(TestProxyHttpServer.this));
                    }
                });
        ChannelFuture cf = b.bind(HttpConstant.zeroIp, HttpConstant.httpProxyPort);
        cf.sync();
        executor.execute(() -> {
            try {
                cf.channel().closeFuture().sync();
            } catch (InterruptedException e) {
                log.error("Executor close exception, exception is {}", ExceptionUtil.getException(e));
            } finally {
                bossGroup.shutdownGracefully();
                workerGroup.shutdownGracefully();
            }
        });
    }

    public void setHttpResponse(FullHttpResponse defaultHttpResponse) {
        synchronized (object) {
            this.response = defaultHttpResponse;
        }
    }

    public void setHttpResponseProvider(HttpResponseProvider httpResponseProvider) {
        synchronized (object) {
            this.httpResponseProvider = httpResponseProvider;
        }
    }

    protected FullHttpResponse askForResponse(String url, Map<String, String> header, String content) {
        if (httpResponseProvider != null) {
            Optional<FullHttpResponse> response = httpResponseProvider.response(url, header, content);
            if (response.isPresent()) {
                return response.get();
            }
        }
        return response;
    }


}
