package com.stringee.messaging;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.event.EventManager;
import com.stringee.database.DBHandler;
import com.stringee.exception.StringeeError;
import com.stringee.exception.StringeeException;
import com.stringee.listener.StatusListener;
import com.stringee.messaging.listeners.CallbackListener;

import java.io.Serializable;

public class ChatRequest implements Serializable {
    public enum RequestType {
        NORMAL(0),
        TRANSFER(1);
        private final short value;

        RequestType(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static RequestType getType(int value) {
            if (value == 1) {
                return TRANSFER;
            }
            return NORMAL;
        }
    }

    @Deprecated
    public enum ChannelType {
        SUPPORT(1),
        FACEBOOK(2),
        ZALO(3);
        private final short value;

        ChannelType(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static ChannelType getType(int value) {
            switch (value) {
                case 2:
                    return FACEBOOK;
                case 3:
                    return ZALO;
                default:
                    return SUPPORT;
            }
        }
    }

    public enum State {
        ACCEPTED(0),
        REJECTED(1);

        private final short value;

        State(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    private String convId;
    private String userId;
    private String name;
    private String customerId;
    private com.stringee.messaging.ChannelType channelType;
    private RequestType requestType = RequestType.NORMAL;
    boolean isTimeout = false;

    public String getConvId() {
        return convId;
    }

    public void setConvId(String convId) {
        this.convId = convId;
    }

    public String getUserId() {
        return userId;
    }

    public void setUserId(String userId) {
        this.userId = userId;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getCustomerId() {
        return customerId;
    }

    public void setCustomerId(String customerId) {
        this.customerId = customerId;
    }

    public ChannelType getChannelType() {
        switch (channelType) {
            case FACEBOOK:
                return ChannelType.FACEBOOK;
            case ZALO:
                return ChannelType.ZALO;
            default:
                return ChannelType.SUPPORT;
        }
    }

    public com.stringee.messaging.ChannelType getChatRequestChannel() {
        return channelType;
    }

    public void setChannelType(com.stringee.messaging.ChannelType channelType) {
        this.channelType = channelType;
    }

    public RequestType getRequestType() {
        return requestType;
    }

    public void setRequestType(RequestType requestType) {
        this.requestType = requestType;
    }

    public void setTimeout(boolean timeout) {
        isTimeout = timeout;
    }

    /**
     * Accept chat request
     *
     * @param client   StringeeClient
     * @param listener callback listener
     */
    public void accept(StringeeClient client, CallbackListener<Conversation> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (isTimeout) {
            listener.onError(new StringeeError(-3, "Request timeout"));
            return;
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            if (requestType == RequestType.NORMAL) {
                SendPacketUtils.sendChatResponse(client, requestId, convId, true, channelType.getValue());
            } else {
                Common.statusListenerMap.put(requestId, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        client.getConversationFromServer(convId, new CallbackListener<Conversation>() {
                            @Override
                            public void onSuccess(Conversation conversation) {
                                if (listener != null) {
                                    listener.onSuccess(conversation);
                                }

                                Conversation localConv = DBHandler.getInstance(client.getContext()).getConversationByRemoteId(convId, client.getUserId());
                                StringeeChange change;
                                if (localConv != null) {
                                    change = new StringeeChange(StringeeChange.Type.UPDATE, conversation);
                                } else {
                                    conversation.setDbId((int) DBHandler.getInstance(client.getContext()).insertConversation(conversation));
                                    change = new StringeeChange(StringeeChange.Type.INSERT, conversation);
                                }
                                EventManager.sendChatChangeEvent(client, change);
                            }
                        });
                    }

                    @Override
                    public void onError(StringeeError errorInfo) {
                        super.onError(errorInfo);
                        if (listener != null) {
                            listener.onError(errorInfo);
                        }
                    }
                });
                SendPacketUtils.sendChatTransferResponse(client, requestId, convId, true);
            }
        });
    }

    /**
     * Reject chat request
     *
     * @param client   StringeeClient
     * @param listener callback listener
     */
    public void reject(StringeeClient client, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (isTimeout) {
            listener.onError(new StringeeError(-3, "Request timeout"));
            return;
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            if (requestType == RequestType.NORMAL) {
                SendPacketUtils.sendChatResponse(client, requestId, convId, false, channelType.getValue());
            } else {
                SendPacketUtils.sendChatTransferResponse(client, requestId, convId, false);
            }
        });
    }
}
