package com.stringee.messaging;

import android.content.Context;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.Constant;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.UploadFileService;
import com.stringee.common.Utils;
import com.stringee.common.event.EventManager;
import com.stringee.database.DBHandler;
import com.stringee.exception.StringeeError;
import com.stringee.exception.StringeeException;
import com.stringee.listener.StatusListener;
import com.stringee.messaging.listeners.CallbackListener;

import org.json.JSONArray;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

public class Conversation extends StringeeObject implements Serializable {
    @Deprecated
    public enum Type {
        DEFAULT(0),
        SUPPORT(1);
        private final short value;

        Type(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static Type getType(int value) {
            if (value == 1) {
                return SUPPORT;
            }
            return DEFAULT;
        }
    }

    public enum State {
        DEFAULT(0),
        LEFT(1);
        private final short value;

        State(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static State getState(int value) {
            if (value == 1) {
                return LEFT;
            }
            return DEFAULT;
        }
    }

    @Deprecated
    public enum ChannelType {
        NORMAL(0),
        LIVECHAT(1),
        FACEBOOK(2),
        ZALO(3);

        private final short value;

        ChannelType(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static ChannelType getType(int value) {
            switch (value) {
                case 1:
                    return LIVECHAT;
                case 2:
                    return FACEBOOK;
                case 3:
                    return ZALO;
                default:
                    return NORMAL;
            }
        }
    }

    public enum Rating {
        BAD(0),
        GOOD(1);
        private final int value;

        Rating(final int value) {
            this.value = (short) value;
        }

        public int getValue() {
            return value;
        }
    }

    private int id;
    private String localId;
    private String convId;
    private String name;
    private List<User> participants;
    private boolean isDistinct;
    private boolean isGroup;
    private long updateAt;   // Last update timestamp
    private long createAt;    // Created timestamp
    private String userId;
    private String text;
    private long lastTimeNewMsg;
    private int totalUnread;
    private String creator;
    private State state;
    private long lastSequence;
    private String lastMsg;
    private long lastMsgSeqReceived;
    private String pinnedMsgId;
    private com.stringee.messaging.ChannelType channelType = com.stringee.messaging.ChannelType.NORMAL;
    private boolean ended;
    private String oaId;
    private String customData;
    private long lastMsgSeqSeen = 0;
    private Message lastMessage;
    private ChatStatus chatStatus = ChatStatus.RINGING;

    public Conversation() {
    }

    public int getDbId() {
        return id;
    }

    public void setDbId(int id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public void setParticipants(List<User> participants) {
        this.participants = participants;
    }

    public List<User> getParticipants() {
        if (participants == null) {
            return new ArrayList<>();
        }
        return participants;
    }

    public void addParticipants(User participant) {
        if (participants == null) {
            participants = new ArrayList<>();
        }
        this.participants.add(participant);
    }

    public boolean isDistinct() {
        return isDistinct;
    }

    public boolean isGroup() {
        return isGroup;
    }

    public void setDistinct(boolean distinct) {
        isDistinct = distinct;
    }

    public void setGroup(boolean group) {
        isGroup = group;
    }

    public long getUpdateAt() {
        return updateAt;
    }

    public void setUpdateAt(long updateAt) {
        this.updateAt = updateAt;
    }

    public long getCreateAt() {
        return createAt;
    }

    public void setCreateAt(long createAt) {
        this.createAt = createAt;
    }

    public String getId() {
        return convId;
    }

    public void setId(String conversationId) {
        this.convId = conversationId;
    }

    public String getLocalId() {
        return localId;
    }

    public void setLocalId(String sId) {
        this.localId = sId;
    }

    public String getClientId() {
        return userId;
    }

    public void setClientId(String userId) {
        this.userId = userId;
    }

    @Deprecated
    public String getLastMsgSender() {
        if (lastMessage != null) {
            User user = lastMessage.getSender();
            if (user != null) {
                return user.getUserId();
            }
        }
        return "";
    }

    @Deprecated
    public String getLastMsgSenderName() {
        if (lastMessage != null) {
            User user = lastMessage.getSender();
            if (user != null) {
                return user.getName();
            }
        }
        return "";
    }

    @Deprecated
    public String getLastMsgSenderAvatar() {
        if (lastMessage != null) {
            User user = lastMessage.getSender();
            if (user != null) {
                return user.getAvatarUrl();
            }
        }
        return "";
    }

    public String getText() {
        return text;
    }

    public void setText(String text) {
        this.text = text;
    }

    public long getLastTimeNewMsg() {
        return lastTimeNewMsg;
    }

    public void setLastTimeNewMsg(long lastTimeNewMsg) {
        this.lastTimeNewMsg = lastTimeNewMsg;
    }

    @Deprecated
    public Message.Type getLastMsgType() {
        if (lastMessage != null) {
            return lastMessage.getType();
        }
        return Message.Type.TEXT;
    }

    public int getTotalUnread() {
        return totalUnread;
    }

    public void setTotalUnread(int totalUnread) {
        this.totalUnread = totalUnread;
    }

    public String getCreator() {
        if (creator == null) {
            return "";
        }
        return creator;
    }

    public void setCreator(String creator) {
        this.creator = creator;
    }

    @Deprecated
    public Message.State getLastMsgState() {
        if (lastMessage != null) {
            return lastMessage.getState();
        }
        return Message.State.INITIALIZE;
    }

    public State getState() {
        return state;
    }

    public void setState(State state) {
        this.state = state;
    }

    @Deprecated
    public Type getType() {
        if (getConversationChannel() == com.stringee.messaging.ChannelType.NORMAL) {
            return Type.DEFAULT;
        } else {
            return Type.SUPPORT;
        }
    }

    public long getLastSequence() {
        return lastSequence;
    }

    public void setLastSequence(long lastSequence) {
        this.lastSequence = lastSequence;
    }

    public long getLastMsgSeqSeen() {
        return lastMsgSeqSeen;
    }

    public void setLastMsgSeqSeen(long lastMsgSeqSeen) {
        this.lastMsgSeqSeen = lastMsgSeqSeen;
    }

    @Deprecated
    public String getLastMsgId() {
        if (lastMessage != null) {
            return lastMessage.getId();
        }
        return "";
    }

    public String getLastMsg() {
        return lastMsg;
    }

    public void setLastMsg(String lastMsg) {
        this.lastMsg = lastMsg;
    }

    public long getLastMsgSeqReceived() {
        return lastMsgSeqReceived;
    }

    public void setLastMsgSeqReceived(long lastMsgSeqReceived) {
        this.lastMsgSeqReceived = lastMsgSeqReceived;
    }

    public String getPinnedMsgId() {
        return pinnedMsgId;
    }

    public void setPinnedMsgId(String pinnedMsgId) {
        this.pinnedMsgId = pinnedMsgId;
    }

    public ChannelType getChannelType() {
        switch (getConversationChannel()) {
            case LIVECHAT:
                return ChannelType.LIVECHAT;
            case FACEBOOK:
                return ChannelType.FACEBOOK;
            case ZALO:
                return ChannelType.ZALO;
            default:
                return ChannelType.NORMAL;
        }
    }

    public com.stringee.messaging.ChannelType getConversationChannel() {
        return channelType != null ? channelType : com.stringee.messaging.ChannelType.NORMAL;
    }

    public void setChannelType(com.stringee.messaging.ChannelType channelType) {
        this.channelType = channelType;
    }

    public boolean isEnded() {
        return ended;
    }

    public void setEnded(boolean ended) {
        this.ended = ended;
    }

    public String getOaId() {
        return oaId;
    }

    public void setOaId(String oaId) {
        this.oaId = oaId;
    }

    public String getCustomData() {
        return customData;
    }

    public void setCustomData(String customData) {
        this.customData = customData;
    }

    public Message getLastMessage() {
        return lastMessage;
    }

    public void setLastMessage(Message lastMessage) {
        this.lastMessage = lastMessage;
    }

    @Deprecated
    public String getLastMsgCustomData() {
        if (lastMessage != null) {
            return lastMessage.getCustomData() != null ? lastMessage.getCustomData().toString() : null;
        }
        return null;
    }

    public ChatStatus getChatStatus() {
        return chatStatus != null ? chatStatus : ChatStatus.RINGING;
    }

    public void setChatStatus(ChatStatus chatStatus) {
        this.chatStatus = chatStatus;
    }

    /**
     * Delete conversation
     *
     * @param client StringeeClient
     */
    public void delete(final StringeeClient client, final StatusListener statusListener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            if (isGroup) {
                List<String> userIdList = new ArrayList<>();
                if (!Utils.isEmpty(participants)) {
                    for (int i = 0; i < participants.size(); i++) {
                        User user = participants.get(i);
                        userIdList.add(user.getUserId());
                    }
                }
                if (userIdList.contains(client.getUserId())) {
                    if (statusListener != null) {
                        statusListener.onError(new StringeeError(-3, "Delete conversation failed. The conversation is group and User has not yet left the room."));
                    }
                } else {
                    client.getDbExecutor().execute(() -> DBHandler.getInstance(client.getContext()).deleteConversationByConvId(convId, client.getUserId()));
                    EventManager.sendChatChangeEvent(client, new StringeeChange(StringeeChange.Type.DELETE, Conversation.this));
                    if (statusListener != null) {
                        statusListener.onSuccess();
                    }
                }
            } else {
                long seq = DBHandler.getInstance(client.getContext()).getLastMsgSeqInConversation(convId, userId);
                if (seq < lastSequence) {
                    seq = lastSequence;
                }
                int requestId;
                synchronized (Common.lock) {
                    requestId = ++Common.requestId;
                }
                if (statusListener != null) {
                    Common.statusListenerMap.put(requestId, statusListener);
                }
                SendPacketUtils.deleteConversation(requestId, client, convId, seq);
            }
        });
    }

    /**
     * Send message
     *
     * @param message Message
     */
    public void sendMessage(final StringeeClient client, final Message message, final StatusListener statusListener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            message.setClientId(userId);
            message.setConversationId(convId);
            message.setConvLocalId(localId);
            message.setState(Message.State.SENDING);
            message.setLocalId(Utils.getDeviceId(client.getContext()) + "-message-" + System.currentTimeMillis());
            User sender = client.getUser(client.getUserId());
            message.setSender(sender);
            message.setCreatedOnLocalAt(System.currentTimeMillis());
            message.setCreatedAt(message.getCreatedOnLocalAt());
            String text = message.getText();
            Message.Type type = message.getType();
            switch (type) {
                case LOCATION:
                    text = "Location";
                    break;
                case CONTACT:
                    text = "Contact";
                    break;
                case AUDIO:
                    text = "Audio";
                    break;
                case PHOTO:
                    text = "Photo";
                    if (!Utils.isEmpty(message.getFilePath())) {
                        Utils.generateImage(message.getFilePath());
                        message.setImageRatio(Utils.getRatio(message.getFilePath()));
                        message.setThumbnail(Utils.getThumbnail(message.getFilePath(), Message.Type.PHOTO));
                    }
                    break;
                case VIDEO:
                    text = "Video";
                    if (!Utils.isEmpty(message.getFilePath())) {
                        message.setThumbnail(Utils.getThumbnail(message.getFilePath(), Message.Type.VIDEO));
                        message.setImageRatio(Utils.getRatio(message.getThumbnail()));
                    }
                    break;
                case FILE:
                    if (!Utils.isEmpty(message.getFilePath())) {
                        String[] parts = message.getFilePath().split("/");
                        text = parts[parts.length - 1];
                        message.setFileName(text);
                        message.setFileLength((new File(message.getFilePath())).length());
                    }
                    break;
                case STICKER:
                    text = "Sticker";
                    break;
            }
            message.setText(text);

            client.getDbExecutor().execute(() -> {
                // Save message to db
                DBHandler.getInstance(client.getContext()).insertMessage(message);
            });

            EventManager.sendChatChangeEvent(client, new StringeeChange(StringeeChange.Type.INSERT, message));
            // Update conversation
            updateAt = message.getCreatedAt();
            lastTimeNewMsg = updateAt;
            lastMessage = message;

            client.getDbExecutor().execute(() -> DBHandler.getInstance(client.getContext()).updateConversationWhenSendMsg(Conversation.this, userId));

            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (statusListener != null) {
                Common.statusListenerMap.put(requestId, statusListener);
            }

            client.getMessagesMap().put(requestId, message);
            if (type == Message.Type.PHOTO || type == Message.Type.AUDIO || type == Message.Type.VIDEO || type == Message.Type.FILE) {
                String fileUrl = message.getFileUrl();
                if (Utils.isEmpty(fileUrl)) {
                    UploadFileService.getInstance().upload(client, requestId, message);
                } else {
                    SendPacketUtils.sendMessage(client, requestId, message);
                }
            } else {
                // Send message to server
                SendPacketUtils.sendMessage(client, requestId, message);
            }
        });
    }

    /**
     * Get local messages
     *
     * @param client   StringeeClient
     * @param count    int
     * @param listener CallbackListener
     */
    public void getLocalMessages(final StringeeClient client, final int count, final CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            List<Message> messages = DBHandler.getInstance(client.getContext()).getLastLocalMessages(count, convId, userId);
            if (!messages.isEmpty()) {
                if (listener != null) {
                    listener.onSuccess(messages);
                }
            } else {
                if (listener != null) {
                    listener.onError(new StringeeError(-3, "No message found."));
                }
            }
        });
    }

    /**
     * Get last messages
     *
     * @param count    int
     * @param listener CallbackListener
     */
    public void getLastMessages(StringeeClient client, int count, CallbackListener<List<Message>> listener) {
        this.getLastMessages(client, count, false, false, false, listener);
    }

    /**
     * Get last messages
     *
     * @param client                StringeeClient
     * @param loadDeletedMsg        boolean
     * @param loadDeletedMsgContent boolean
     * @param count                 int
     * @param listener              CallbackListener
     * @param loadAll               boolean
     */
    public void getLastMessages(StringeeClient client, int count, boolean loadDeletedMsg, boolean loadDeletedMsgContent, boolean loadAll, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            int intLoadMsg, intLoadContent;
            if (loadDeletedMsg) {
                intLoadMsg = 1;
            } else {
                intLoadMsg = 0;
            }
            if (loadDeletedMsgContent) {
                intLoadContent = 1;
            } else {
                intLoadContent = 0;
            }
            SendPacketUtils.loadMessages(requestId, client, convId, 0, Long.MAX_VALUE, count, "DESC", intLoadMsg, intLoadContent, loadAll);
        });
    }

    /**
     * Get messages with sequence greater than seq
     *
     * @param client   StringeeClient
     * @param seq      long
     * @param count    int
     * @param listener CallbackListener
     */
    public void getMessagesAfter(StringeeClient client, long seq, int count, CallbackListener<List<Message>> listener) {
        this.getMessagesAfter(client, seq, count, false, false, false, listener);
    }

    /**
     * Get messages with sequence greater than seq
     *
     * @param client                StringeeClient
     * @param seq                   long
     * @param count                 int
     * @param loadDeletedMsg        boolean
     * @param loadDeletedMsgContent boolean
     * @param listener              CallbackListener
     * @param loadAll               boolean
     */
    public void getMessagesAfter(final StringeeClient client, long seq, int count, boolean loadDeletedMsg, boolean loadDeletedMsgContent, boolean loadAll, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            int intLoadMsg, intLoadContent;
            if (loadDeletedMsg) {
                intLoadMsg = 1;
            } else {
                intLoadMsg = 0;
            }
            if (loadDeletedMsgContent) {
                intLoadContent = 1;
            } else {
                intLoadContent = 0;
            }
            SendPacketUtils.loadMessages(requestId, client, convId, seq, Long.MAX_VALUE, count, "ASC", intLoadMsg, intLoadContent, loadAll);
        });
    }

    /**
     * Get messages with sequence smaller than seq
     *
     * @param client   StringeeClient
     * @param seq      long
     * @param count    int
     * @param listener CallbackListener
     */
    public void getMessagesBefore(StringeeClient client, long seq, int count, CallbackListener<List<Message>> listener) {
        this.getMessagesBefore(client, seq, count, false, false, false, listener);
    }

    /**
     * Get messages with sequence smaller than seq
     *
     * @param client                StringeeClient
     * @param seq                   long
     * @param count                 int
     * @param loadDeletedMsg        boolean
     * @param loadDeletedMsgContent boolean
     * @param listener              CallbackListener
     * @param loadAll               boolean
     */
    public void getMessagesBefore(StringeeClient client, long seq, int count, boolean loadDeletedMsg, boolean loadDeletedMsgContent, boolean loadAll, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            int intLoadMsg, intLoadContent;
            if (loadDeletedMsg) {
                intLoadMsg = 1;
            } else {
                intLoadMsg = 0;
            }
            if (loadDeletedMsgContent) {
                intLoadContent = 1;
            } else {
                intLoadContent = 0;
            }
            SendPacketUtils.loadMessages(requestId, client, convId, 0, seq, count, "DESC", intLoadMsg, intLoadContent, loadAll);
        });
    }

    /**
     * Get the last received message
     *
     * @param context Context
     */
    public Message getLastMessage(Context context) {
        return DBHandler.getInstance(context).getLastMessage(convId, userId);
    }

    /**
     * Add participants to a conversation
     *
     * @param client       StringeeClient
     * @param participants List<User>
     * @param listener     CallbackListener
     */
    public void addParticipants(StringeeClient client, List<User> participants, CallbackListener<List<User>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.addParticipants(requestId, client, convId, participants);
        });
    }

    /**
     * Remove participants from a conversation
     *
     * @param client       StringeeClient
     * @param participants List<User>
     * @param listener     CallbackListener
     */
    public void removeParticipants(StringeeClient client, List<User> participants, CallbackListener<List<User>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.removeParticipants(requestId, client, convId, participants);
        });
    }

    /**
     * Delete messages
     *
     * @param client   StringeeClient
     * @param messages List<Message>
     * @param listener StatusListener
     */
    @Deprecated
    public void deleteMessages(StringeeClient client, List<Message> messages, StatusListener listener) {
        if (Utils.isEmpty(messages)) {
            if (listener != null) {
                listener.onError(new StringeeError(-3, "Message list is empty"));
            }
            return;
        }
        JSONArray messageIds = new JSONArray();
        if (!Utils.isEmpty(messages)) {
            for (int i = 0; i < messages.size(); i++) {
                Message message = messages.get(i);
                messageIds.put(message.getId());
            }
        }
        deleteMessages(client, messageIds, listener);
    }

    /**
     * Delete messages
     *
     * @param client     StringeeClient
     * @param messageIds JSONArray
     * @param listener   StatusListener
     */
    public void deleteMessages(StringeeClient client, JSONArray messageIds, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (Utils.isEmpty(messageIds)) {
            if (listener != null) {
                listener.onError(new StringeeError(-3, "Message id list is empty"));
            }
            return;
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.deleteMessages(requestId, client, convId, messageIds);
        });
    }

    /**
     * Update a conversation
     *
     * @param client   StringeeClient
     * @param name     String
     * @param avatar   String
     * @param listener StatusListener
     */
    public void updateConversation(StringeeClient client, String name, String avatar, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            Common.statusListenerMap.put(requestId, new StatusListener() {
                @Override
                public void onSuccess() {
                    if (listener != null) {
                        listener.onSuccess();
                    }
                    setName(name);
                    client.getDbExecutor().execute(() -> DBHandler.getInstance(client.getContext()).updateConversation(Conversation.this));
                    EventManager.sendChatChangeEvent(client, new StringeeChange(StringeeChange.Type.UPDATE, Conversation.this));
                }

                @Override
                public void onError(StringeeError error) {
                    if (listener != null) {
                        listener.onError(error);
                    }
                }
            });
            SendPacketUtils.updateConversation(requestId, client, convId, name, avatar);
        });
    }

    /**
     * End a live chat
     *
     * @param client   StringeeClient
     * @param listener StatusListener
     */
    public void endChat(StringeeClient client, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.endChat(client, requestId, convId);
        });
    }

    /**
     * Rate live chat
     *
     * @param client   StringeeClient
     * @param comment  String
     * @param rating   Rating
     * @param listener StatusListener
     */
    public void rateChat(StringeeClient client, String comment, Rating rating, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.rateChat(client, requestId, convId, comment, rating.getValue());
        });
    }

    /**
     * Set participant as admin
     *
     * @param client   StringeeClient
     * @param userId   String
     * @param listener StatusListener
     */
    @Deprecated
    public void setAsAdmin(StringeeClient client, String userId, StatusListener listener) {
        this.setRole(client, userId, User.Role.ADMIN, listener);
    }

    /**
     * Set participant as member
     *
     * @param client   StringeeClient
     * @param userId   String
     * @param listener StatusListener
     */
    @Deprecated
    public void setAsMember(StringeeClient client, String userId, StatusListener listener) {
        this.setRole(client, userId, User.Role.MEMBER, listener);
    }

    /**
     * Change participant's role
     *
     * @param client   StringeeClient
     * @param userId   String
     * @param role     User.Role
     * @param listener StatusListener
     */
    public void setRole(StringeeClient client, String userId, User.Role role, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.setRole(client, requestId, convId, userId, role);
        });
    }

    /**
     * Get message by ids
     *
     * @param client   StringeeClient
     * @param msgIds   String[]
     * @param listener CallbackListener
     */
    public void getMessages(StringeeClient client, String[] msgIds, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.getMessagesByIds(client, requestId, convId, msgIds);
        });
    }

    /**
     * Transfer to another user
     *
     * @param client       StringeeClient
     * @param userId       String
     * @param customerId   String
     * @param customerName String
     * @param listener     StatusListener
     */
    public void transferTo(StringeeClient client, String userId, String customerId, String customerName, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            if (getConversationChannel() != com.stringee.messaging.ChannelType.NORMAL) {
                int requestId;
                synchronized (Common.lock) {
                    requestId = ++Common.requestId;
                }
                if (listener != null) {
                    Common.statusListenerMap.put(requestId, listener);
                }
                SendPacketUtils.transferChat(client, requestId, convId, userId, customerId, customerName);
            } else {
                listener.onError(new StringeeError(-1, "This conversation cannot transfer"));
            }
        });
    }

    /**
     * Send user begin typing
     *
     * @param client   StringeeClient
     * @param listener StatusListener
     */
    public void beginTyping(StringeeClient client, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.sendBeginTyping(client, requestId, convId);
        });
    }

    /**
     * Send user end typing
     *
     * @param client   StringeeClient
     * @param listener StatusListener
     */
    public void endTyping(StringeeClient client, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.sendEndTyping(client, requestId, convId);
        });
    }

    /**
     * Send chat transcript
     *
     * @param email    String
     * @param domain   String
     * @param listener StatusListener
     */
    public void sendChatTranscriptTo(StringeeClient client, String email, String domain, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.sendChatTranscriptResponse(client, requestId, convId, email, domain);
        });
    }

    /**
     * Get attachment messages
     *
     * @param client         StringeeClient
     * @param type           Message.Type
     * @param createdSmaller long
     * @param count          int
     * @param listener       CallbackListener
     */
    public void getAttachmentMessages(StringeeClient client, Message.Type type, long createdSmaller, int count, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.getAttachmentMessages(client, requestId, convId, type, createdSmaller, count);
        });
    }

    /**
     * Continue chat
     *
     * @param client   StringeeClient
     * @param listener CallbackListener
     */
    public void continueChatting(StringeeClient client, CallbackListener<Conversation> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            Common.statusListenerMap.put(requestId, new StatusListener() {
                @Override
                public void onSuccess() {
                    client.getConversationFromServer(convId, new CallbackListener<Conversation>() {
                        @Override
                        public void onSuccess(Conversation conversation) {
                            if (listener != null) {
                                listener.onSuccess(conversation);
                            }
                        }
                    });

                }

                @Override
                public void onError(StringeeError errorInfo) {
                    super.onError(errorInfo);
                    if (listener != null) {
                        listener.onError(errorInfo);
                    }
                }
            });
            SendPacketUtils.continueChatting(client, requestId, convId);
        });
    }

    /**
     * Mark all messages as read
     *
     * @param client   StringeeClient
     * @param listener StatusListener
     */
    public void markAllAsRead(final StringeeClient client, final StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (convId == null || lastSequence == 0) {
            listener.onError(new StringeeError(-1, "Conversation's id is null or message sequence is undefined."));
            return;
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            Message message = DBHandler.getInstance(client.getContext()).getLastMessage(convId, client.getUserId());
            client.getMessagesMap().put(requestId, message);
            SendPacketUtils.sendMessageReport(client, requestId, Conversation.this, Constant.MESSAGE_STATE_READ);
        });
    }

    /**
     * Get last X chat messages
     *
     * @param client   StringeeClient
     * @param count    int
     * @param listener CallbackListener
     */
    public void getLastXChatMessages(StringeeClient client, int count, CallbackListener<List<Message>> listener) {
        this.getLastXChatMessages(client, count, false, false, false, false, listener);
    }

    /**
     * Get last X chat messages
     *
     * @param client                StringeeClient
     * @param count                 int
     * @param isAscending           boolean
     * @param loadDeletedMsg        boolean
     * @param loadDeletedMsgContent boolean
     * @param loadAll               boolean
     * @param listener              CallbackListener
     */
    public void getLastXChatMessages(StringeeClient client, int count, boolean isAscending, boolean loadDeletedMsg, boolean loadDeletedMsgContent, boolean loadAll, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            int intLoadMsg, intLoadContent;
            if (loadDeletedMsg) {
                intLoadMsg = 1;
            } else {
                intLoadMsg = 0;
            }
            if (loadDeletedMsgContent) {
                intLoadContent = 1;
            } else {
                intLoadContent = 0;
            }
            SendPacketUtils.xChatLoadMessages(requestId, client, convId, 0, Long.MAX_VALUE, count, isAscending, intLoadMsg, intLoadContent, loadAll);
        });
    }

    /**
     * Get X chat messages after a sequence
     *
     * @param client   StringeeClient
     * @param seq      long
     * @param count    int
     * @param listener CallbackListener
     */
    public void getXChatMessagesAfter(StringeeClient client, long seq, int count, CallbackListener<List<Message>> listener) {
        this.getXChatMessagesAfter(client, seq, count, true, false, false, false, listener);
    }

    /**
     * Get X chat messages after a sequence
     *
     * @param client                StringeeClient
     * @param seq                   long
     * @param count                 int
     * @param isAscending           boolean
     * @param loadDeletedMsg        boolean
     * @param loadDeletedMsgContent boolean
     * @param loadAll               boolean
     * @param listener              CallbackListener
     */
    public void getXChatMessagesAfter(StringeeClient client, long seq, int count, boolean isAscending, boolean loadDeletedMsg, boolean loadDeletedMsgContent, boolean loadAll, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            int intLoadMsg, intLoadContent;
            if (loadDeletedMsg) {
                intLoadMsg = 1;
            } else {
                intLoadMsg = 0;
            }
            if (loadDeletedMsgContent) {
                intLoadContent = 1;
            } else {
                intLoadContent = 0;
            }
            SendPacketUtils.xChatLoadMessages(requestId, client, convId, seq, Long.MAX_VALUE, count, isAscending, intLoadMsg, intLoadContent, loadAll);
        });
    }

    /**
     * Get X chat messages before a sequence
     *
     * @param client   StringeeClient
     * @param seq      long
     * @param count    int
     * @param listener CallbackListener
     */
    public void getXChatMessagesBefore(StringeeClient client, long seq, int count, CallbackListener<List<Message>> listener) {
        this.getXChatMessagesBefore(client, seq, count, false, false, false, false, listener);
    }

    /**
     * Get X chat messages before a sequence
     *
     * @param client                StringeeClient
     * @param seq                   long
     * @param count                 int
     * @param isAscending           boolean
     * @param loadDeletedMsg        boolean
     * @param loadDeletedMsgContent boolean
     * @param loadAll               boolean
     * @param listener              CallbackListener
     */
    public void getXChatMessagesBefore(StringeeClient client, long seq, int count, boolean isAscending, boolean loadDeletedMsg, boolean loadDeletedMsgContent, boolean loadAll, CallbackListener<List<Message>> listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            int intLoadMsg, intLoadContent;
            if (loadDeletedMsg) {
                intLoadMsg = 1;
            } else {
                intLoadMsg = 0;
            }
            if (loadDeletedMsgContent) {
                intLoadContent = 1;
            } else {
                intLoadContent = 0;
            }
            SendPacketUtils.xChatLoadMessages(requestId, client, convId, 0, seq, count, isAscending, intLoadMsg, intLoadContent, loadAll);
        });
    }

    /**
     * Revoke messages
     *
     * @param client   StringeeClient
     * @param msgIds   List<String>
     * @param deleted  boolean
     * @param listener StatusListener
     */
    public void revokeMessages(final StringeeClient client, final List<String> msgIds, final boolean deleted, final StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (Utils.isEmpty(msgIds)) {
            listener.onError(new StringeeError(-1, "No message id found."));
        }
        client.getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            JSONArray msgIdArray = new JSONArray();
            if (!Utils.isEmpty(msgIds)) {
                for (int i = 0; i < msgIds.size(); i++) {
                    String msgId = msgIds.get(i);
                    msgIdArray.put(msgId);
                }
            }
            SendPacketUtils.revokeMessages(client, requestId, convId, msgIdArray, deleted);
        });
    }
}
