package com.stringee.video;

import android.Manifest.permission;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;

import androidx.core.content.ContextCompat;

import com.stringee.StringeeClient;
import com.stringee.call.StringeeCall2;
import com.stringee.call.StringeeCall2.EndCallCause;
import com.stringee.call.StringeeIceServer;
import com.stringee.call.VideoQuality;
import com.stringee.common.Common;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StatusListener;
import com.stringee.listener.StringeeRoomListener;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.video.StringeeVideoTrack.MediaType;
import com.stringee.video.StringeeVideoTrack.TrackType;

import org.json.JSONObject;

import java.util.LinkedList;
import java.util.List;

public class StringeeVideo {
    private static LinkedList<StringeeIceServer> iceServers;

    public static LinkedList<StringeeIceServer> getIceServers() {
        return iceServers;
    }

    public static void setIceServers(LinkedList<StringeeIceServer> iceServers) {
        StringeeVideo.iceServers = iceServers;
    }

    /**
     * Connect a room
     *
     * @param client   The client
     * @param token    The token
     * @param listener The listener
     * @return The room
     */
    public static StringeeRoom connect(StringeeClient client, String token, StringeeRoomListener listener) {
        StringeeRoom room = new StringeeRoom(client);
        room.setListener(listener);
        room.setState(StringeeRoom.State.CONNECTING);
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        client.getRoomRequest().put(requestId, room);
        SendPacketUtils.connectRoom(client, requestId, token);
        return room;
    }

    /**
     * Create the local video track
     *
     * @param context  The context
     * @param options  The options
     * @param listener The listener
     * @return The local video track
     */
    public static StringeeVideoTrack createLocalVideoTrack(Context context, StringeeVideoTrack.Options options, StatusListener listener) {
        if (options == null) {
            options = new StringeeVideoTrack.Options();
        }
        return createLocalVideoTrack(context, options, null, listener);
    }

    /**
     * Create the local video track
     *
     * @param context  The context
     * @param options  The options
     * @param listener The listener
     * @return The local video track
     */
    public static StringeeVideoTrack createLocalVideoTrack(Context context, StringeeVideoTrack.Options options, StringeeVideoTrack.CaptureSessionListener captureSessionListener, StatusListener listener) {
        if (options.videoEnabled()) {
            int permissionCamera = ContextCompat.checkSelfPermission(context, permission.CAMERA);
            if (permissionCamera != PackageManager.PERMISSION_GRANTED) {
                if (listener != null) {
                    listener.onError(new StringeeError(-1, "Permission is not granted"));
                }
                return null;
            }
        }

        if (options.audioEnabled()) {
            int permissionCamera = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO);
            if (permissionCamera != PackageManager.PERMISSION_GRANTED) {
                if (listener != null) {
                    listener.onError(new StringeeError(-1, "Permission is not granted"));
                }
                return null;
            }
        }

        StringeeVideoTrack localVideoTrack = new StringeeVideoTrack();
        localVideoTrack.setLocal(true);
        localVideoTrack.setAudio(options.audioEnabled());
        localVideoTrack.setVideo(options.videoEnabled());
        localVideoTrack.setScreenCapture(options.isScreenCapture());
        localVideoTrack.setCaptureSessionListener(captureSessionListener);
        localVideoTrack.create(context, options);
        String localId = Utils.getDeviceId(context) + "-" + System.currentTimeMillis();
        localVideoTrack.setLocalId(localId);

        if (listener != null) {
            listener.onSuccess();
        }
        return localVideoTrack;
    }

    /**
     * Release resource
     *
     * @param room The room
     */
    public static void release(StringeeRoom room) {
        release(room.getClient().getContext(), room);
    }

    /**
     * Release resource
     *
     * @param room The room
     */
    public static void release(Context context, StringeeRoom room) {
        if (room != null) {
            List<StringeeVideoTrack> videoTracks = room.getVideoTracks();
            for (int i = 0; i < videoTracks.size(); i++) {
                StringeeVideoTrack videoTrack = videoTracks.get(i);
                videoTrack.release();
            }
            room.getVideoTrackMap().clear();
        }
        StringeeVideoFactory videoFactory = StringeeVideoFactory.getInstance(context);
        videoFactory.release();
    }

    public static void initRoomCall(StringeeClient client, StringeeCall2 stringeeCall) {
        StringeeVideoTrack.Options options = new StringeeVideoTrack.Options();
        options.audio(true);
        options.video(stringeeCall.isVideoCall());
        options.screen(false);
        VideoQuality videoQuality = stringeeCall.getVideoQuality();
        if (videoQuality != null) {
            switch (videoQuality) {
                case QUALITY_288P:
                    options.videoDimensions(VideoDimensions.CIF_VIDEO_DIMENSIONS);
                    break;
                case QUALITY_480P:
                    options.videoDimensions(VideoDimensions.WVGA_VIDEO_DIMENSIONS);
                    break;
                case QUALITY_720P:
                    options.videoDimensions(VideoDimensions.HD_720P_VIDEO_DIMENSIONS);
                    break;
                case QUALITY_1080P:
                    options.videoDimensions(VideoDimensions.HD_1080P_VIDEO_DIMENSIONS);
                    break;
            }
        }
        StringeeVideoTrack localVideoTrack = StringeeVideo.createLocalVideoTrack(client.getContext(), options, stringeeCall.getCaptureSessionListener(), null);
        if (localVideoTrack != null) {
            localVideoTrack.mute(stringeeCall.isMute());
            localVideoTrack.enableVideo(stringeeCall.isVideoEnable());
        }

        stringeeCall.setLocalVideoTrack(localVideoTrack);
        StringeeCall2.StringeeCallListener callListener = stringeeCall.getCallListener();
        StringeeRoom stringeeRoom = StringeeVideo.connect(client, stringeeCall.getRoomToken(), new StringeeRoomListener() {
            @Override
            public void onConnected(StringeeRoom stringeeRoom) {
                client.getExecutor().execute(() -> {
                    // Publish local video track to room
                    if (localVideoTrack != null) {
                        stringeeRoom.publish(localVideoTrack, new StatusListener() {
                            @Override
                            public void onSuccess() {
                                stringeeCall.startTimer();
                                stringeeCall.setReJoining(false);
                                if (callListener != null && stringeeCall.isVideoCall()) {
                                    callListener.onLocalStream(stringeeCall);
                                }
                            }

                            @Override
                            public void onError(StringeeError errorInfo) {
                                if (callListener != null) {
                                    callListener.onError(stringeeCall, -1, "Can not publish the local video track");
                                }
                                stringeeCall.hangup("Publish track error: " + errorInfo.getMessage(), null);
                                if (callListener != null) {
                                    callListener.onSignalingStateChange(stringeeCall, StringeeCall2.SignalingState.ENDED, EndCallCause.NORMAL.getValue(), -1, "");
                                }
                                client.getCallMap2().remove(stringeeCall.getCallId());
                            }
                        });
                    }

                    if (stringeeCall.getLocalCaptureTrack() != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                        String localId = stringeeCall.getLocalCaptureTrack().getLocalId();
                        StringeeScreenCapture capture = client.getScreenCaptureMap().get(localId);
                        if (capture != null) {
                            capture.reCreateCaptureTrack(new CallbackListener<StringeeVideoTrack>() {
                                @Override
                                public void onSuccess(StringeeVideoTrack captureTrack) {
                                    stringeeCall.setLocalCaptureTrack(captureTrack);
                                    stringeeRoom.publish(captureTrack, new StatusListener() {
                                        @Override
                                        public void onSuccess() {
                                            stringeeCall.setLocalCaptureTrack(captureTrack);
                                            client.getScreenCaptureMap().put(captureTrack.getLocalId(), capture);
                                            client.getScreenCaptureMap().remove(localId);
                                            if (callListener != null) {
                                                callListener.onVideoTrackAdded(captureTrack);
                                            }
                                        }
                                    });
                                }
                            });
                        }
                    }

                    // Subscribe all remote tracks (ringtone, waiting video...)
                    List<StringeeVideoTrack> videoTracks = stringeeRoom.getRemoteVideoTracks();
                    for (int j = 0; j < videoTracks.size(); j++) {
                        StringeeVideoTrack videoTrack = videoTracks.get(j);
                        videoTrack.setListener(videoTrackListener(client, stringeeCall, videoTrack));
                        StringeeVideoTrack.Options opts = new StringeeVideoTrack.Options();
                        opts.audio(true);
                        opts.video(stringeeCall.isVideoCall());
                        opts.screen(false);
                        stringeeRoom.subscribe(videoTrack, opts, new StatusListener() {
                            @Override
                            public void onSuccess() {
                            }
                        });
                    }
                });
            }

            @Override
            public void onDisconnected(StringeeRoom stringeeRoom) {

            }

            @Override
            public void onError(StringeeRoom stringeeRoom, StringeeError error) {
                if (callListener != null) {
                    callListener.onError(stringeeCall, -1, "The call is not existence");
                }
                stringeeCall.hangup("Connect room error: " + error.getMessage(), null);
                if (callListener != null) {
                    callListener.onSignalingStateChange(stringeeCall, StringeeCall2.SignalingState.ENDED, EndCallCause.CALL_NOT_EXIST_IN_SERVER.getValue(), -1, "");
                }
                client.getCallMap2().remove(stringeeCall.getCallId());
            }

            @Override
            public void onParticipantConnected(StringeeRoom stringeeRoom, RemoteParticipant remoteParticipant) {

            }

            @Override
            public void onParticipantDisconnected(StringeeRoom stringeeRoom, RemoteParticipant remoteParticipant) {

            }

            @Override
            public void onVideoTrackAdded(StringeeRoom stringeeRoom, StringeeVideoTrack videoTrack) {
                client.getExecutor().execute(() -> {
                    videoTrack.setListener(videoTrackListener(client, stringeeCall, videoTrack));
                    StringeeVideoTrack.Options opts = new StringeeVideoTrack.Options();
                    opts.audio(true);
                    opts.video(stringeeCall.isVideoCall());
                    opts.screen(false);
                    stringeeRoom.subscribe(videoTrack, opts, new StatusListener() {
                        @Override
                        public void onSuccess() {
                        }
                    });
                });
            }

            @Override
            public void onVideoTrackRemoved(StringeeRoom stringeeRoom, StringeeVideoTrack videoTrack) {
                client.getExecutor().execute(() -> {
                    stringeeRoom.unsubscribe(videoTrack, new StatusListener() {
                        @Override
                        public void onSuccess() {

                        }
                    });

                    StringeeVideoTrack currentRemoteTrack = stringeeCall.getRemoteVideoTrack();
                    if (!currentRemoteTrack.getId().equals(videoTrack.getId())) {
                        if (callListener != null) {
                            callListener.onVideoTrackRemoved(videoTrack);
                        }
                    } else {
                        stringeeCall.setRemoteVideoTrack(null);
                    }
                });
            }

            @Override
            public void onMessage(StringeeRoom stringeeRoom, JSONObject msg, RemoteParticipant remoteParticipant) {

            }

            @Override
            public void onVideoTrackNotification(RemoteParticipant participant, StringeeVideoTrack videoTrack, MediaType mediaType) {
                client.getExecutor().execute(() -> {
                    if (callListener != null) {
                        switch (mediaType) {
                            case AUDIO:
                                callListener.onTrackMediaStateChange(participant.getId(), mediaType, videoTrack.audioEnabled());
                                break;
                            case VIDEO:
                                callListener.onTrackMediaStateChange(participant.getId(), mediaType, videoTrack.videoEnabled());
                                break;
                        }
                    }
                });
            }
        });
        stringeeCall.setRoom(stringeeRoom);
    }

    private static StringeeVideoTrack.Listener videoTrackListener(StringeeClient client, StringeeCall2 stringeeCall, StringeeVideoTrack videoTrack) {
        StringeeCall2.StringeeCallListener callListener = stringeeCall.getCallListener();
        return new StringeeVideoTrack.Listener() {
            @Override
            public void onMediaAvailable() {
                client.getExecutor().execute(() -> {
                    switch (videoTrack.getTrackType()) {
                        case PLAYER:
                            stringeeCall.setRemoteVideoTrack(videoTrack);
                            if (callListener != null) {
                                callListener.onRemoteStream(stringeeCall);
                            }
                            break;
                        case SCREEN:
                            if (callListener != null) {
                                callListener.onVideoTrackAdded(videoTrack);
                            }
                            break;
                        case CAMERA:
                            StringeeVideoTrack currentRemoteTrack = stringeeCall.getRemoteVideoTrack();
                            if (currentRemoteTrack == null) {
                                stringeeCall.setRemoteVideoTrack(videoTrack);
                                if (callListener != null) {
                                    callListener.onRemoteStream(stringeeCall);
                                }
                            } else {
                                if (currentRemoteTrack.getTrackType() == TrackType.CAMERA && !currentRemoteTrack.getId().equals(videoTrack.getId())) {
                                    if (callListener != null) {
                                        callListener.onVideoTrackAdded(videoTrack);
                                    }
                                } else {
                                    stringeeCall.setRemoteVideoTrack(videoTrack);
                                    if (callListener != null) {
                                        callListener.onRemoteStream(stringeeCall);
                                    }
                                }
                            }
                            break;
                    }
                });
            }

            @Override
            public void onMediaStateChange(StringeeVideoTrack.MediaState mediaState) {
                client.getExecutor().execute(() -> {
                    if (callListener != null) {
                        StringeeCall2.MediaState callState;
                        if (mediaState == StringeeVideoTrack.MediaState.CONNECTED) {
                            callState = StringeeCall2.MediaState.CONNECTED;
                        } else {
                            callState = StringeeCall2.MediaState.DISCONNECTED;
                        }
                        callListener.onMediaStateChange(stringeeCall, callState);
                    }
                });
            }
        };
    }
}
