package com.stringee.call;

import android.content.Context;
import android.graphics.Bitmap;
import android.util.Log;

import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StatusListener;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.video.VideoDimensions;

import org.json.JSONObject;
import org.webrtc.AudioSource;
import org.webrtc.AudioTrack;
import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.CapturerObserver;
import org.webrtc.DefaultVideoDecoderFactory;
import org.webrtc.DefaultVideoEncoderFactory;
import org.webrtc.EglBase;
import org.webrtc.IceCandidate;
import org.webrtc.MediaConstraints;
import org.webrtc.MediaStream;
import org.webrtc.PeerConnection;
import org.webrtc.PeerConnection.SdpSemantics;
import org.webrtc.PeerConnectionFactory;
import org.webrtc.RTCStats;
import org.webrtc.RtpSender;
import org.webrtc.SdpObserver;
import org.webrtc.SessionDescription;
import org.webrtc.SurfaceTextureHelper;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoDecoderFactory;
import org.webrtc.VideoEncoderFactory;
import org.webrtc.VideoFrame;
import org.webrtc.VideoSource;
import org.webrtc.VideoTrack;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Created by luannguyen on 9/27/16.
 */
public class StringeeCallFactory {

    private static final String AUDIO_ECHO_CANCELLATION_CONSTRAINT = "googEchoCancellation";
    private static final String AUDIO_EXPERIMENTAL__ECHO_CANCELLATION_CONSTRAINT = "googExperimentalEchoCancellation";
    private static final String AUDIO_AUTO_GAIN_CONTROL_CONSTRAINT = "googAutoGainControl";
    private static final String AUDIO_EXPERIMENTAL_AUTO_GAIN_CONTROL_CONSTRAINT = "googExperimentalAutoGainControl";
    private static final String AUDIO_HIGH_PASS_FILTER_CONSTRAINT = "googHighpassFilter";
    private static final String AUDIO_NOISE_SUPPRESSION_CONSTRAINT = "googNoiseSuppression";
    private static final String AUDIO_EXPERIMENTAL_NOISE_SUPPRESSION_CONSTRAINT = "googExperimentalNoiseSuppression";
    private static final String AUDIO_MIRROR_CONSTRAINT = "googAudioMirroring";

    public interface StringeeCallListener {

        enum StringeeConnectionState {
            NEW, CHECKING, CONNECTED, COMPLETED, FAILED, DISCONNECTED, CLOSED;

            StringeeConnectionState() {

            }
        }

        void onSetSdpSuccess(StringeeSessionDescription sdp);

        void onCreateIceCandidate(StringeeIceCandidate iceCandidate);

        void onChangeConnectionState(String callId, StringeeConnectionState state);

        void onAddStream(MediaStream mediaStream);

    }

    private final String AUDIO_TRACK_ID = "ARDAMSa0";
    private final String VIDEO_TRACK_ID = "ARDAMSv0";
    private AudioTrack localAudioTrack;
    private VideoTrack localVideoTrack;
    private MediaStream localMediaStream;

    static EglBase eglBase;
    static EglBase.Context rootEglBaseContext;
    static VideoSource videoSource;
    static VideoCapturer videoCapturer;//to switch camera
    static AudioSource audioSource;
    static PeerConnectionFactory sFactory;
    private SurfaceTextureHelper surfaceTextureHelper;
    private MediaConstraints sdpMediaConstraints;

    private PeerConnection peerConnection;
    private final Context mContext;
    private boolean callStarted = false;
    private SessionDescription localDescription;
    private final StringeeCall mStringeeCall;

    private final LinkedList<StringeeIceServer> iceServers;

    private final StringeeCallListener listener;
    public boolean iceConnected = false;

    static ScheduledExecutorService executor;

    private VideoFrame snapshotFrame;
    private CallbackListener<Bitmap> snapshotListener;
    private StringeeCall.CaptureSessionListener captureSessionListener;
    private final List<RtpSender> rtpSenders = new ArrayList<>();

    public void setCaptureSessionListener(StringeeCall.CaptureSessionListener captureSessionListener) {
        this.captureSessionListener = captureSessionListener;
    }

    public StringeeCallListener getListener() {
        return listener;
    }

    public StringeeCallFactory(StringeeCall stringeeCall, LinkedList<StringeeIceServer> iceServers, StringeeCallListener listener) {
        mStringeeCall = stringeeCall;
        mContext = mStringeeCall.getContext();
        this.listener = listener;
        this.iceServers = iceServers;
        executor = mStringeeCall.getClient().getExecutor();
    }

    public void startCall(final boolean isCaller) {
        callStarted = true;
        boolean isVideoCall = mStringeeCall.isVideoCall();

        PeerConnectionFactory.InitializationOptions.Builder optionBuilder = PeerConnectionFactory.InitializationOptions.builder(mContext.getApplicationContext());
        PeerConnectionFactory.initialize(optionBuilder.createInitializationOptions());

        eglBase = EglBase.create();
        rootEglBaseContext = eglBase.getEglBaseContext();
        VideoEncoderFactory encoderFactory = new DefaultVideoEncoderFactory(rootEglBaseContext, false, false);
        VideoDecoderFactory decoderFactory = new DefaultVideoDecoderFactory(rootEglBaseContext);
        sFactory = PeerConnectionFactory.builder().setVideoEncoderFactory(encoderFactory).setVideoDecoderFactory(decoderFactory).createPeerConnectionFactory();

        MediaConstraints audioConstraints = new MediaConstraints();
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_ECHO_CANCELLATION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_EXPERIMENTAL__ECHO_CANCELLATION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_AUTO_GAIN_CONTROL_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_EXPERIMENTAL_AUTO_GAIN_CONTROL_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_HIGH_PASS_FILTER_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_NOISE_SUPPRESSION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_EXPERIMENTAL_NOISE_SUPPRESSION_CONSTRAINT, "true"));
        audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(AUDIO_MIRROR_CONSTRAINT, "false"));
        audioSource = sFactory.createAudioSource(audioConstraints);
        localMediaStream = sFactory.createLocalMediaStream("ARDAMS" + System.currentTimeMillis());
        localAudioTrack = sFactory.createAudioTrack(AUDIO_TRACK_ID + System.currentTimeMillis(), audioSource);
        localAudioTrack.setEnabled(true);
        localMediaStream.addTrack(localAudioTrack);

        if (isVideoCall) {
            videoCapturer = createVideoCapturer(mContext);
            surfaceTextureHelper = SurfaceTextureHelper.create("CaptureThread", rootEglBaseContext);
            videoSource = sFactory.createVideoSource(videoCapturer.isScreencast());
            int minWidth = VideoDimensions.WVGA_VIDEO_WIDTH;
            int minHeight = VideoDimensions.WVGA_VIDEO_HEIGHT;
            VideoQuality videoQuality = mStringeeCall.getVideoQuality();
            if (videoQuality != null) {
                switch (mStringeeCall.getVideoQuality()) {
                    case QUALITY_288P:
                        minWidth = VideoDimensions.CIF_VIDEO_WIDTH;
                        minHeight = VideoDimensions.CIF_VIDEO_HEIGHT;
                        break;
                    case QUALITY_720P:
                        minWidth = VideoDimensions.HD_720P_VIDEO_WIDTH;
                        minHeight = VideoDimensions.HD_720P_VIDEO_HEIGHT;
                        break;
                    case QUALITY_1080P:
                        minWidth = VideoDimensions.HD_1080P_VIDEO_WIDTH;
                        minHeight = VideoDimensions.HD_1080P_VIDEO_HEIGHT;
                        break;
                }
            }
            videoCapturer.initialize(surfaceTextureHelper, mContext, new CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean b) {
                    videoSource.getCapturerObserver().onCapturerStarted(b);
                    if (captureSessionListener != null) {
                        captureSessionListener.onCapturerStarted();
                    }
                }

                @Override
                public void onCapturerStopped() {
                    videoSource.getCapturerObserver().onCapturerStopped();
                    if (captureSessionListener != null) {
                        captureSessionListener.onCapturerStopped();
                    }
                }

                @Override
                public void onFrameCaptured(VideoFrame videoFrame) {
                    videoSource.getCapturerObserver().onFrameCaptured(videoFrame);
                    if (snapshotListener != null) {
                        if (snapshotFrame != null) {
                            return;
                        }
                        snapshotFrame = videoFrame;

                        Bitmap bitmap = Utils.generateBitmap(snapshotFrame);
                        Bitmap emptyBitmap = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), bitmap.getConfig());
                        if (!bitmap.sameAs(emptyBitmap)) {
                            snapshotListener.onSuccess(bitmap);
                        } else {
                            snapshotListener.onError(new StringeeError(1, "Snapshot false"));
                        }
                        snapshotListener = null;
                        snapshotFrame = null;
                    }
                }
            });
            videoCapturer.startCapture(minWidth, minHeight, 30);
            localVideoTrack = sFactory.createVideoTrack(VIDEO_TRACK_ID + System.currentTimeMillis(), videoSource);
            localVideoTrack.setEnabled(true);
            localMediaStream.addTrack(localVideoTrack);

            mStringeeCall.setRootContext(rootEglBaseContext);
        }

        mStringeeCall.setLocalStream(localMediaStream);
        StringeeCall.StringeeCallListener callListener = mStringeeCall.getCallListener();
        if (callListener != null) {
            callListener.onLocalStream(mStringeeCall);
        }

        final StringeeCreateSdpObserver createSdpObserver = new StringeeCreateSdpObserver(true, mStringeeCall);

        final PCObserver pcObserver = new PCObserver(mStringeeCall);

        LinkedList<PeerConnection.IceServer> peerIceServers = createIceServers(iceServers);
        PeerConnection.RTCConfiguration rtcConfig = new PeerConnection.RTCConfiguration(peerIceServers);
        if (mStringeeCall.getCallType() == CallType.APP_TO_PHONE) {
            rtcConfig.bundlePolicy = PeerConnection.BundlePolicy.MAXCOMPAT;
        } else {
            rtcConfig.bundlePolicy = PeerConnection.BundlePolicy.BALANCED;
        }
        rtcConfig.rtcpMuxPolicy = PeerConnection.RtcpMuxPolicy.REQUIRE;
        rtcConfig.continualGatheringPolicy = PeerConnection.ContinualGatheringPolicy.GATHER_CONTINUALLY;
        rtcConfig.sdpSemantics = SdpSemantics.UNIFIED_PLAN;

        //disable TCP
        rtcConfig.tcpCandidatePolicy = PeerConnection.TcpCandidatePolicy.DISABLED;
        peerConnection = sFactory.createPeerConnection(rtcConfig, pcObserver);
        mStringeeCall.setInitComplete(true);

        List<String> mediaStreamLabels = Collections.singletonList("ARDAMS");
        if (!Utils.isEmpty(localMediaStream.audioTracks)) {
            List<AudioTrack> audioTracks = localMediaStream.audioTracks;
            for (int i = 0; i < audioTracks.size(); i++) {
                AudioTrack audioTrack = audioTracks.get(i);
                RtpSender rtpSender = peerConnection.addTrack(audioTrack, mediaStreamLabels);
                rtpSenders.add(rtpSender);
            }
        }
        if (!Utils.isEmpty(localMediaStream.videoTracks)) {
            List<VideoTrack> videoTracks = localMediaStream.videoTracks;
            for (int i = 0; i < videoTracks.size(); i++) {
                VideoTrack videoTrack = videoTracks.get(i);
                RtpSender rtpSender = peerConnection.addTrack(videoTrack, mediaStreamLabels);
                rtpSenders.add(rtpSender);
            }
        }

        if (isCaller) {
            // Create SDP constraints.
            sdpMediaConstraints = new MediaConstraints();
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "true"));
            if (isVideoCall) {
                sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "true"));
            } else {
                sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "false"));
            }
            peerConnection.createOffer(createSdpObserver, sdpMediaConstraints);
        }

    }

    public void stopCall(final boolean isHangup) {
        if (!callStarted) {
            return;
        }
        callStarted = false;

        if (peerConnection != null) {
            peerConnection.dispose();
            peerConnection = null;
        }

        if (isHangup) {
            if (audioSource != null) {
                audioSource.dispose();
                audioSource = null;
            }

            if (videoCapturer != null) {
                try {
                    videoCapturer.stopCapture();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                videoCapturer.dispose();
                videoCapturer = null;
            }

            if (videoSource != null) {
                videoSource.dispose();
                videoSource = null;
            }

            if (sFactory != null) {
                sFactory.dispose();
                sFactory = null;
            }

            if (eglBase != null) {
                eglBase.release();
                eglBase = null;
            }

            rootEglBaseContext = null;
        }
    }

    private void setRemoteDescription(StringeeCall stringeeCall, final StringeeSessionDescription sessionDescription) {
        boolean setOffer = true;
        final SessionDescription.Type type;
        if (sessionDescription.type == StringeeSessionDescription.Type.OFFER) {
            type = SessionDescription.Type.OFFER;
        } else if (sessionDescription.type == StringeeSessionDescription.Type.PRANSWER) {
            type = SessionDescription.Type.PRANSWER;
            setOffer = false;
        } else if (sessionDescription.type == StringeeSessionDescription.Type.ANSWER) {
            type = SessionDescription.Type.ANSWER;
            setOffer = false;
        } else {
            type = SessionDescription.Type.OFFER;
        }

        final StringeeSetSdpObserver setSdpObserver = new StringeeSetSdpObserver(false, setOffer, stringeeCall);
        SessionDescription webrtcSdp = new SessionDescription(type, sessionDescription.description);
        peerConnection.setRemoteDescription(setSdpObserver, webrtcSdp);
    }

    public void addIceCandidate(final StringeeIceCandidate iceCandidate) {
        Log.d("Stringee", "add ice candidate " + iceCandidate.sdp);
        IceCandidate webrtcIceCandidate = new IceCandidate(iceCandidate.sdpMid, iceCandidate.sdpMLineIndex, iceCandidate.sdp);
        peerConnection.addIceCandidate(webrtcIceCandidate);
    }

    public static LinkedList<PeerConnection.IceServer> createIceServers(LinkedList<StringeeIceServer> iceServers) {
        LinkedList<PeerConnection.IceServer> list = new LinkedList<>();

        for (int i = 0; i < iceServers.size(); i++) {
            StringeeIceServer ice = iceServers.get(i);
            PeerConnection.IceServer.Builder builder = PeerConnection.IceServer.builder(ice.uri);
            builder.setUsername(ice.username);
            builder.setPassword(ice.password);
            list.add(builder.createIceServer());
        }

        return list;
    }

    public void mute(final boolean isMute) {
        executor.execute(() -> {
            if (localAudioTrack != null) {
                boolean enabled = !isMute;
                localAudioTrack.setEnabled(enabled);
            }
        });

    }

    public static boolean useCamera2(Context context) {
        return Camera2Enumerator.isSupported(context.getApplicationContext());
    }

    public static VideoCapturer createVideoCapturer(Context context) {
        VideoCapturer videoCapturer;
        if (useCamera2(context)) {
            videoCapturer = createCameraCapturer(new Camera2Enumerator(context.getApplicationContext()));
        } else {
            videoCapturer = createCameraCapturer(new Camera1Enumerator(true));
        }
        return videoCapturer;
    }

    public static VideoCapturer createCameraCapturer(CameraEnumerator enumerator) {
        final String[] deviceNames = enumerator.getDeviceNames();

        // First, try to find front facing camera
        for (String deviceName : deviceNames) {
            if (enumerator.isFrontFacing(deviceName)) {
                VideoCapturer videoCapturer = enumerator.createCapturer(deviceName, null);

                if (videoCapturer != null) {
                    return videoCapturer;
                }
            }
        }

        // Front facing camera not found, try something else
        for (String deviceName : deviceNames) {
            if (!enumerator.isFrontFacing(deviceName)) {
                VideoCapturer videoCapturer = enumerator.createCapturer(deviceName, null);

                if (videoCapturer != null) {
                    return videoCapturer;
                }
            }
        }

        return null;
    }

    public void switchCamera(StatusListener listener) {
        if (videoCapturer != null) {
            if (videoCapturer instanceof CameraVideoCapturer) {
                CameraVideoCapturer cameraVideoCapturer = (CameraVideoCapturer) videoCapturer;
                cameraVideoCapturer.switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean b) {
                        if (listener != null) {
                            listener.onSuccess();
                        }
                    }

                    @Override
                    public void onCameraSwitchError(String s) {
                        if (listener != null) {
                            listener.onError(new StringeeError(-1, s));
                        }
                    }
                });
            }
        }
    }

    public void switchCamera(StatusListener listener, String cameraName) {
        if (videoCapturer != null) {
            if (videoCapturer instanceof CameraVideoCapturer) {
                CameraVideoCapturer cameraVideoCapturer = (CameraVideoCapturer) videoCapturer;
                cameraVideoCapturer.switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean b) {
                        if (listener != null) {
                            listener.onSuccess();
                        }
                    }

                    @Override
                    public void onCameraSwitchError(String s) {
                        if (listener != null) {
                            listener.onError(new StringeeError(-1, s));
                        }
                    }
                }, cameraName);
            }
        }
    }

    public SessionDescription getLocalDescription() {
        return localDescription;
    }

    public void setLocalDescription(SessionDescription localDescription) {
        this.localDescription = localDescription;
    }

    public void processSdp(StringeeCall stringeeCall, JSONObject data) {
        try {
            String sdpType = data.getString("type");
            String sdp = data.getString("sdp");

            StringeeSessionDescription.Type type = StringeeSessionDescription.Type.OFFER;
            if (sdpType.equalsIgnoreCase("0") || sdpType.equalsIgnoreCase("offer")) {
                Log.d("Stringee", "+++++++++++++++++++++++ sdp OFFER received");
            } else if (sdpType.equalsIgnoreCase("1") || sdpType.equalsIgnoreCase("pranswer")) {
                type = StringeeSessionDescription.Type.PRANSWER;
                Log.d("Stringee", "+++++++++++++++++++++++ sdp PRANSWER received");
            } else if (sdpType.equalsIgnoreCase("2") || sdpType.equalsIgnoreCase("answer")) {
                Log.d("Stringee", "+++++++++++++++++++++++ sdp ANSWER received");
                type = StringeeSessionDescription.Type.ANSWER;
            }
            StringeeSessionDescription sessionDescription = new StringeeSessionDescription(type, sdp);
            setRemoteDescription(stringeeCall, sessionDescription);
        } catch (Exception ex) {
            Utils.reportException(StringeeCallFactory.class, ex);
        }
    }

    public void processCandidate(JSONObject data) {
        try {
            String sdpMid = data.getString("sdpMid");
            int sdpMLineIndex = data.getInt("sdpMLineIndex");
            String sdp2 = data.getString("candidate");

            StringeeIceCandidate iceCandidate = new StringeeIceCandidate(sdpMid, sdpMLineIndex, sdp2);
            addIceCandidate(iceCandidate);
        } catch (Exception ex) {
            Utils.reportException(StringeeCallFactory.class, ex);
        }
    }

    public void setLocalDescription(final SdpObserver observer, final SessionDescription sessionDescription) {
        peerConnection.setLocalDescription(observer, sessionDescription);
    }

    public void createAnswer(final SdpObserver observer, final MediaConstraints mediaConstraints) {
        executor.execute(() -> peerConnection.createAnswer(observer, mediaConstraints));
    }

    public void getStats(final StringeeCall.CallStatsListener statsListener) {
        executor.execute(() -> peerConnection.getStats(rtcStatsReport -> {
            StringeeCall.StringeeCallStats stringeeCallStats = new StringeeCall.StringeeCallStats();
            stringeeCallStats.timeStamp = System.currentTimeMillis();
            for (Map.Entry<String, RTCStats> e : rtcStatsReport.getStatsMap().entrySet()) {
                Map<String, Object> members = e.getValue().getMembers();
                if (e.getValue().getType().equals("inbound-rtp") && members.containsKey("kind")) {
                    String kind = (String) members.get("kind");
                    if (kind != null) {
                        BigInteger bytesReceived = (BigInteger) members.get("bytesReceived");
                        Long packetsReceived = (Long) members.get("packetsReceived");
                        Integer packetsLost = (Integer) members.get("packetsLost");

                        if (kind.equals("audio")) {
                            stringeeCallStats.callBytesReceived = bytesReceived != null ? bytesReceived.intValue() : 0;
                            stringeeCallStats.callPacketsReceived = packetsReceived != null ? packetsReceived.intValue() : 0;
                            stringeeCallStats.callPacketsLost = packetsLost != null ? packetsLost : 0;
                        }
                        if (kind.equals("video")) {
                            stringeeCallStats.videoBytesReceived = bytesReceived != null ? bytesReceived.intValue() : 0;
                            stringeeCallStats.videoPacketsReceived = packetsReceived != null ? packetsReceived.intValue() : 0;
                            stringeeCallStats.videoPacketsLost = packetsLost != null ? packetsLost : 0;
                        }
                    }
                }
            }
            statsListener.onCallStats(stringeeCallStats);
        }));
    }

    public void enableVideo(final boolean enabled) {
        if (localVideoTrack != null) {
            localVideoTrack.setEnabled(enabled);
        } else {
            if (enabled) {
                upgradeVideo(true, true);
            }
        }
    }

    public void upgradeVideo(boolean createOffer, boolean isVideo) {
        mStringeeCall.setRootContext(rootEglBaseContext);

        for (int i = 0; i < rtpSenders.size(); i++) {
            RtpSender rtpSender = rtpSenders.get(i);
            peerConnection.removeTrack(rtpSender);
        }
        rtpSenders.clear();
        localMediaStream.dispose();

        localMediaStream = sFactory.createLocalMediaStream("ARDAMS" + System.currentTimeMillis());

        localAudioTrack = sFactory.createAudioTrack(AUDIO_TRACK_ID + System.currentTimeMillis(), audioSource);
        boolean enabledAudio = !mStringeeCall.isMute();
        localAudioTrack.setEnabled(enabledAudio);
        localMediaStream.addTrack(localAudioTrack);

        if (isVideo) {
            videoCapturer = createVideoCapturer(mContext);
            surfaceTextureHelper = SurfaceTextureHelper.create("CaptureThread", rootEglBaseContext);
            videoSource = sFactory.createVideoSource(videoCapturer.isScreencast());
            int minWidth = VideoDimensions.WVGA_VIDEO_WIDTH;
            int minHeight = VideoDimensions.WVGA_VIDEO_HEIGHT;
            VideoQuality videoQuality = mStringeeCall.getVideoQuality();
            if (videoQuality != null) {
                switch (mStringeeCall.getVideoQuality()) {
                    case QUALITY_288P:
                        minWidth = VideoDimensions.CIF_VIDEO_WIDTH;
                        minHeight = VideoDimensions.CIF_VIDEO_HEIGHT;
                        break;
                    case QUALITY_720P:
                        minWidth = VideoDimensions.HD_720P_VIDEO_WIDTH;
                        minHeight = VideoDimensions.HD_720P_VIDEO_HEIGHT;
                        break;
                    case QUALITY_1080P:
                        minWidth = VideoDimensions.HD_1080P_VIDEO_WIDTH;
                        minHeight = VideoDimensions.HD_1080P_VIDEO_HEIGHT;
                        break;
                }
            }
            videoCapturer.initialize(surfaceTextureHelper, mContext, new CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean b) {
                    videoSource.getCapturerObserver().onCapturerStarted(b);
                    if (captureSessionListener != null) {
                        captureSessionListener.onCapturerStarted();
                    }
                }

                @Override
                public void onCapturerStopped() {
                    videoSource.getCapturerObserver().onCapturerStopped();
                    if (captureSessionListener != null) {
                        captureSessionListener.onCapturerStopped();
                    }
                }

                @Override
                public void onFrameCaptured(VideoFrame videoFrame) {
                    videoSource.getCapturerObserver().onFrameCaptured(videoFrame);
                    if (snapshotListener != null) {
                        if (snapshotFrame != null) {
                            return;
                        }
                        snapshotFrame = videoFrame;

                        Bitmap bitmap = Utils.generateBitmap(snapshotFrame);
                        Bitmap emptyBitmap = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), bitmap.getConfig());
                        if (!bitmap.sameAs(emptyBitmap)) {
                            snapshotListener.onSuccess(bitmap);
                        } else {
                            snapshotListener.onError(new StringeeError(1, "Snapshot false"));
                        }
                        snapshotListener = null;
                        snapshotFrame = null;
                    }
                }
            });
            videoCapturer.startCapture(minWidth, minHeight, 30);
            localVideoTrack = sFactory.createVideoTrack(VIDEO_TRACK_ID + System.currentTimeMillis(), videoSource);
            localVideoTrack.setEnabled(true);
            localMediaStream.addTrack(localVideoTrack);

            mStringeeCall.setVideoCall(true);
            mStringeeCall.setLocalStream(localMediaStream);
        }
        List<String> mediaStreamLabels = Collections.singletonList("ARDAMS");
        if (!Utils.isEmpty(localMediaStream.audioTracks)) {
            List<AudioTrack> audioTracks = localMediaStream.audioTracks;
            for (int i = 0; i < audioTracks.size(); i++) {
                AudioTrack audioTrack = audioTracks.get(i);
                RtpSender rtpSender = peerConnection.addTrack(audioTrack, mediaStreamLabels);
                rtpSenders.add(rtpSender);
            }
        }
        if (!Utils.isEmpty(localMediaStream.videoTracks)) {
            List<VideoTrack> videoTracks = localMediaStream.videoTracks;
            for (int i = 0; i < videoTracks.size(); i++) {
                VideoTrack videoTrack = videoTracks.get(i);
                RtpSender rtpSender = peerConnection.addTrack(videoTrack, mediaStreamLabels);
                rtpSenders.add(rtpSender);
            }
        }

        StringeeCall.StringeeCallListener callListener = mStringeeCall.getCallListener();
        if (callListener != null) {
            callListener.onLocalStream(mStringeeCall);
        }

        if (createOffer) {
            sdpMediaConstraints = new MediaConstraints();
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "true"));
            if (isVideo) {
                sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "true"));
            } else {
                sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "false"));
            }
            final StringeeCreateSdpObserver createSdpObserver = new StringeeCreateSdpObserver(true, mStringeeCall);
            peerConnection.createOffer(createSdpObserver, sdpMediaConstraints);
        }
    }

    public void hold(boolean hold) {
        for (int i = 0; i < rtpSenders.size(); i++) {
            RtpSender rtpSender = rtpSenders.get(i);
            peerConnection.removeTrack(rtpSender);
        }
        rtpSenders.clear();
        localMediaStream.dispose();

        localMediaStream = sFactory.createLocalMediaStream("ARDAMS" + System.currentTimeMillis());

        localAudioTrack = sFactory.createAudioTrack(AUDIO_TRACK_ID + System.currentTimeMillis(), audioSource);
        localAudioTrack.setEnabled(true);
        localMediaStream.addTrack(localAudioTrack);

        mStringeeCall.setLocalStream(localMediaStream);

        StringeeCall.StringeeCallListener callListener = mStringeeCall.getCallListener();
        if (callListener != null) {
            callListener.onLocalStream(mStringeeCall);
        }

        List<String> mediaStreamLabels = Collections.singletonList("ARDAMS");
        if (!Utils.isEmpty(localMediaStream.audioTracks)) {
            List<AudioTrack> audioTracks = localMediaStream.audioTracks;
            for (int i = 0; i < audioTracks.size(); i++) {
                AudioTrack audioTrack = audioTracks.get(i);
                RtpSender rtpSender = peerConnection.addTrack(audioTrack, mediaStreamLabels);
                rtpSenders.add(rtpSender);
            }
        }
        if (!Utils.isEmpty(localMediaStream.videoTracks)) {
            List<VideoTrack> videoTracks = localMediaStream.videoTracks;
            for (int i = 0; i < videoTracks.size(); i++) {
                VideoTrack videoTrack = videoTracks.get(i);
                RtpSender rtpSender = peerConnection.addTrack(videoTrack, mediaStreamLabels);
                rtpSenders.add(rtpSender);
            }
        }

        sdpMediaConstraints = new MediaConstraints();
        sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "true"));
        sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "false"));

        int holdType;
        if (hold) {
            holdType = 1;
        } else {
            holdType = 2;
        }
        final StringeeCreateSdpObserver createSdpObserver = new StringeeCreateSdpObserver(true, mStringeeCall, holdType);
        peerConnection.createOffer(createSdpObserver, sdpMediaConstraints);
    }

    public void restartICE() {
        if (sdpMediaConstraints == null) {
            sdpMediaConstraints = new MediaConstraints();
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveAudio", "true"));
            if (mStringeeCall.isVideoCall()) {
                sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "true"));
            } else {
                sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("OfferToReceiveVideo", "false"));
            }
        }

        sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair("IceRestart", "true"));
        mStringeeCall.setRemoteSdpSet(false);
        mStringeeCall.setLocalSdpSet(false);
        StringeeCreateSdpObserver createSdpObserver = new StringeeCreateSdpObserver(true, mStringeeCall);
        peerConnection.createOffer(createSdpObserver, sdpMediaConstraints);
    }

    public void snapshot(CallbackListener<Bitmap> snapshotListener) {
        this.snapshotListener = snapshotListener;
    }
}
