package com.stringee.common;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.Point;
import android.media.ThumbnailUtils;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.opengl.GLES20;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.provider.MediaStore;
import android.util.Base64;
import android.util.Log;
import android.util.Size;
import android.view.Display;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.exifinterface.media.ExifInterface;

import com.android.volley.Request;
import com.android.volley.toolbox.JsonObjectRequest;
import com.stringee.messaging.Message.Type;
import com.stringee.messaging.User;
import com.stringee.video.VideoFrameDrawer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.GlRectDrawer;
import org.webrtc.GlTextureFrameBuffer;
import org.webrtc.GlUtil;
import org.webrtc.VideoFrame;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Created by luannguyen on 6/14/2017.
 */

public class Utils {

    public static Point getDisplaySize(Context context) {
        Display display = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
        return Utils.getDisplaySize(display);
    }

    public static Point getDisplaySize(Display display) {
        Point point = new Point();
        display.getSize(point);
        return point;
    }

    public static String getMetaDataValue(Context context, String metaDataName) {
        try {
            ApplicationInfo ai = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            if (ai.metaData != null) {
                return ai.metaData.getString(metaDataName);

            }
        } catch (PackageManager.NameNotFoundException e) {
            reportException(Utils.class, e);
            return null;
        }
        return null;
    }

    public static String getThumbnail(String filePath, Type type) {
        if (isEmpty(filePath)) {
            return "";
        }
        String timeStamp = new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.getDefault()).format(new Date());
        String thumbFileName = "thumbnail_" + timeStamp + ".jpeg";
        int index = 0;
        for (int i = filePath.length() - 1; i >= 0; i--) {
            char c = filePath.charAt(i);
            if (c == '/') {
                index = i;
                break;
            }
        }
        String dirPath = filePath.substring(0, index) + "/.Thumbnail";
        File thumbDir = new File(dirPath);
        if (!thumbDir.exists()) {
            thumbDir.mkdir();
        }
        File thumbnailFile = new File(dirPath, thumbFileName);
        FileOutputStream out = null;
        if (type == Type.VIDEO) {
            try {
                Bitmap thumbnail;
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    File videoFile = new File(filePath);
                    Size size = new Size(512, 384);
                    thumbnail = ThumbnailUtils.createVideoThumbnail(videoFile, size, null);
                } else {
                    thumbnail = ThumbnailUtils.createVideoThumbnail(filePath, MediaStore.Video.Thumbnails.MINI_KIND);
                }
                out = new FileOutputStream(thumbnailFile);
                if (thumbnail != null) {
                    thumbnail.compress(Bitmap.CompressFormat.PNG, 100, out);
                }
            } catch (Exception e) {
                reportException(Utils.class, e);
            } finally {
                try {
                    if (out != null) {
                        out.close();
                    }
                } catch (IOException e) {
                    reportException(Utils.class, e);
                }
            }
        } else if (type == Type.PHOTO) {
            float ratio = getRatio(filePath);
            int width;
            int height;
            if (ratio > 1) {
                width = 96;
                height = (int) (width / ratio);
            } else {
                height = 96;
                width = (int) (ratio * height);
            }

            try {
                Bitmap image = BitmapFactory.decodeFile(filePath);
                Bitmap thumbnail = ThumbnailUtils.extractThumbnail(image, width, height);
                out = new FileOutputStream(thumbnailFile);
                thumbnail.compress(Bitmap.CompressFormat.PNG, 100, out);
            } catch (Exception e) {
                reportException(Utils.class, e);
            } finally {
                try {
                    if (out != null) {
                        out.close();
                    }
                } catch (IOException e) {
                    reportException(Utils.class, e);
                }
            }
        }
        return thumbnailFile.getAbsolutePath();
    }

    public static float getRatio(String thumbnail) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(thumbnail, options);
        int imageHeight = options.outHeight;
        int imageWidth = options.outWidth;
        return (float) imageWidth / (float) imageHeight;
    }

    public static void generateImage(String filePath) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(filePath, options);
        int imageHeight = options.outHeight;
        int imageWidth = options.outWidth;
        int width;
        int height;
        if (imageWidth > 1280 || imageHeight > 1280) {
            float ratio = (float) imageWidth / (float) imageHeight;
            if (imageWidth > imageHeight) {
                width = 1280;
                height = (int) (width / ratio);
            } else {
                height = 1280;
                width = (int) (height * ratio);
            }
        } else {
            width = imageWidth;
            height = imageHeight;
        }
        Bitmap bitmap = BitmapFactory.decodeFile(filePath);
        Bitmap newBitmap = Bitmap.createScaledBitmap(bitmap, width, height, true);
        newBitmap = rotate(newBitmap, filePath);
        try (FileOutputStream out = new FileOutputStream(filePath)) {
            newBitmap.compress(Bitmap.CompressFormat.JPEG, 75, out);
        } catch (IOException e) {
            reportException(Utils.class, e);
        }
    }

    private static Bitmap rotate(Bitmap bitmap, String path) {
        try {
            // Find if the picture is rotated
            ExifInterface exif = new ExifInterface(path);
            int orientation = exif.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_UNDEFINED);

            Matrix matrix = new Matrix();
            switch (orientation) {
                case ExifInterface.ORIENTATION_FLIP_HORIZONTAL:
                    matrix.setScale(-1, 1);
                    break;
                case ExifInterface.ORIENTATION_ROTATE_180:
                    matrix.setRotate(180);
                    break;
                case ExifInterface.ORIENTATION_FLIP_VERTICAL:
                    matrix.setRotate(180);
                    matrix.postScale(-1, 1);
                    break;
                case ExifInterface.ORIENTATION_TRANSPOSE:
                    matrix.setRotate(90);
                    matrix.postScale(-1, 1);
                    break;
                case ExifInterface.ORIENTATION_ROTATE_90:
                    matrix.setRotate(90);
                    break;
                case ExifInterface.ORIENTATION_TRANSVERSE:
                    matrix.setRotate(-90);
                    matrix.postScale(-1, 1);
                    break;
                case ExifInterface.ORIENTATION_ROTATE_270:
                    matrix.setRotate(-90);
                    break;
                default:
                    return bitmap;
            }

            int w = bitmap.getWidth();
            int h = bitmap.getHeight();
            Bitmap bitmapRotated = Bitmap.createBitmap(bitmap, 0, 0, w, h, matrix, true);
            bitmap.recycle();
            return bitmapRotated;
        } catch (Exception e) {
            reportException(Utils.class, e);
        }

        return bitmap;
    }

    public static void uploadStatsToServer(Context context, String token, String callId, String deviceId, JSONObject content, long firstTime, long lastTime) {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("content", content);
            jsonObject.put("call_id", callId);
            jsonObject.put("device_id", deviceId);
        } catch (JSONException e) {
            reportException(Utils.class, e);
        }
        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, APIUrlUtils.getInstance().getUploadStatsUrl(), jsonObject, response -> {
            Log.d("Stringee", response.toString());
            try {
                int r = response.getInt("r");
                if (r == 0) {
                    String strStats = PrefUtils.getInstance(context).getString(Constant.STATS);
                    if (!isEmpty(strStats)) {
                        int index = 0;
                        JSONArray jsonArray = new JSONArray(strStats);
                        for (int i = 0; i < jsonArray.length(); i++) {
                            JSONObject o = jsonArray.getJSONObject(i);
                            String key = o.keys().next();
                            if (key.equals(callId)) {
                                index = i;
                                break;
                            }
                        }

                        JSONObject statsObject = jsonArray.getJSONObject(index);
                        if (statsObject != null) {
                            JSONArray statsArray = statsObject.optJSONArray(callId);
                            if (!isEmpty(statsArray)) {
                                JSONArray tempArray = new JSONArray();
                                for (int i = 0; i < statsArray.length(); i++) {
                                    JSONObject object = statsArray.getJSONObject(i);
                                    String k = object.keys().next();
                                    long t = Long.parseLong(k);
                                    if (t > lastTime || t < firstTime) {
                                        tempArray.put(object);
                                    }
                                }
                                if (tempArray.length() == 0) {
                                    JSONArray newArray = new JSONArray();
                                    for (int i = 0; i < jsonArray.length(); i++) {
                                        if (i != index) {
                                            newArray.put(jsonArray.getJSONObject(i));
                                        }
                                    }
                                    if (!Utils.isEmpty(newArray)) {
                                        PrefUtils.getInstance(context).putString(Constant.STATS, newArray.toString());
                                    } else {
                                        PrefUtils.getInstance(context).remove(Constant.STATS);
                                    }
                                } else {
                                    statsObject.put(callId, tempArray);
                                    jsonArray.put(index, statsObject);
                                    PrefUtils.getInstance(context).putString(Constant.STATS, jsonArray.toString());
                                }
                            }
                        }
                    }
                }
            } catch (JSONException e) {
                reportException(Utils.class, e);
            }
        }, error -> {
        }) {
            @Override
            public Map<String, String> getHeaders() {
                Map<String, String> params = new HashMap<>();
                params.put("X-STRINGEE-AUTH", token);
                return params;
            }
        };
        VolleyUtils.getInstance(context).add(request);
    }

    public static void uploadStatsArray(Context context, String token, String callId, String deviceId, JSONArray statsArray) {
        try {
            JSONObject contentObject = new JSONObject();
            contentObject.put("stats", statsArray);
            contentObject.put("callId", callId);
            contentObject.put("platform", 2);
            // Upload stats here
            JSONObject firstStat = statsArray.getJSONObject(0);
            long firstTime = Long.parseLong(firstStat.keys().next());
            JSONObject lastStat = statsArray.getJSONObject(statsArray.length() - 1);
            long lastTime = Long.parseLong(lastStat.keys().next());
            Utils.uploadStatsToServer(context, token, callId, deviceId, contentObject, firstTime, lastTime);
        } catch (JSONException e) {
            reportException(Utils.class, e);
        }
    }

    public static String limitBandwidth(String sdp, int audioBandwidth, int videoBandwidth) {
        if (sdp == null) {
            return "";
        }
        StringBuilder result = new StringBuilder();
        int audioIndex = 0;
        int videoIndex = 0;
        String[] lines = sdp.split("\r\n");
        for (int i = 0; i < lines.length; i++) {
            String line = lines[i];
            if (audioBandwidth > 0 && line.startsWith("m=audio")) {
                audioIndex = i + 1;
                if (audioIndex < lines.length) {
                    for (int j = audioIndex; j < lines.length; j++) {
                        if (!(lines[j].startsWith("i=") || lines[j].startsWith("c="))) {
                            audioIndex = j;
                            break;
                        }
                    }
                }
                if (videoBandwidth == 0) {
                    break;
                }
            }
            if (videoBandwidth > 0 && line.startsWith("m=video")) {
                videoIndex = i + 1;
                if (videoIndex < lines.length) {
                    for (int j = videoIndex; j < lines.length; j++) {
                        if (!(lines[j].startsWith("i=") || lines[j].startsWith("c="))) {
                            videoIndex = j;
                            break;
                        }
                    }
                }
                if (audioBandwidth == 0) {
                    break;
                }
            }
        }
        if (audioIndex > 0 && videoIndex > 0) {
            boolean audioFirst = true;
            int startIndex = audioIndex;
            int endIndex = videoIndex;
            if (audioIndex > videoIndex) {
                audioFirst = false;
                startIndex = videoIndex;
                endIndex = audioIndex;
            }
            for (int i = 0; i < startIndex; i++) {
                result.append(lines[i]).append("\r\n");
            }
            if (startIndex < lines.length) {
                int nextIndex = startIndex;
                if (lines[startIndex].startsWith("b=AS:")) {
                    nextIndex = startIndex + 1;
                }
                if (audioFirst) {
                    result.append("b=AS:").append(audioBandwidth).append("\r\n");
                } else {
                    result.append("b=AS:").append(videoBandwidth).append("\r\n");
                }
                for (int i = nextIndex; i < endIndex; i++) {
                    result.append(lines[i]).append("\r\n");
                }
                if (endIndex < lines.length) {
                    int nextEndIndex = endIndex;
                    if (lines[endIndex].startsWith("b=AS:")) {
                        nextEndIndex = endIndex + 1;
                    }
                    if (audioFirst) {
                        result.append("b=AS:").append(videoBandwidth).append("\r\n");
                    } else {
                        result.append("b=AS:").append(audioBandwidth).append("\r\n");
                    }
                    for (int i = nextEndIndex; i < lines.length - 1; i++) {
                        result.append(lines[i]).append("\r\n");
                    }
                    result.append(lines[lines.length - 1]);
                } else {
                    if (audioFirst) {
                        result.append("b=AS:").append(videoBandwidth);
                    } else {
                        result.append("b=AS:").append(audioBandwidth);
                    }
                }
            } else {
                if (audioFirst) {
                    result.append("b=AS:").append(audioBandwidth);
                } else {
                    result.append("b=AS:").append(videoBandwidth);
                }
            }
        } else {
            if (audioIndex > 0) {
                for (int i = 0; i < audioIndex; i++) {
                    result.append(lines[i]).append("\r\n");
                }
                if (audioIndex < lines.length) {
                    int nextIndex = audioIndex;
                    if (lines[audioIndex].startsWith("b=AS:")) {
                        nextIndex = audioIndex + 1;
                    }
                    result.append("b=AS:").append(audioBandwidth).append("\r\n");
                    for (int i = nextIndex; i < lines.length - 1; i++) {
                        result.append(lines[i]).append("\r\n");
                    }
                    result.append(lines[lines.length - 1]);
                } else {
                    result.append("b=AS:").append(audioBandwidth);
                }
            }

            if (videoIndex > 0) {
                for (int i = 0; i < videoIndex; i++) {
                    result.append(lines[i]).append("\r\n");
                }
                if (videoIndex < lines.length) {
                    int nextIndex = videoIndex;
                    if (lines[videoIndex].startsWith("b=AS:")) {
                        nextIndex = videoIndex + 1;
                    }
                    result.append("b=AS:").append(videoBandwidth).append("\r\n");
                    for (int i = nextIndex; i < lines.length - 1; i++) {
                        result.append(lines[i]).append("\r\n");
                    }
                    result.append(lines[lines.length - 1]);
                } else {
                    result.append("b=AS:").append(videoBandwidth);
                }
            }
        }

        return result.toString();
    }

    public static void post(Runnable runnable) {
        post(runnable, 0);
    }

    public static void post(Runnable runnable, long delayMillis) {
        Handler handler = new Handler(Looper.getMainLooper());
        handler.postDelayed(runnable, delayMillis);
    }

    public static boolean isEmpty(@Nullable Object object) {
        if (object == null) {
            return true;
        }
        if (object instanceof CharSequence) {
            return object.toString().trim().isEmpty();
        } else if (object instanceof List) {
            return ((List<?>) object).isEmpty();
        } else if (object instanceof JSONArray) {
            return ((JSONArray) object).length() == 0;
        } else {
            return false;
        }
    }

    public static String decodeToBase64String(Bitmap bitmap) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.JPEG, 70, outputStream);
        return Base64.encodeToString(outputStream.toByteArray(), Base64.NO_WRAP);
    }

    public static Bitmap generateBitmap(VideoFrame frame) {
        Matrix drawMatrix = new Matrix();
        drawMatrix.reset();
        drawMatrix.preTranslate(0.5f, 0.5f);
        drawMatrix.preScale(1f, -1f); // We want the output to be upside down for Bitmap.
        drawMatrix.preTranslate(-0.5f, -0.5f);

        final int scaledWidth = frame.getRotatedWidth();
        final int scaledHeight = frame.getRotatedHeight();

        GlTextureFrameBuffer bitmapTextureFramebuffer = new GlTextureFrameBuffer(GLES20.GL_RGBA);
        bitmapTextureFramebuffer.setSize(scaledWidth, scaledHeight);

        GLES20.glBindFramebuffer(GLES20.GL_FRAMEBUFFER, bitmapTextureFramebuffer.getFrameBufferId());
        GLES20.glFramebufferTexture2D(GLES20.GL_FRAMEBUFFER, GLES20.GL_COLOR_ATTACHMENT0, GLES20.GL_TEXTURE_2D, bitmapTextureFramebuffer.getTextureId(), 0);

        GLES20.glClearColor(0 /* red */, 0 /* green */, 0 /* blue */, 0 /* alpha */);
        GLES20.glClear(GLES20.GL_COLOR_BUFFER_BIT);
        VideoFrameDrawer frameDrawer = new VideoFrameDrawer();
        frameDrawer.drawFrame(frame, new GlRectDrawer(), drawMatrix, 0 /* viewportX */, 0 /* viewportY */, scaledWidth, scaledHeight, false);

        final ByteBuffer bitmapBuffer = ByteBuffer.allocateDirect(scaledWidth * scaledHeight * 4);
        GLES20.glViewport(0, 0, scaledWidth, scaledHeight);
        GLES20.glReadPixels(0, 0, scaledWidth, scaledHeight, GLES20.GL_RGBA, GLES20.GL_UNSIGNED_BYTE, bitmapBuffer);

        GLES20.glBindFramebuffer(GLES20.GL_FRAMEBUFFER, 0);
        GlUtil.checkNoGLES2Error("Stringee.generateBitmap");

        final Bitmap bitmap = Bitmap.createBitmap(scaledWidth, scaledHeight, Bitmap.Config.ARGB_8888);
        bitmap.copyPixelsFromBuffer(bitmapBuffer);
        return bitmap;
    }

    public static User fillNewUserInfo(User user, User newUser) {
        if (!isEmpty(newUser.getName())) {
            user.setName(newUser.getName());
        }
        if (!isEmpty(newUser.getEmail())) {
            user.setEmail(newUser.getEmail());
        }
        if (!isEmpty(newUser.getPhone())) {
            user.setPhone(newUser.getPhone());
        }
        if (!isEmpty(newUser.getAvatarUrl())) {
            user.setAvatarUrl(newUser.getAvatarUrl());
        }
        if (!isEmpty(newUser.getLocation())) {
            user.setLocation(newUser.getLocation());
        }
        if (!isEmpty(newUser.getBrowser())) {
            user.setBrowser(newUser.getBrowser());
        }
        if (!isEmpty(newUser.getPlatform())) {
            user.setPlatform(newUser.getPlatform());
        }
        if (!isEmpty(newUser.getDevice())) {
            user.setDevice(newUser.getDevice());
        }
        if (!isEmpty(newUser.getIpAddress())) {
            user.setIpAddress(newUser.getIpAddress());
        }
        if (!isEmpty(newUser.getHostName())) {
            user.setHostName(newUser.getHostName());
        }
        if (!isEmpty(newUser.getUserAgent())) {
            user.setUserAgent(newUser.getUserAgent());
        }
        return user;
    }

    @SuppressLint("MissingPermission")
    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetInfo = connectivityManager.getActiveNetworkInfo();
        return activeNetInfo != null && activeNetInfo.isConnected();
    }

    public static String getServiceId(String connectToken) {
        String serviceId = null;
        String[] parts = connectToken.split("[.]");
        try {
            for (int i = 0; i < 2; i++) {
                String part = parts[i];
                byte[] partAsBytes;
                String decodedPart;
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    partAsBytes = part.getBytes(StandardCharsets.UTF_8);
                    decodedPart = new String(Base64.decode(partAsBytes, Base64.DEFAULT), StandardCharsets.UTF_8);
                } else {
                    partAsBytes = part.getBytes("UTF-8");
                    decodedPart = new String(Base64.decode(partAsBytes, Base64.DEFAULT), "UTF-8");
                }
                JSONObject jsonObject = new JSONObject(decodedPart);
                if (jsonObject.has("serviceId")) {
                    serviceId = jsonObject.optString("serviceId");
                    break;
                }
            }
        } catch (JSONException | UnsupportedEncodingException e) {
            reportException(Utils.class, e);
        }
        return serviceId;
    }

    public static String getUserId(String connectToken) {
        if (isEmpty(connectToken)) {
            return null;
        }
        String userId = null;
        String[] parts = connectToken.split("[.]");
        try {
            for (int i = 0; i < 2; i++) {
                String part = parts[i];
                byte[] partAsBytes;
                String decodedPart;
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    partAsBytes = part.getBytes(StandardCharsets.UTF_8);
                    decodedPart = new String(Base64.decode(partAsBytes, Base64.DEFAULT), StandardCharsets.UTF_8);
                } else {
                    partAsBytes = part.getBytes("UTF-8");
                    decodedPart = new String(Base64.decode(partAsBytes, Base64.DEFAULT), "UTF-8");
                }
                JSONObject jsonObject = new JSONObject(decodedPart);
                if (jsonObject.has("userId")) {
                    userId = jsonObject.optString("userId");
                    break;
                }
            }
        } catch (JSONException | IllegalArgumentException | UnsupportedEncodingException e) {
            reportException(Utils.class, e);
        }
        return userId;
    }

    public static <T> void reportException(@NonNull Class<T> clazz, Exception exception) {
        Logger.getLogger(clazz.getName()).log(Level.SEVERE, null, exception);
    }

    public static String getDeviceId(Context context) {
        SharedPreferences preferences = context.getSharedPreferences(Constant.PREF_NAME, Context.MODE_PRIVATE);
        String deviceId = preferences.getString(Constant.DEVICE_ID, null);
        if (deviceId == null) {
            deviceId = "android-" + UUID.randomUUID().toString();
            SharedPreferences.Editor editor = preferences.edit();
            editor.putString(Constant.DEVICE_ID, deviceId);
            editor.apply();
        }
        return deviceId;
    }

    /**
     * Helper method which throws an exception  when an assertion has failed.
     */
    public static void assertIsTrue(boolean condition) throws AssertionError {
        if (!condition) {
            throw new AssertionError("Expected condition to be true");
        }
    }

    /**
     * Helper method for building a string of thread information.
     */
    public static String getThreadInfo() {
        return "@[name=" + Thread.currentThread().getName() + ", id=" + Thread.currentThread().getId() + "]";
    }

    /**
     * Information about the current build, taken from system properties.
     */
    public static void logDeviceInfo(String tag) {
        Log.d(tag, "Android SDK: " + Build.VERSION.SDK_INT + ", " + "Release: " + Build.VERSION.RELEASE + ", " + "Brand: " + Build.BRAND + ", " + "Device: " + Build.DEVICE + ", " + "Id: " + Build.ID + ", " + "Hardware: " + Build.HARDWARE + ", " + "Manufacturer: " + Build.MANUFACTURER + ", " + "Model: " + Build.MODEL + ", " + "Product: " + Build.PRODUCT);
    }
}
