package com.stringee.network.processor;

import android.content.Context;

import androidx.core.util.PatternsCompat;

import com.stringee.StringeeClient;
import com.stringee.call.StringeeIceServer;
import com.stringee.common.Common;
import com.stringee.common.Constant;
import com.stringee.common.NetworkCommon;
import com.stringee.common.PrefUtils;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.SocketAddress;
import com.stringee.common.Utils;
import com.stringee.common.event.EventManager;
import com.stringee.database.DBHandler;
import com.stringee.exception.StringeeError;
import com.stringee.messaging.ChatRequest;
import com.stringee.messaging.User;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.network.tcpclient.CheckTimeOutThread;
import com.stringee.network.tcpclient.IoHandler;
import com.stringee.network.tcpclient.PacketSenderThread;
import com.stringee.network.tcpclient.TcpClient;
import com.stringee.network.tcpclient.packet.Packet;
import com.stringee.video.StringeeVideo;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;

public class Authenticate extends ProcessorBase {

    @Override
    public void process(StringeeClient client, Packet packet) {
        client.getExecutor().execute(() -> {
            packet.decodeJson();
            final int r;
            try {
                r = packet.getFieldInt("r");
                if (r == 0) {
                    client.setConnected(true);
                    client.getErrorTokens().clear();
                    JSONObject userInfo = new JSONObject();
                    String userId = packet.getFieldString("userId");
                    String displayName = packet.optFieldString("displayName");
                    int projectId = packet.optFieldInt("projectId", 0);
                    int sessionId = packet.optFieldInt("connectionId");
                    boolean chatAgent = packet.optFieldBoolean("chatAgent");
                    userInfo.put("userId", userId);
                    userInfo.put("connectionId", sessionId);
                    JSONArray iceArray = packet.optFieldJSONArray("ice_servers");
                    if (!Utils.isEmpty(iceArray)) {
                        LinkedList<StringeeIceServer> iceServers = new LinkedList<>();
                        for (int i = 0; i < iceArray.length(); i++) {
                            JSONObject jsonObject = iceArray.getJSONObject(i);
                            StringeeIceServer iceServer = new StringeeIceServer(jsonObject.getString("urls"), jsonObject.getString("username"), jsonObject.getString("credential"));
                            iceServers.add(iceServer);
                        }
                        StringeeVideo.setIceServers(iceServers);
                    }

                    client.setUserId(userId);
                    client.setProjectId(projectId);
                    client.setUserInfo(userInfo);

                    client.getDbExecutor().execute(() -> {
                        DBHandler dbHandler = DBHandler.getInstance(client.getContext());
                        User user = dbHandler.getUser(userId);
                        if (user == null) {
                            user = new User(userId);
                        }
                        if (!Utils.isEmpty(displayName)) {
                            user.setName(displayName);
                        }
                        dbHandler.syncUser(user);
                    });

                    String clientIp = packet.optFieldString("clientIp");
                    if (!Utils.isEmpty(clientIp)) {
                        Matcher matcher = PatternsCompat.IP_ADDRESS.matcher(clientIp);
                        while (matcher.find()) {
                            int start = matcher.start();
                            int end = matcher.end();
                            clientIp = clientIp.substring(start, end);
                            client.setClientIp(clientIp);
                        }
                    }

                    PacketSenderThread senderThread = client.getPacketSenderThread();
                    if (senderThread != null) {
                        senderThread.sendAllPacketInQueue();
                    }

                    // Retry to register push token
                    Context context = client.getContext();
                    String packageName = context.getPackageName();
                    String name = Constant.PUSH_REGISTER + "_" + packageName;
                    String data = PrefUtils.getInstance(context).getString(name);
                    if (!Utils.isEmpty(data)) {
                        try {
                            JSONObject jsonObject = new JSONObject(data);
                            String status = jsonObject.optString(Constant.STATUS);
                            if (!status.equals(Constant.REGISTERED)) {
                                String token = jsonObject.getString(Constant.TOKEN);
                                int requestId;
                                synchronized (Common.lock) {
                                    requestId = ++Common.requestId;
                                }
                                SendPacketUtils.registerPushToken(client, token, packageName, requestId);
                            }
                        } catch (JSONException e) {
                            Utils.reportException(Authenticate.class, e);
                        }
                    }

                    // Retry to unregister push token
                    String unregisterData = PrefUtils.getInstance(context).getString(Constant.PUSH_UNREGISTER);
                    if (unregisterData != null) {
                        try {
                            JSONArray jsonArray = new JSONArray(unregisterData);
                            if (!Utils.isEmpty(jsonArray)) {
                                for (int i = 0; i < jsonArray.length(); i++) {
                                    JSONObject jsonObject = jsonArray.getJSONObject(i);
                                    String token = jsonObject.getString(Constant.TOKEN);
                                    String userId2 = jsonObject.getString(Constant.USER_ID);
                                    int projectId2 = jsonObject.getInt(Constant.PROJECT_ID);
                                    int requestId;
                                    synchronized (Common.lock) {
                                        requestId = ++Common.requestId;
                                    }
                                    SendPacketUtils.unregisterPushToken(client, token, packageName, userId2, projectId2, requestId);
                                }
                            }
                        } catch (JSONException e) {
                            Utils.reportException(Authenticate.class, e);
                        }
                    }

                    if (!client.isAlreadyAuthenticated()) {
                        client.setAlreadyAuthenticated(true);
                        EventManager.sendClientConnectedEvent(client, false);
                        CheckTimeOutThread checkTimeOutThread = client.getCheckTimeOutThread();
                        if (!checkTimeOutThread.isRunning()) {
                            checkTimeOutThread.start();
                        }
                    } else {
                        EventManager.sendClientConnectedEvent(client, true);
                    }

                    if (chatAgent) {
                        getChatRequest(client);
                    }

                    String strStats = PrefUtils.getInstance(context).getString(Constant.STATS);
                    if (!Utils.isEmpty(strStats)) {
                        JSONArray jsonArray = new JSONArray(strStats);
                        if (!Utils.isEmpty(jsonArray)) {
                            for (int i = 0; i < jsonArray.length(); i++) {
                                JSONObject jsonObject = jsonArray.getJSONObject(i);
                                String callId = jsonObject.keys().next();
                                JSONArray statsArray = jsonObject.optJSONArray(callId);
                                if (!Utils.isEmpty(statsArray)) {
                                    if (statsArray.length() <= 60) {
                                        Utils.uploadStatsArray(context, client.getToken(), callId, Utils.getDeviceId(context), statsArray);
                                    } else {
                                        JSONArray uploadArray = new JSONArray();
                                        for (int j = 0; j < statsArray.length(); j++) {
                                            uploadArray.put(statsArray.getJSONObject(j));
                                            if ((j > 0 && j % 60 == 0) || (j == (statsArray.length() - 1))) {
                                                Utils.uploadStatsArray(context, client.getToken(), callId, Utils.getDeviceId(context), uploadArray);
                                                uploadArray = new JSONArray();
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                } else {
                    TcpClient tcpClient = client.getTcpClient();
                    if (tcpClient != null) {
                        IoHandler ioHandler = (IoHandler) tcpClient.getHandler();
                        if (ioHandler != null) {
                            ioHandler.setNeedReconnect(false);
                        }
                        tcpClient.disconnect();
                    }
                    if (r == 15) {
                        String strChangeServer = packet.optFieldString("change_to_server", "");
                        if (!Utils.isEmpty(strChangeServer)) {
                            NetworkCommon.addRetriedAddress();
                            try {
                                URI uri = new URI(strChangeServer);
                                NetworkCommon.changeServer = new SocketAddress(uri.getHost(), uri.getPort(), uri.getHost());
                                NetworkCommon.currentServer = NetworkCommon.changeServer;
                                client.setAddressIndex(-1);
                                client.setConnectRetry(0);

                                String v4ServerIp = "";
                                String v6ServerIp = "";
                                String serverIp = NetworkCommon.changeServer.getIp();

                                InetAddress[] machines;
                                try {
                                    machines = InetAddress.getAllByName(serverIp);
                                    if (machines != null) {
                                        for (InetAddress address : machines) {
                                            if (address != null) {
                                                if (address instanceof Inet4Address) {
                                                    v4ServerIp = address.getHostAddress();
                                                } else if (address instanceof Inet6Address) {
                                                    v6ServerIp = address.getHostAddress();
                                                }
                                            }
                                        }
                                    }
                                } catch (Exception e) {
                                    Utils.reportException(Authenticate.class, e);
                                }

                                boolean isIpv4Support = false;
                                try {
                                    Enumeration<NetworkInterface> n = NetworkInterface.getNetworkInterfaces();
                                    if (n != null) {
                                        while (n.hasMoreElements()) { //for each interface
                                            NetworkInterface e = n.nextElement();
                                            Enumeration<InetAddress> a = e.getInetAddresses();
                                            if (a != null) {
                                                while (a.hasMoreElements()) {
                                                    InetAddress addr = a.nextElement();
                                                    String add = addr.getHostAddress();
                                                    if (add != null && !addr.isLoopbackAddress() && add.length() < 17) {
                                                        isIpv4Support = true;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                } catch (Exception e) {
                                    Utils.reportException(Authenticate.class, e);
                                }

                                if (isIpv4Support) {
                                    if (v4ServerIp != null && !v4ServerIp.isEmpty()) {
                                        serverIp = v4ServerIp;
                                    }
                                } else {
                                    if (v6ServerIp != null && !v6ServerIp.isEmpty()) {
                                        serverIp = v6ServerIp;
                                    }
                                }
                                IoHandler ioHandler = new IoHandler(client);
                                TcpClient newTcpClient = new TcpClient(ioHandler, serverIp, NetworkCommon.changeServer.getPort(), client.getTcpClient().isUseSsl(), NetworkCommon.changeServer.getSslAcceptedIP(), client.getCertificates(), client.isTrustAll(), client.getPublicKeys());
                                client.setTcpClient(newTcpClient);
                                PacketSenderThread senderThread = client.getPacketSenderThread();
                                if (senderThread != null) {
                                    senderThread.setTcpClient(newTcpClient);
                                }
                                CheckTimeOutThread timeOutThread = client.getCheckTimeOutThread();
                                if (timeOutThread != null) {
                                    timeOutThread.setTcpClient(newTcpClient);
                                }
                                try {
                                    newTcpClient.start();
                                } catch (Exception e) {
                                    Utils.reportException(Authenticate.class, e);
                                }
                            } catch (URISyntaxException e) {
                                Utils.reportException(Authenticate.class, e);
                            }
                        } else {
                            client.setAlreadyConnected(false);
                        }
                        EventManager.sendClientErrorEvent(client, new StringeeError(r, "Current server is slave."));
                    } else {
                        client.setAlreadyConnected(false);
                        switch (r) {
                            case 1:
                            case 2:
                            case 3:
                            case 4:
                            case 5:
                            case 6:
                            case 7:
                            case 8:
                            case 10:
                                client.getErrorTokens().add(client.getToken());
                                break;
                        }
                        String error = "Authentication failed.";
                        switch (r) {
                            case 1:
                                error = "Access token is missing.";
                                break;
                            case 2:
                                error = "Access token can not be decoded.";
                                break;
                            case 3:
                                error = "Access token format is invalid.";
                                break;
                            case 4:
                                error = "HS256 must be used to generate the access token.";
                                break;
                            case 5:
                                error = "API key is not found.";
                                break;
                            case 7:
                                error = "Access token payload is incorrect.";
                                break;
                            case 8:
                                error = "Project is not found.";
                                break;
                            case 10:
                                error = "Invalid signature.";
                                break;
                            case 11:
                                error = "Account expired.";
                                break;
                            case 12:
                                error = "Account deactivated.";
                                break;
                            case 6:
                                error = "Access token is expired or invalid.";
                                EventManager.sendClientNewTokenEvent(client);
                                break;
                        }
                        EventManager.sendClientErrorEvent(client, new StringeeError(r, error));
                    }
                }
            } catch (JSONException e) {
                Utils.reportException(Authenticate.class, e);
            }
        });
    }

    private void getChatRequest(StringeeClient client) {
        client.getChatRequests(new CallbackListener<List<ChatRequest>>() {
            @Override
            public void onSuccess(List<ChatRequest> chatRequestList) {
                for (int i = 0; i <= chatRequestList.size(); i++) {
                    EventManager.sendChatSupportReceiceEvent(client, chatRequestList.get(i));
                }
            }
        });
    }
}
