package com.stringee.network.tcpclient;

import android.util.Log;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.NetworkCommon;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.SocketAddress;
import com.stringee.common.Utils;
import com.stringee.common.event.EventManager;
import com.stringee.exception.StringeeError;
import com.stringee.network.processor.ProcessorBase;
import com.stringee.network.tcpclient.packet.Packet;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.logging.Level;
import java.util.logging.Logger;

public class IoHandler extends HandlerBase {

    private boolean needReconnect = true;
    private int minDelayReconnect = 500;
    private int maxDelayReconnect = 2000;
    private int currentDelayReconnect;
    private boolean retriedAllServer = false;
    private static final String TAG = "Stringee";
    private final StringeeClient client;

    public IoHandler(StringeeClient client) {
        currentDelayReconnect = minDelayReconnect;
        this.client = client;
    }

    @Override
    protected void onMessageReceived(Packet packet, TcpClient tcpClient) {
        synchronized (Common.tcpClientLock) {
            if (tcpClient == null) {
                return;
            }
        }
        tcpClient.setLastPacketReceivedTime(System.currentTimeMillis());
        Log.d(TAG, "Stringee " + packet);
        ProcessorBase p = ProcessorBase.getProcessor(packet.getService());
        if (p != null) {
            try {
                p.process(client, packet);
            } catch (Exception ex) {
                Utils.reportException(IoHandler.class, ex);
                Log.d(TAG, "Error in processor: " + p);
            }
        } else {
            Log.d(TAG, "Could not found processor for: " + packet);
        }
    }

    @Override
    protected void onConnected(TcpClient tcpClient) {
        Log.d(TAG, "onConnected");
        currentDelayReconnect = minDelayReconnect;
        client.setConnectRetry(0);
        SendPacketUtils.login(client);
        NetworkCommon.retriedAddresses.clear();
        retriedAllServer = false;
    }

    @Override
    protected void onDisconnected(final TcpClient tcpClient) {
        synchronized (Common.tcpClientLock) {
            if (tcpClient == null) {
                return;
            }
            Log.d(TAG, "onDisconnected");
            client.setConnected(false);
            EventManager.sendClientDisconnectedEvent(client, needReconnect);
            if (client.getConnectRetry() >= 2) {
                NetworkCommon.addRetriedAddress();
                if (NetworkCommon.retriedAddresses.size() == NetworkCommon.addresses.size() && !retriedAllServer) {
                    retriedAllServer = true;
                    EventManager.sendClientErrorEvent(client, new StringeeError(-100, "Failed to connect after retrying all servers"));
                }
            }
            if (needReconnect) {
                Log.d(TAG, "Trying reconnect after " + currentDelayReconnect + " ms");
                try {
                    Thread.sleep(currentDelayReconnect);
                } catch (InterruptedException ex) {
                    Logger.getLogger(IoHandler.class.getName()).log(Level.SEVERE, null, ex);
                }
                currentDelayReconnect = currentDelayReconnect * 2;
                if (currentDelayReconnect > maxDelayReconnect) {
                    currentDelayReconnect = maxDelayReconnect;
                }

                // Connect
                int addressIndex = client.getAddressIndex();
                if (client.getConnectRetry() >= 2) {
                    if (addressIndex == -1) {
                        addressIndex = 0;
                    } else if (addressIndex <= (NetworkCommon.addresses.size() - 2)) {
                        addressIndex++;
                    } else {
                        addressIndex = 0;
                    }
                    client.setAddressIndex(addressIndex);
                    client.setConnectRetry(-1);
                }
                SocketAddress socketAddress;
                if (addressIndex == -1) {
                    socketAddress = NetworkCommon.changeServer;
                } else {
                    socketAddress = NetworkCommon.addresses.get(addressIndex);
                }
                NetworkCommon.currentServer = socketAddress;
                String serverIp = socketAddress.getIp();
                int serverPort = socketAddress.getPort();
                String sslAcceptedIp = socketAddress.getSslAcceptedIP();
                String v4ServerIp = "";
                String v6ServerIp = "";

                InetAddress[] machines;
                try {
                    machines = InetAddress.getAllByName(serverIp);
                    if (machines != null) {
                        for (InetAddress address : machines) {
                            if (address != null) {
                                if (address instanceof Inet4Address) {
                                    v4ServerIp = address.getHostAddress();
                                } else if (address instanceof Inet6Address) {
                                    v6ServerIp = address.getHostAddress();
                                }
                            }
                        }
                    }
                } catch (Exception e) {
                    Utils.reportException(IoHandler.class, e);
                }

                boolean isIpv4Support = false;
                try {
                    Enumeration<NetworkInterface> n = NetworkInterface.getNetworkInterfaces();
                    if (n != null) {
                        while (n.hasMoreElements()) { //for each interface
                            NetworkInterface e = n.nextElement();
                            Enumeration<InetAddress> a = e.getInetAddresses();
                            if (a != null) {
                                while (a.hasMoreElements()) {
                                    InetAddress inetAddress = a.nextElement();
                                    String add = inetAddress.getHostAddress();
                                    if (add != null && !inetAddress.isLoopbackAddress() && add.length() < 17) {
                                        isIpv4Support = true;
                                    }
                                }
                            }
                        }
                    }
                } catch (Exception e) {
                    Utils.reportException(IoHandler.class, e);
                }

                if (isIpv4Support) {
                    if (!Utils.isEmpty(v4ServerIp)) {
                        serverIp = v4ServerIp;
                    }
                } else {
                    if (!Utils.isEmpty(v6ServerIp)) {
                        serverIp = v6ServerIp;
                    }
                }
                TcpClient newTcpClient = new TcpClient(IoHandler.this, serverIp, serverPort, tcpClient.isUseSsl(), sslAcceptedIp, client.getCertificates(), client.isTrustAll(), client.getPublicKeys());
                client.setTcpClient(newTcpClient);
                PacketSenderThread senderThread = client.getPacketSenderThread();
                if (senderThread != null) {
                    senderThread.setTcpClient(newTcpClient);
                }
                CheckTimeOutThread timeOutThread = client.getCheckTimeOutThread();
                if (timeOutThread != null) {
                    timeOutThread.setTcpClient(newTcpClient);
                }
                try {
                    newTcpClient.start();
                } catch (Exception ex) {
                    Utils.reportException(IoHandler.class, ex);
                }
                int retry = client.getConnectRetry() + 1;
                client.setConnectRetry(retry);
            }
        }
    }

    public boolean isNeedReconnect() {
        return needReconnect;
    }

    public void setNeedReconnect(boolean needReconnect) {
        this.needReconnect = needReconnect;
    }

    public int getMinDelayReconnect() {
        return minDelayReconnect;
    }

    public void setMinDelayReconnect(int minDelayReconnect) {
        this.minDelayReconnect = minDelayReconnect;
    }

    public int getMaxDelayReconnect() {
        return maxDelayReconnect;
    }

    public void setMaxDelayReconnect(int maxDelayReconnect) {
        this.maxDelayReconnect = maxDelayReconnect;
    }

    @Override
    protected void onWriteCompleted(Packet packet, TcpClient tcpClient) {
    }

    @Override
    protected void onUnacceptedCertificate() {
        synchronized (Common.tcpClientLock) {
            Log.d(TAG, "onUnacceptedCertificate");
            EventManager.sendClientErrorEvent(client, new StringeeError(-101, "Certificate is invalid"));
        }
    }
}
