package com.stringee;

import android.content.Context;
import android.os.Build;
import android.util.Log;

import com.android.volley.Request.Method;
import com.android.volley.toolbox.JsonObjectRequest;
import com.stringee.call.StringeeCall;
import com.stringee.call.StringeeCall2;
import com.stringee.call.StringeeCallData;
import com.stringee.common.APIUrlUtils;
import com.stringee.common.Common;
import com.stringee.common.Constant;
import com.stringee.common.NetworkCommon;
import com.stringee.common.PrefUtils;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.SocketAddress;
import com.stringee.common.Utils;
import com.stringee.common.VolleyUtils;
import com.stringee.common.event.EventManager;
import com.stringee.database.DBHandler;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StatusListener;
import com.stringee.listener.StringeeConnectionListener;
import com.stringee.messaging.ChannelType;
import com.stringee.messaging.ChatProfile;
import com.stringee.messaging.ChatRequest;
import com.stringee.messaging.Conversation;
import com.stringee.messaging.ConversationFilter;
import com.stringee.messaging.ConversationOptions;
import com.stringee.messaging.Message;
import com.stringee.messaging.Queue;
import com.stringee.messaging.StringeeChange;
import com.stringee.messaging.User;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.messaging.listeners.ChangeEventListener;
import com.stringee.messaging.listeners.LiveChatEventListener;
import com.stringee.messaging.listeners.UserTypingEventListener;
import com.stringee.network.tcpclient.CheckTimeOutThread;
import com.stringee.network.tcpclient.IoHandler;
import com.stringee.network.tcpclient.PacketSenderThread;
import com.stringee.network.tcpclient.StringeeCertificate;
import com.stringee.network.tcpclient.TcpClient;
import com.stringee.network.tcpclient.packet.Packet;
import com.stringee.video.StringeeRoom;
import com.stringee.video.StringeeScreenCapture;
import com.stringee.video.StringeeVideoTrack;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Created by luannguyen on 10/12/2017.
 */

public class StringeeClient {
    private List<StringeeConnectionListener> connectionListeners = new ArrayList<>();
    private List<ChangeEventListener> changeEventListeners = new ArrayList<>();
    private List<UserTypingEventListener> userTypingListeners = new ArrayList<>();
    private List<LiveChatEventListener> liveChatListeners = new ArrayList<>();

    private TcpClient tcpClient;
    private PacketSenderThread packetSenderThread;
    private CheckTimeOutThread checkTimeOutThread;
    private boolean isConnected;
    private boolean alreadyConnected;
    private Context context;
    private String token;
    private boolean alreadyInRoom;
    private ScheduledExecutorService executor;
    private ScheduledExecutorService dbExecutor;
    private ExecutorService listenerExecutor;
    private String userId;
    private int projectId;
    private JSONObject userInfo;
    private String sessionId;
    private int connectRetry;
    private int addressIndex;
    private boolean alreadyAuthenticated;
    private boolean needGetAddress;
    private int currentDelayReconnect = 500;
    private static final int maxDelayReconnect = 5000;
    private String clientIp = "";
    private List<StringeeCertificate> stringeeCertificates;
    private boolean trustAll;
    private String serviceId;
    private List<String> publicKeys;
    private final Set<String> errorTokens = new HashSet<>();
    private String tokenGetAddress = "";

    // Call data queues
    private final ConcurrentHashMap<Integer, StringeeCall> callRequest = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Integer, StringeeCall2> callRequest2 = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, StringeeCall> callMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, StringeeCall2> callMap2 = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Packet> callInPacket = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, LinkedBlockingQueue<StringeeCallData>> candidatesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, StringeeCallData> sdpMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Integer, Message> messagesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Integer, StringeeVideoTrack> localVideoTrackMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, StringeeScreenCapture> screenCaptureMap = new ConcurrentHashMap<>();

    // Room queue
    private final ConcurrentHashMap<Integer, StringeeRoom> roomRequest = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, StringeeRoom> roomMap = new ConcurrentHashMap<>();

    // Chat request
    private final ConcurrentHashMap<String, ChatRequest> chatRequestMap = new ConcurrentHashMap<>();

    public StringeeClient(Context context) {
        this.context = context.getApplicationContext();
        packetSenderThread = new PacketSenderThread(StringeeClient.this);
        checkTimeOutThread = new CheckTimeOutThread();
    }

    /**
     * Using {@link #addConnectionListener(StringeeConnectionListener)} instead
     */
    @Deprecated
    public void setConnectionListener(StringeeConnectionListener listener) {
        addConnectionListener(listener);
    }

    public List<StringeeConnectionListener> getConnectionListeners() {
        if (Utils.isEmpty(connectionListeners)) {
            connectionListeners = new ArrayList<>();
        }
        return connectionListeners;
    }

    public void addConnectionListener(StringeeConnectionListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getConnectionListeners().add(listener));
    }

    public void removeConnectionListener(StringeeConnectionListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getConnectionListeners().remove(listener));
    }

    /**
     * Using {@link #addChangeEventListener(ChangeEventListener)} instead
     */
    @Deprecated
    public void setChangeEventListener(ChangeEventListener listener) {
        addChangeEventListener(listener);
    }

    public List<ChangeEventListener> getChangeEventListeners() {
        if (Utils.isEmpty(changeEventListeners)) {
            changeEventListeners = new ArrayList<>();
        }
        return changeEventListeners;
    }

    public void addChangeEventListener(ChangeEventListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getChangeEventListeners().add(listener));
    }

    public void removeChangeEventListener(ChangeEventListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getChangeEventListeners().remove(listener));
    }

    /**
     * Using {@link #addUserTypingEventListener(UserTypingEventListener)} instead
     */
    @Deprecated
    public void setUserTypingEventListener(UserTypingEventListener listener) {
        addUserTypingEventListener(listener);
    }

    public List<UserTypingEventListener> getUserTypingListeners() {
        if (Utils.isEmpty(userTypingListeners)) {
            userTypingListeners = new ArrayList<>();
        }
        return userTypingListeners;
    }

    public void addUserTypingEventListener(UserTypingEventListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getUserTypingListeners().add(listener));
    }

    public void removeUserTypingEventListener(UserTypingEventListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getUserTypingListeners().remove(listener));
    }

    /**
     * Using {@link #addLiveChatEventListener(LiveChatEventListener)} instead
     */
    @Deprecated
    public void setLiveChatEventListener(LiveChatEventListener listener) {
        addLiveChatEventListener(listener);
    }

    public List<LiveChatEventListener> getLiveChatListeners() {
        if (Utils.isEmpty(liveChatListeners)) {
            liveChatListeners = new ArrayList<>();
        }
        return liveChatListeners;
    }

    public void addLiveChatEventListener(LiveChatEventListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getLiveChatListeners().add(listener));
    }

    public void removeLiveChatEventListener(LiveChatEventListener listener) {
        if (listener == null) {
            return;
        }
        getListenerExecutor().execute(() -> getLiveChatListeners().remove(listener));
    }

    public PacketSenderThread getPacketSenderThread() {
        return packetSenderThread;
    }

    public CheckTimeOutThread getCheckTimeOutThread() {
        return checkTimeOutThread;
    }

    public boolean isConnected() {
        return isConnected;
    }

    public void setConnected(boolean connected) {
        isConnected = connected;
    }

    public TcpClient getTcpClient() {
        return tcpClient;
    }

    public void setTcpClient(TcpClient tcpClient) {
        this.tcpClient = tcpClient;
    }

    public boolean isAlreadyConnected() {
        return alreadyConnected;
    }

    public void setAlreadyConnected(boolean alreadyConnected) {
        this.alreadyConnected = alreadyConnected;
    }

    public boolean isAlreadyAuthenticated() {
        return alreadyAuthenticated;
    }

    public void setAlreadyAuthenticated(boolean alreadyAuthenticated) {
        this.alreadyAuthenticated = alreadyAuthenticated;
    }

    public Context getContext() {
        return context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public Set<String> getErrorTokens() {
        return errorTokens;
    }

    public boolean isAlreadyInRoom() {
        return alreadyInRoom;
    }

    public void setAlreadyInRoom(boolean alreadyInRoom) {
        this.alreadyInRoom = alreadyInRoom;
    }

    public ConcurrentHashMap<String, StringeeRoom> getRoomMap() {
        return roomMap;
    }

    public ConcurrentHashMap<Integer, StringeeCall> getCallRequest() {
        return callRequest;
    }

    public ConcurrentHashMap<Integer, StringeeCall2> getCallRequest2() {
        return callRequest2;
    }

    public ConcurrentHashMap<String, StringeeCall> getCallMap() {
        return callMap;
    }

    public ConcurrentHashMap<String, StringeeCall2> getCallMap2() {
        return callMap2;
    }

    public ConcurrentHashMap<String, Packet> getCallInPacket() {
        return callInPacket;
    }

    public ConcurrentHashMap<String, LinkedBlockingQueue<StringeeCallData>> getCandidatesMap() {
        return candidatesMap;
    }

    public ConcurrentHashMap<String, StringeeCallData> getSdpMap() {
        return sdpMap;
    }

    public ConcurrentHashMap<Integer, Message> getMessagesMap() {
        return messagesMap;
    }

    public ConcurrentHashMap<Integer, StringeeRoom> getRoomRequest() {
        return roomRequest;
    }

    public ConcurrentHashMap<Integer, StringeeVideoTrack> getLocalVideoTrackMap() {
        return localVideoTrackMap;
    }

    public ConcurrentHashMap<String, StringeeScreenCapture> getScreenCaptureMap() {
        return screenCaptureMap;
    }

    public ConcurrentHashMap<String, ChatRequest> getChatRequestMap() {
        return chatRequestMap;
    }

    public ScheduledExecutorService getExecutor() {
        if (executor == null || executor.isShutdown() || executor.isTerminated()) {
            executor = Executors.newSingleThreadScheduledExecutor();
        }
        return executor;
    }

    public ScheduledExecutorService getDbExecutor() {
        if (dbExecutor == null || dbExecutor.isShutdown() || dbExecutor.isTerminated()) {
            dbExecutor = Executors.newSingleThreadScheduledExecutor();
        }
        return dbExecutor;
    }

    public ExecutorService getListenerExecutor() {
        if (listenerExecutor == null || listenerExecutor.isShutdown() || listenerExecutor.isTerminated()) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                listenerExecutor = Executors.newWorkStealingPool();
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                listenerExecutor = new ForkJoinPool(Runtime.getRuntime().availableProcessors(), ForkJoinPool.defaultForkJoinWorkerThreadFactory, null, true);
            } else {
                listenerExecutor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
            }
        }
        return listenerExecutor;
    }

    public String getUserId() {
        return userId;
    }

    public void setUserId(String userId) {
        this.userId = userId;
    }

    public int getProjectId() {
        return projectId;
    }

    public void setProjectId(int projectId) {
        this.projectId = projectId;
    }

    public JSONObject getUserInfo() {
        return userInfo;
    }

    public void setUserInfo(JSONObject userInfo) {
        this.userInfo = userInfo;
    }

    public String getSessionId() {
        return sessionId;
    }

    public void setSessionsId(String sessionId) {
        this.sessionId = sessionId;
    }

    public int getConnectRetry() {
        return connectRetry;
    }

    public void setConnectRetry(int connectRetry) {
        this.connectRetry = connectRetry;
    }

    public int getAddressIndex() {
        return addressIndex;
    }

    public void setAddressIndex(int addressIndex) {
        this.addressIndex = addressIndex;
    }

    public String getClientIp() {
        return clientIp;
    }

    public void setClientIp(String clientIp) {
        this.clientIp = clientIp;
    }

    public void enableSSLSpinning(List<StringeeCertificate> stringeeCertificates) {
        this.stringeeCertificates = stringeeCertificates;
    }

    public void enableSSLSpinningWithPublicKeys(List<String> publicKeys) {
        this.publicKeys = publicKeys;
    }

    public List<StringeeCertificate> getCertificates() {
        return stringeeCertificates;
    }

    public boolean isTrustAll() {
        return trustAll;
    }

    public void setTrustAllSsl(boolean trustAll) {
        this.trustAll = trustAll;
    }

    public void setServiceId(String serviceId) {
        this.serviceId = serviceId;
    }

    public List<String> getPublicKeys() {
        return publicKeys;
    }

    /**
     * Connect to Stringee server
     */
    public void connect(final String connectToken) {
        getExecutor().execute(() -> {
            if (errorTokens.contains(connectToken)) {
                Log.w("Stringee", "Your access token has expired or is invalid | " + token);
                return;
            }
            tokenGetAddress = connectToken;
            if (alreadyConnected) {
                tokenGetAddress = null;
                return;
            }
            alreadyConnected = true;

            String serviceId = Utils.getServiceId(connectToken);
            if (!Utils.isEmpty(serviceId)) {
                setServiceId(serviceId);
            }
            String userId = Utils.getUserId(connectToken);
            if (!Utils.isEmpty(userId)) {
                setUserId(userId);
            }
            if (NetworkCommon.addresses.isEmpty()) {
                getServerAddress();
            } else {
                startConnect(connectToken);
            }
        });
    }

    /**
     * Disconnect from Stringee server
     */
    public void disconnect() {
        getExecutor().execute(() -> {
            callMap.clear();
            callMap2.clear();
            callRequest.clear();
            callInPacket.clear();
            roomRequest.clear();
            roomMap.clear();
            currentDelayReconnect = 500;
            needGetAddress = false;
            synchronized (Common.tcpClientLock) {
                if (packetSenderThread != null) {
                    packetSenderThread.setRunning(false);
                    packetSenderThread = null;
                }
                if (checkTimeOutThread != null) {
                    checkTimeOutThread.setRunning(false);
                    checkTimeOutThread = null;
                }

                if (tcpClient != null) {
                    IoHandler ioHandler = (IoHandler) tcpClient.getHandler();
                    if (ioHandler != null) {
                        ioHandler.setNeedReconnect(false);
                    }
                    if (tcpClient.isConnected()) {
                        tcpClient.disconnect();
                    }
                    tcpClient = null;
                }
            }
            alreadyConnected = false;
            alreadyAuthenticated = false;
            if (executor != null) {
                if (!executor.isShutdown() || !executor.isTerminated()) {
                    executor.shutdown();
                }
            }
            if (dbExecutor != null) {
                if (!dbExecutor.isShutdown() || !dbExecutor.isTerminated()) {
                    dbExecutor.shutdown();
                }
            }
            if (listenerExecutor != null) {
                if (!listenerExecutor.isShutdown() || !listenerExecutor.isTerminated()) {
                    listenerExecutor.shutdown();
                }
            }
        });
    }

    /**
     * Add call data to queue
     *
     * @param id       String
     * @param callData StringeeCallData
     */
    public void addCallDataToQueue(String id, StringeeCallData callData) {
        String type = callData.getType();
        if (type.equals("sdp")) {
            sdpMap.put(id, callData);
        } else {
            LinkedBlockingQueue<StringeeCallData> callDataQueue = candidatesMap.get(id);
            if (callDataQueue == null) {
                callDataQueue = new LinkedBlockingQueue<>();
                callDataQueue.add(callData);
                candidatesMap.put(id, callDataQueue);
            } else {
                callDataQueue.add(callData);
            }
        }
    }

    /**
     * Register a token for push notification to Stringee Server
     *
     * @param token String
     */
    public void registerPushToken(final String token, final StatusListener listener) {
        if (!isConnected) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeClient not connected."));
            }
            return;
        }
        if (Utils.isEmpty(token)) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Device token can not empty"));
            }
            return;
        }
        String packageName = context.getPackageName();
        String key = Constant.PUSH_REGISTER + "_" + packageName;
        PrefUtils.getInstance(context).savePushToken(key, token, Constant.REGISTERING);
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.registerPushToken(this, token, packageName, requestId);
    }

    /**
     * Register a token for push notification to Stringee Server and delete other token
     *
     * @param token        String
     * @param packageNames List<String>
     */
    public void registerPushTokenAndDeleteOthers(final String token, final List<String> packageNames, final StatusListener listener) {
        if (!isConnected) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeClient not connected."));
            }
            return;
        }
        if (Utils.isEmpty(token)) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Device token can not empty"));
            }
            return;
        }

        if (Utils.isEmpty(packageNames)) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Package names can not empty"));
            }
            return;
        }
        String packageName = context.getPackageName();
        String key = Constant.PUSH_REGISTER + "_" + packageName;
        PrefUtils.getInstance(context).savePushToken(key, token, Constant.REGISTERING);
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.registerPushTokenAndDeleteOthers(this, token, packageName, packageNames, requestId);
    }

    /**
     * Delete a push notification token from Stringee Server
     *
     * @param token String
     */
    public void unregisterPushToken(final String token, final StatusListener listener) {
        if (!isConnected) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeClient not connected."));
            }
            return;
        }
        if (Utils.isEmpty(token)) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Device token can not empty"));
            }
            return;
        }
        String packageName = context.getPackageName();
        String username = packageName + "." + projectId + "." + userId;
        PrefUtils.getInstance(context).saveUnregisteredToken(username, token, userId, projectId);
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.unregisterPushToken(this, token, packageName, userId, projectId, requestId);
    }

    /**
     * Send custom message to a user
     *
     * @param toUser   String
     * @param msg      JSONObject
     * @param listener CallbackListener
     */
    public void sendCustomMessage(final String toUser, final JSONObject msg, final StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendCustomMessage(StringeeClient.this, requestId, toUser, msg);
    }

    /**
     * Change an attribute
     *
     * @param name     String
     * @param value    String
     * @param listener CallbackListener
     */
    public void changeAttribute(final String name, final String value, final StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.changeAttribute(this, requestId, name, value);
    }

    /**
     * Create a conversation
     *
     * @param participants List<User>
     * @param listener     CallbackListener
     */
    public void createConversation(final List<User> participants, final CallbackListener<Conversation> listener) {
        this.createConversation(participants, null, listener);
    }

    /**
     * Create a conversation with options
     *
     * @param participants List<User>
     * @param options      ConversationOptions
     * @param listener     CallbackListener
     */
    public void createConversation(final List<User> participants, final ConversationOptions options, final CallbackListener<Conversation> listener) {
        if (userId == null) {
            listener.onError(new StringeeError(-1, "StringeeClient has not connected yet."));
            return;
        }

        String name = "";
        boolean isGroup = false;
        boolean isDistinct = false;
        String oaId = "";
        String customData = "";
        String creatorId = "";
        if (options != null) {
            name = options.getName();
            isGroup = options.isGroup();
            isDistinct = options.isDistinct();
            oaId = options.getOaId();
            customData = options.getCustomData();
            creatorId = options.getCreatorId();
        }

        if (name == null) {
            name = "";
        }

        // Standardize participants
        boolean isExists = false;
        for (int i = 0; i < participants.size(); i++) {
            String participantId = participants.get(i).getUserId();
            if (participantId != null && participantId.equals(userId)) {
                isExists = true;
                break;
            }
        }
        if (!isExists) {
            User identity = new User(userId);
            participants.add(identity);
        }

        String localId = Utils.getDeviceId(context) + "-conversation-" + System.currentTimeMillis();
        Conversation conversation = new Conversation();
        conversation.setClientId(userId);
        conversation.setName(name);
        conversation.setGroup(isGroup);
        conversation.setDistinct(isDistinct);
        conversation.setParticipants(participants);
        conversation.setLocalId(localId);
        conversation.setOaId(oaId);
        conversation.setCustomData(customData);
        conversation.setCreator(creatorId);
        conversation.setCreateAt(System.currentTimeMillis());

        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.createConversation(requestId, StringeeClient.this, conversation);
    }

    /**
     * Get local conversations
     *
     * @param clientId String
     * @param listener CallbackListener
     */
    public void getLocalConversations(final String clientId, final CallbackListener<List<Conversation>> listener) {
        this.getLocalConversations(clientId, null, listener);
    }

    /**
     * Get local conversations
     *
     * @param clientId String
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getLocalConversations(final String clientId, final String oaId, final CallbackListener<List<Conversation>> listener) {
        getDbExecutor().execute(() -> {
            if (clientId == null) {
                listener.onError(new StringeeError(-1, "User id is null."));
                return;
            }
            List<Conversation> conversations;

            if (oaId != null) {
                conversations = DBHandler.getInstance(context).getAllOaConversations(clientId, oaId);
            } else {
                conversations = DBHandler.getInstance(context).getAllConversations(clientId);
            }

            if (listener != null) {
                listener.onSuccess(conversations);
            }
        });
    }

    /**
     * Get local conversations by channel type
     *
     * @param clientId     String
     * @param isEnded      boolean
     * @param channelTypes List<Conversation.ChannelType>
     * @param listener     CallbackListener
     */
    public void getLocalConversationsByChannelType(final String clientId, final boolean isEnded, final List<ChannelType> channelTypes, final CallbackListener<List<Conversation>> listener) {
        getDbExecutor().execute(() -> {
            if (clientId == null) {
                listener.onError(new StringeeError(-1, "User id is null."));
                return;
            }

            if (channelTypes == null) {
                listener.onError(new StringeeError(-1, "Channel type is null."));
                return;
            }

            StringBuilder channelType = new StringBuilder();

            for (int i = 0; i < channelTypes.size(); i++) {
                channelType.append(channelTypes.get(i).getValue()).append(",");
            }
            if (!Utils.isEmpty(channelType.toString())) {
                channelType = new StringBuilder(channelType.substring(0, channelType.length() - 1));
            }

            List<Conversation> conversations = DBHandler.getInstance(context).getConversationByChannelType(clientId, isEnded, channelType.toString());

            if (listener != null) {
                listener.onSuccess(conversations);
            }
        });
    }

    /**
     * Get last conversations
     *
     * @param count    count
     * @param listener CallbackListener
     */
    public void getLastConversations(final int count, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        this.getLastConversations(count, conversationFilter, listener);
    }

    /**
     * Get last conversations
     *
     * @param count    count
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getLastConversations(final int count, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        this.getLastConversations(count, conversationFilter, listener);
    }

    /**
     * Get last conversations
     *
     * @param count    count
     * @param loadAll  loadAll
     * @param listener CallbackListener
     */
    public void getLastConversations(final int count, final boolean loadAll, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setDeleted(loadAll);
        this.getLastConversations(count, conversationFilter, listener);
    }

    /**
     * Get last conversations
     *
     * @param count    count
     * @param loadAll  loadAll
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getLastConversations(final int count, final boolean loadAll, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        conversationFilter.setDeleted(loadAll);
        this.getLastConversations(count, conversationFilter, listener);
    }

    /**
     * Get last conversations
     *
     * @param count    count
     * @param filter   ConversationFilter
     * @param listener CallbackListener
     */
    public void getLastConversations(final int count, final ConversationFilter filter, final CallbackListener<List<Conversation>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadConversations(requestId, StringeeClient.this, 0, Long.MAX_VALUE, count, filter.isDeleted(), filter.isUnread(), filter.getOaId(), filter.getChannelTypes(), filter.getFilterChatStatus());
    }

    /**
     * Get latest unread conversations
     *
     * @param count    count
     * @param listener CallbackListener
     */
    public void getLastUnreadConversations(final int count, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        this.getLastUnreadConversations(count, conversationFilter, listener);
    }

    /**
     * Get latest unread conversations
     *
     * @param count    count
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getLastUnreadConversations(final int count, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        this.getLastUnreadConversations(count, conversationFilter, listener);
    }

    /**
     * Get latest unread conversations
     *
     * @param count    count
     * @param filter   ConversationFilter
     * @param listener CallbackListener
     */
    public void getLastUnreadConversations(final int count, final ConversationFilter filter, final CallbackListener<List<Conversation>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadConversations(requestId, StringeeClient.this, 0, Long.MAX_VALUE, count, filter.isDeleted(), filter.isUnread(), filter.getOaId(), filter.getChannelTypes(), filter.getFilterChatStatus());
    }

    /**
     * Get latest conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param listener CallbackListener
     */
    public void getConversationsAfter(final long updateAt, final int count, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        this.getConversationsAfter(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getConversationsAfter(final long updateAt, final int count, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        this.getConversationsAfter(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param loadAll  loadAll
     * @param listener CallbackListener
     */
    public void getConversationsAfter(final long updateAt, final int count, final boolean loadAll, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setDeleted(loadAll);
        this.getConversationsAfter(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param loadAll  loadAll
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getConversationsAfter(final long updateAt, final int count, final boolean loadAll, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        conversationFilter.setDeleted(loadAll);
        this.getConversationsAfter(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param filter   ConversationFilter
     * @param listener CallbackListener
     */
    public void getConversationsAfter(final long updateAt, final int count, final ConversationFilter filter, final CallbackListener<List<Conversation>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadConversations(requestId, StringeeClient.this, updateAt, Long.MAX_VALUE, count, filter.isDeleted(), filter.isUnread(), filter.getOaId(), filter.getChannelTypes(), filter.getFilterChatStatus());
    }

    /**
     * Get latest unread conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param listener CallbackListener
     */
    public void getUnreadConversationsAfter(final long updateAt, final int count, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        this.getUnreadConversationsAfter(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest unread conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getUnreadConversationsAfter(final long updateAt, final int count, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        this.getUnreadConversationsAfter(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest unread conversations with lastUpdate > updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param filter   ConversationFilter
     * @param listener CallbackListener
     */
    public void getUnreadConversationsAfter(final long updateAt, final int count, final ConversationFilter filter, final CallbackListener<List<Conversation>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadConversations(requestId, StringeeClient.this, updateAt, Long.MAX_VALUE, count, filter.isDeleted(), filter.isUnread(), filter.getOaId(), filter.getChannelTypes(), filter.getFilterChatStatus());
    }

    /***
     * Get latest conversations with lastUpdate < updateAt
     * @param updateAt updateAt
     * @param count count
     * @param listener CallbackListener
     */
    public void getConversationsBefore(final long updateAt, final int count, final CallbackListener<List<Conversation>> listener) {
        this.getConversationsBefore(updateAt, count, false, null, listener);
    }

    /***
     * Get latest conversations with lastUpdate < updateAt
     * @param updateAt updateAt
     * @param count count
     * @param oaId oaId
     * @param listener CallbackListener
     */
    public void getConversationsBefore(final long updateAt, final int count, final String oaId, final CallbackListener<List<Conversation>> listener) {
        this.getConversationsBefore(updateAt, count, false, oaId, listener);
    }

    /***
     * Get latest conversations with lastUpdate < updateAt
     * @param updateAt updateAt
     * @param count count
     * @param loadAll loadAll
     * @param listener CallbackListener
     */
    public void getConversationsBefore(final long updateAt, final int count, final boolean loadAll, final CallbackListener<List<Conversation>> listener) {
        this.getConversationsBefore(updateAt, count, loadAll, null, listener);
    }

    /***
     * Get latest conversations with lastUpdate < updateAt
     * @param updateAt updateAt
     * @param count count
     * @param loadAll loadAll
     * @param oaId oaId
     * @param listener CallbackListener
     */
    public void getConversationsBefore(final long updateAt, final int count, final boolean loadAll, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        conversationFilter.setDeleted(loadAll);
        this.getConversationsBefore(updateAt, count, conversationFilter, listener);
    }

    /***
     * Get latest conversations with lastUpdate < updateAt
     * @param updateAt updateAt
     * @param count count
     * @param filter ConversationFilter
     * @param listener CallbackListener
     */
    public void getConversationsBefore(final long updateAt, final int count, final ConversationFilter filter, final CallbackListener<List<Conversation>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadConversations(requestId, StringeeClient.this, 0, updateAt, count, filter.isDeleted(), filter.isUnread(), filter.getOaId(), filter.getChannelTypes(), filter.getFilterChatStatus());
    }

    /**
     * Get latest unread conversations with lastUpdate < updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param listener CallbackListener
     */
    public void getUnreadConversationsBefore(final long updateAt, final int count, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        this.getUnreadConversationsBefore(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest unread conversations with lastUpdate < updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param oaId     oaId
     * @param listener CallbackListener
     */
    public void getUnreadConversationsBefore(final long updateAt, final int count, final String oaId, final CallbackListener<List<Conversation>> listener) {
        ConversationFilter conversationFilter = new ConversationFilter();
        conversationFilter.setOaId(oaId);
        this.getUnreadConversationsBefore(updateAt, count, conversationFilter, listener);
    }

    /**
     * Get latest unread conversations with lastUpdate < updateAt
     *
     * @param updateAt updateAt
     * @param count    count
     * @param filter   ConversationFilter
     * @param listener CallbackListener
     */
    public void getUnreadConversationsBefore(final long updateAt, final int count, final ConversationFilter filter, final CallbackListener<List<Conversation>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadConversations(requestId, StringeeClient.this, 0, updateAt, count, filter.isDeleted(), filter.isUnread(), filter.getOaId(), filter.getChannelTypes(), filter.getFilterChatStatus());
    }

    /**
     * Get conversation by remote id
     *
     * @param convId   Conversation id
     * @param listener CallbackListener
     */
    public void getConversation(final String convId, final CallbackListener<Conversation> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            Conversation conversation = null;
            if (userId != null) {
                conversation = DBHandler.getInstance(context).getConversationByRemoteId(convId, userId);
            }
            if (conversation != null) {
                if (listener != null) {
                    listener.onSuccess(conversation);
                }
            } else {
                SendPacketUtils.getConversation(requestId, StringeeClient.this, convId);
            }
        });
    }

    /**
     * Get conversation from server
     *
     * @param convId   Conversation id
     * @param listener CallbackListener
     */
    public void getConversationFromServer(final String convId, final CallbackListener<Conversation> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.getConversation(requestId, StringeeClient.this, convId);
        });
    }

    /***
     * Get a user by user id
     *
     * @param userId User id
     */
    public User getUser(final String userId) {
        return DBHandler.getInstance(context).getUser(userId);
    }

    /**
     * Clear local database
     */
    public void clearDb() {
        getDbExecutor().execute(() -> DBHandler.getInstance(context).clearDb());
    }

    /**
     * Using the function deleteMessages in Conversation instead
     * Delete messages
     *
     * @param convId     Conversation id
     * @param messageIds JSONArray
     * @param listener   StatusListener
     */
    @Deprecated
    public void deleteMessages(final String convId, final JSONArray messageIds, final StatusListener listener) {
        if (Utils.isEmpty(messageIds)) {
            listener.onError(new StringeeError(-1, "No message id found."));
            return;
        }
        getConversationFromServer(convId, new CallbackListener<Conversation>() {
            @Override
            public void onSuccess(Conversation conversation) {
                conversation.deleteMessages(StringeeClient.this, messageIds, listener);
            }

            @Override
            public void onError(StringeeError errorInfo) {
                super.onError(errorInfo);
                if (listener != null) {
                    listener.onError(errorInfo);
                }
            }
        });
    }

    /**
     * Get all pending requests
     *
     * @param listener CallbackListener
     */
    public void getChatRequests(final CallbackListener<List<ChatRequest>> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.loadChatRequest(this, requestId);
    }

    /**
     * Block user
     *
     * @param userId   User id
     * @param listener StatusListener
     */
    public void blockUser(final String userId, final StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.block(this, requestId, userId, null);
    }

    /**
     * Prevent adding to group
     *
     * @param convId   Conversation id
     * @param listener StatusListener
     */
    public void preventAddingToGroup(final String convId, final StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.block(this, requestId, null, convId);
    }

    /**
     * Update attachment
     *
     * @param message Message
     */
    public void updateAttachment(final Message message) {
        getDbExecutor().execute(() -> {
            DBHandler.getInstance(context).updateFilePath(message);
            EventManager.sendChatChangeEvent(StringeeClient.this, new StringeeChange(StringeeChange.Type.UPDATE, message));
        });
    }

    public int getTotalUnread(String userId) {
        return DBHandler.getInstance(context).getTotalUnread(userId);
    }

    /**
     * Get chat 1-1 by user id
     *
     * @param id       User id
     * @param listener CallbackListener
     */
    public void getConversationByUserId(final String id, final CallbackListener<Conversation> listener) {
        if (userId == null) {
            listener.onError(new StringeeError(-1, "StringeeClient is not connected yet."));
            return;
        }
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        JSONArray jsonArray = new JSONArray();
        jsonArray.put(userId);
        jsonArray.put(id);
        SendPacketUtils.getConversations(this, requestId, jsonArray);
    }

    /**
     * Get total number of unread conversations
     *
     * @param listener CallbackListener
     */
    public void getTotalUnread(final CallbackListener<Integer> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.getTotalUnread(this, requestId);
    }


    /**
     * Get portal's chat profile
     *
     * @param key Key
     */
    public void getChatProfile(final String key, final CallbackListener<ChatProfile> listener) {
        String url = APIUrlUtils.getInstance().getLiveChatProfileUrl() + key;
        JsonObjectRequest request = new JsonObjectRequest(Method.GET, url, null, response -> {
            try {
                int r = response.getInt("r");
                if (r == 0) {
                    ChatProfile chatProfile = new ChatProfile();
                    JSONObject profileObject = response.optJSONObject("chatProfile");
                    if (profileObject != null) {
                        chatProfile.setId(profileObject.optString("id", ""));
                        chatProfile.setPortalId(profileObject.optString("portal", ""));
                        chatProfile.setProjectId(profileObject.getInt("project_id"));
                        chatProfile.setLanguage(profileObject.optString("language", ""));
                        chatProfile.setBackground(profileObject.optString("background", ""));
                        chatProfile.setAutoCreateTicket(profileObject.getInt("auto_create_ticket") == 1);
                        chatProfile.setPopupAnswerUrl(profileObject.optString("popup_answer_url", ""));
                        chatProfile.setNumberOfAgents(profileObject.optInt("agent_max_serve"));
                        chatProfile.setEnabledBusinessHour(profileObject.optInt("enabled", 0) == 1);
                        chatProfile.setBusinessHourId(profileObject.optString("business_hour_id", ""));
                        chatProfile.setBusinessHour(profileObject.optString("hour", ""));
                        chatProfile.setFacebookAsLivechat((profileObject.optInt("facebook_as_livechat", 0)) == 1);
                        chatProfile.setZaloAsLivechat((profileObject.optInt("zalo_as_livechat", 0)) == 1);
                        chatProfile.setLogoUrl(profileObject.optString("logo_url", ""));
                    }
                    JSONArray queuesArray = response.optJSONArray("queues");
                    if (!Utils.isEmpty(queuesArray)) {
                        List<Queue> queues = new ArrayList<>();
                        for (int i = 0; i < queuesArray.length(); i++) {
                            JSONObject queueObject = queuesArray.getJSONObject(i);
                            Queue queue = new Queue();
                            queue.setId(queueObject.optString("id", ""));
                            queue.setName(queueObject.optString("name", ""));
                            queues.add(queue);
                        }
                        chatProfile.setQueues(queues);
                    }
                    if (listener != null) {
                        listener.onSuccess(chatProfile);
                    }
                } else {
                    if (listener != null) {
                        listener.onError(new StringeeError(r, response.optString("msg", "")));
                    }
                }
            } catch (JSONException e) {
                Utils.reportException(StringeeClient.class, e);
                if (listener != null) {
                    listener.onError(new StringeeError(-1, e.getMessage()));
                }
            }
        }, error -> {
            if (listener != null) {
                listener.onError(new StringeeError(-1, error.getMessage()));
            }
        });
        VolleyUtils.getInstance(context).add(request);
    }

    /**
     * Get access token
     *
     * @param key      Key
     * @param name     Name
     * @param email    Email
     * @param listener StatusListener
     */
    public void getLiveChatToken(final String key, final String name, final String email, final CallbackListener<String> listener) {
        String url = APIUrlUtils.getInstance().getLiveChatCreateTokenUrl() + key + "&username=" + name + "&email=" + email + "&userId=" + UUID.randomUUID().toString();
        JsonObjectRequest request = new JsonObjectRequest(Method.GET, url, null, response -> {
            try {
                int r = response.getInt("r");
                if (r == 0) {
                    String token = response.optString("access_token", "");
                    if (listener != null) {
                        listener.onSuccess(token);
                    }
                } else {
                    if (listener != null) {
                        listener.onError(new StringeeError(r, response.optString("msg", "")));
                    }
                }
            } catch (JSONException e) {
                Utils.reportException(StringeeClient.class, e);
                if (listener != null) {
                    listener.onError(new StringeeError(-1, e.getMessage()));
                }
            }
        }, error -> {
            if (listener != null) {
                listener.onError(new StringeeError(-1, error.getMessage()));
            }
        });
        VolleyUtils.getInstance(context).add(request);
    }

    /**
     * Create a live chat conversation
     *
     * @param queueId  queue id
     * @param listener StatusListener
     */
    public void createLiveChat(final String queueId, final CallbackListener<Conversation> listener) {
        this.createLiveChat(queueId, null, listener);
    }

    /**
     * Create a live chat conversation
     *
     * @param queueId    queue id
     * @param customData custom data
     * @param listener   StatusListener
     */
    public void createLiveChat(final String queueId, final String customData, final CallbackListener<Conversation> listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.callbackListenerMap.put(requestId, listener);
        }
        SendPacketUtils.createLiveChat(this, requestId, queueId, customData, serviceId);
    }

    /**
     * Create ticket if not in business hour
     *
     * @param key      Key
     * @param name     Name
     * @param email    Email
     * @param note     Note
     * @param listener StatusListener
     */
    public void createLiveChatTicket(final String key, final String name, final String email, final String note, final StatusListener listener) {
        createLiveChatTicket(key, name, email, "", note, listener);
    }

    /**
     * Create ticket if not in business hour
     *
     * @param key      Key
     * @param name     Name
     * @param email    Email
     * @param phone    Phone number
     * @param note     Note
     * @param listener StatusListener
     */
    public void createLiveChatTicket(final String key, final String name, final String email, final String phone, final String note, final StatusListener listener) {
        String url = APIUrlUtils.getInstance().getLiveChatCreateTicketUrl() + key + "&username=" + name + "&email=" + email + "&userId=" + UUID.randomUUID().toString() + "&note=" + note + "&phone=" + phone;
        JsonObjectRequest request = new JsonObjectRequest(Method.GET, url, null, response -> {
            try {
                int r = response.getInt("r");
                if (r == 0) {
                    if (listener != null) {
                        listener.onSuccess();
                    }
                } else {
                    if (listener != null) {
                        listener.onError(new StringeeError(r, response.optString("msg", "")));
                    }
                }
            } catch (JSONException e) {
                Utils.reportException(StringeeClient.class, e);
                if (listener != null) {
                    listener.onError(new StringeeError(-1, e.getMessage()));
                }
            }

        }, error -> {
            if (listener != null) {
                listener.onError(new StringeeError(-1, error.getMessage()));
            }
        });
        VolleyUtils.getInstance(context).add(request);
    }

    /**
     * Update user info
     *
     * @param name      User
     * @param email     email
     * @param avatarUrl avatar url
     * @param listener  StatusListener
     */
    public void updateUser(final String name, final String email, final String avatarUrl, final StatusListener listener) {
        User user = new User();
        user.setName(name);
        user.setEmail(email);
        user.setAvatarUrl(avatarUrl);
        this.updateUser(user, listener);
    }

    /**
     * Update user info
     *
     * @param name      User
     * @param email     email
     * @param avatarUrl avatar url
     * @param phone     phone number
     * @param listener  StatusListener
     */
    public void updateUser(final String name, final String email, final String avatarUrl, final String phone, final StatusListener listener) {
        User user = new User();
        user.setName(name);
        user.setEmail(email);
        user.setAvatarUrl(avatarUrl);
        user.setPhone(phone);
        this.updateUser(user, listener);
    }

    /**
     * Update user info
     *
     * @param newUser  User
     * @param listener StatusListener
     */
    public void updateUser(final User newUser, final StatusListener listener) {
        List<String> userIds = new ArrayList<>();
        userIds.add(userId);
        getUserInfo(userIds, new CallbackListener<List<User>>() {
            @Override
            public void onSuccess(List<User> users) {
                User user = Utils.fillNewUserInfo(users.get(0), newUser);
                StatusListener statusListener = new StatusListener() {
                    @Override
                    public void onSuccess() {
                        getDbExecutor().execute(() -> {
                            DBHandler dbHandler = DBHandler.getInstance(context);
                            User oldUser = dbHandler.getUser(userId);
                            if (oldUser == null) {
                                dbHandler.insertUser(user);
                            } else {
                                dbHandler.updateUser(user);
                            }
                        });
                        if (listener != null) {
                            listener.onSuccess();
                        }
                    }

                    @Override
                    public void onError(StringeeError errorInfo) {
                        super.onError(errorInfo);
                        if (listener != null) {
                            listener.onError(errorInfo);
                        }
                    }
                };
                int requestId;
                synchronized (Common.lock) {
                    requestId = ++Common.requestId;
                }
                if (listener != null) {
                    Common.statusListenerMap.put(requestId, statusListener);
                }
                SendPacketUtils.updateUser(StringeeClient.this, requestId, user);
            }

            @Override
            public void onError(StringeeError errorInfo) {
                super.onError(errorInfo);
                DBHandler dbHandler = DBHandler.getInstance(context);
                User localUser = dbHandler.getUser(userId);
                if (localUser != null) {
                    User user = Utils.fillNewUserInfo(localUser, newUser);
                    StatusListener statusListener = new StatusListener() {
                        @Override
                        public void onSuccess() {
                            getDbExecutor().execute(() -> {
                                DBHandler dbHandler1 = DBHandler.getInstance(context);
                                User oldUser = dbHandler1.getUser(userId);
                                if (oldUser == null) {
                                    dbHandler1.insertUser(user);
                                } else {
                                    dbHandler1.updateUser(user);
                                }
                            });
                            if (listener != null) {
                                listener.onSuccess();
                            }
                        }

                        @Override
                        public void onError(StringeeError errorInfo) {
                            super.onError(errorInfo);
                            if (listener != null) {
                                listener.onError(errorInfo);
                            }
                        }
                    };
                    int requestId;
                    synchronized (Common.lock) {
                        requestId = ++Common.requestId;
                    }
                    if (listener != null) {
                        Common.statusListenerMap.put(requestId, statusListener);
                    }
                    SendPacketUtils.updateUser(StringeeClient.this, requestId, user);
                } else {
                    if (listener != null) {
                        listener.onError(new StringeeError(-3, "No user found"));
                    }
                }
            }
        });
    }

    /**
     * Using the function revokeMessages in Conversation instead
     * Revoke messages
     *
     * @param msgIds List of message id
     */
    @Deprecated
    public void revokeMessages(final String convId, final JSONArray msgIds, final boolean deleted, final StatusListener listener) {
        if (Utils.isEmpty(msgIds)) {
            listener.onError(new StringeeError(-1, "No message id found."));
            return;
        }
        List<String> idList = new ArrayList<>();
        for (int i = 0; i < msgIds.length(); i++) {
            String id = msgIds.optString(i);
            if (!Utils.isEmpty(id)) {
                idList.add(id);
            }
        }
        if (Utils.isEmpty(idList)) {
            listener.onError(new StringeeError(-1, "No message id found."));
            return;
        }
        getConversationFromServer(convId, new CallbackListener<Conversation>() {
            @Override
            public void onSuccess(Conversation conversation) {
                conversation.revokeMessages(StringeeClient.this, idList, deleted, listener);
            }

            @Override
            public void onError(StringeeError errorInfo) {
                super.onError(errorInfo);
                if (listener != null) {
                    listener.onError(errorInfo);
                }
            }
        });
    }

    public void joinOaConversation(final String convId, final StatusListener listener) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.joinOaConversation(this, requestId, convId);
    }

    /**
     * Set host
     *
     * @param addresses List of SocketAddress
     */
    public void setHost(final List<SocketAddress> addresses) {
        if (!NetworkCommon.addresses.isEmpty()) {
            NetworkCommon.addresses.clear();
        }

        NetworkCommon.addresses.addAll(addresses);
    }

    /**
     * Set API url
     *
     * @param url API url
     */
    public void setBaseAPIUrl(final String url) {
        APIUrlUtils.getInstance().setBaseAPIUrl(url);
    }

    /**
     * Set base StringeeX url
     *
     * @param url StringeeX url
     */
    public void setStringeeXBaseUrl(final String url) {
        APIUrlUtils.getInstance().setStringeeXBaseUrl(url);
    }

    private void getServerAddress() {
        if (Utils.isEmpty(tokenGetAddress)) {
            return;
        }

        if (needGetAddress) {
            try {
                Thread.sleep(currentDelayReconnect);
            } catch (InterruptedException ex) {
                Logger.getLogger(StringeeClient.class.getName()).log(Level.SEVERE, null, ex);
            }
            currentDelayReconnect = currentDelayReconnect * 2;
            if (currentDelayReconnect > maxDelayReconnect) {
                currentDelayReconnect = maxDelayReconnect;
            }
        }

        String url = APIUrlUtils.getInstance().getStringeeServerAddressUrl() + "?access_token=" + tokenGetAddress;
        JsonObjectRequest request = new JsonObjectRequest(Method.GET, url, null, response -> getExecutor().execute(() -> {
            if (!alreadyConnected) {
                return;
            }
            try {
                int r = response.getInt("r");
                if (r == 0) {
                    JSONObject data = response.getJSONObject("data");
                    JSONObject serverAddressObj = data.getJSONObject("stringeeServerAddrs");
                    JSONArray mobileArray = serverAddressObj.getJSONArray("mobile");
                    if (!Utils.isEmpty(mobileArray)) {
                        if (!NetworkCommon.addresses.isEmpty()) {
                            NetworkCommon.addresses.clear();
                        }

                        for (int i = 0; i < mobileArray.length(); i++) {
                            JSONObject mobileObject = mobileArray.getJSONObject(i);
                            String ip = mobileObject.getString("ip");
                            int port = mobileObject.getInt("port");
                            NetworkCommon.addresses.add(new SocketAddress(ip, port));
                        }
                        startConnect(tokenGetAddress);
                    }
                } else {
                    String message = response.optString("message", "");
                    catchGetServerAddressError(r, message);
                }
                tokenGetAddress = null;
            } catch (JSONException e) {
                Utils.reportException(StringeeClient.class, e);
                needGetAddress = true;
                getServerAddress();
            }
        }), error -> getExecutor().execute(() -> {
            if (!alreadyConnected) {
                return;
            }
            if (error.networkResponse == null) {
                needGetAddress = true;
                getServerAddress();
                return;
            }
            try {
                String data = new String(error.networkResponse.data);
                JSONObject response = new JSONObject(data);
                int r = response.getInt("r");
                String message = response.optString("message", "");
                catchGetServerAddressError(r, message);
                tokenGetAddress = null;
            } catch (JSONException e) {
                Utils.reportException(StringeeClient.class, e);
                needGetAddress = true;
                getServerAddress();
            }
        }));
        VolleyUtils.getInstance(context).add(request);
    }

    private void catchGetServerAddressError(int r, String message) {
        switch (r) {
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
            case 10:
                errorTokens.add(tokenGetAddress);
                break;
        }

        if (r == 1) {
            message = "Access token is invalid";
        }
        if (r == 4 || r == 5) {
            message = "Key SID not found or access token invalid";
        }
        if (r == 2) {
            message = "Access token is expired or invalid.";
            EventManager.sendClientNewTokenEvent(this);
        }
        EventManager.sendClientErrorEvent(this, new StringeeError(r, message));
        alreadyConnected = false;
    }

    private void startConnect(String connectToken) {
        needGetAddress = false;
        SocketAddress socketAddress = NetworkCommon.addresses.get(0);
        addressIndex = 0;
        NetworkCommon.currentServer = socketAddress;

        setToken(connectToken);
        String connectIp = socketAddress.getIp();
        String v4ServerIp = "";
        String v6ServerIp = "";
        InetAddress[] machines;
        try {
            machines = InetAddress.getAllByName(socketAddress.getIp());
            if (machines != null) {
                for (InetAddress address : machines) {
                    if (address != null) {
                        if (address instanceof Inet4Address) {
                            v4ServerIp = address.getHostAddress();
                        } else if (address instanceof Inet6Address) {
                            v6ServerIp = address.getHostAddress();
                        }
                    }
                }
            }
        } catch (Exception e) {
            Utils.reportException(StringeeClient.class, e);
        }

        boolean isIpv4Support = false;
        try {
            Enumeration<NetworkInterface> n = NetworkInterface.getNetworkInterfaces();
            if (n != null) {
                while (n.hasMoreElements()) { //for each interface
                    NetworkInterface e = n.nextElement();
                    Enumeration<InetAddress> a = e.getInetAddresses();
                    if (a != null) {
                        while (a.hasMoreElements()) {
                            InetAddress inetAddress = a.nextElement();
                            String add = inetAddress.getHostAddress();
                            if (add != null && !inetAddress.isLoopbackAddress() && add.length() < 17) {
                                isIpv4Support = true;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            Utils.reportException(StringeeClient.class, e);
        }

        if (isIpv4Support) {
            if (!Utils.isEmpty(v4ServerIp)) {
                connectIp = v4ServerIp;
            }
        } else {
            if (!Utils.isEmpty(v6ServerIp)) {
                connectIp = v6ServerIp;
            }
        }

        if (packetSenderThread == null) {
            packetSenderThread = new PacketSenderThread(StringeeClient.this);
        }
        if (checkTimeOutThread == null) {
            checkTimeOutThread = new CheckTimeOutThread();
        }

        boolean useSsl = true;
        IoHandler ioHandler = new IoHandler(StringeeClient.this);
        tcpClient = new TcpClient(ioHandler, connectIp, socketAddress.getPort(), useSsl, socketAddress.getSslAcceptedIP(), stringeeCertificates, trustAll, publicKeys);

        packetSenderThread.setTcpClient(tcpClient);
        checkTimeOutThread.setTcpClient(tcpClient);
        try {
            if (!packetSenderThread.isRunning()) {
                packetSenderThread.start();
            }
        } catch (Exception ex) {
            Utils.reportException(StringeeClient.class, ex);
        }
        try {
            tcpClient.start();
        } catch (Exception ex) {
            Utils.reportException(StringeeClient.class, ex);
        }
    }

    /**
     * Get user info
     *
     * @param userIdList List of user id
     * @param listener   Callback listener
     */
    public void getUserInfo(final List<String> userIdList, final CallbackListener<List<User>> listener) {
        getExecutor().execute(() -> {
            if (Utils.isEmpty(userIdList)) {
                if (listener != null) {
                    listener.onError(new StringeeError(-2, "UserIds are invalid"));
                }
                return;
            }
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            String userIds = "";
            if (!Utils.isEmpty(userIdList)) {
                for (int i = 0; i < userIdList.size(); i++) {
                    String userId = userIdList.get(i);
                    if (Utils.isEmpty(userIds)) {
                        userIds += userId;
                    } else {
                        userIds += ", " + userId;
                    }
                }
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.getUserInfo(this, requestId, userIds);
        });
    }

    public void getLastCustomerConversations(final int count, final String customerId, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatLoadCustomerConversations(requestId, StringeeClient.this, 0, Long.MAX_VALUE, count, customerId, false);
        });
    }

    public void getCustomerConversationsAfter(final long updateAt, final int count, final String customerId, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatLoadCustomerConversations(requestId, StringeeClient.this, updateAt, Long.MAX_VALUE, count, customerId, true);
        });
    }

    public void getCustomerConversationsBefore(final long updateAt, final int count, final String customerId, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatLoadCustomerConversations(requestId, StringeeClient.this, 0, updateAt, count, customerId, false);
        });
    }

    public void getLastXChatConversations(final int count, final CallbackListener<List<Conversation>> listener) {
        this.getLastXChatConversations(count, false, listener);
    }

    public void getLastXChatConversations(final int count, final boolean loadAll, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatLoadConversations(requestId, StringeeClient.this, 0, Long.MAX_VALUE, count, loadAll, false, false);
        });
    }

    public void getXChatConversationsAfter(final long updateAt, final int count, final CallbackListener<List<Conversation>> listener) {
        this.getXChatConversationsAfter(updateAt, count, false, listener);
    }

    public void getXChatConversationsAfter(final long updateAt, final int count, final boolean loadAll, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatLoadConversations(requestId, StringeeClient.this, updateAt, Long.MAX_VALUE, count, loadAll, false, true);
        });
    }

    public void getXChatConversationsBefore(final long updateAt, final int count, final CallbackListener<List<Conversation>> listener) {
        this.getXChatConversationsBefore(updateAt, count, false, listener);
    }

    public void getXChatConversationsBefore(final long updateAt, final int count, final boolean loadAll, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatLoadConversations(requestId, StringeeClient.this, 0, updateAt, count, loadAll, false, false);
        });
    }

    public void getXChatConversationFromServer(final List<String> convIds, final CallbackListener<List<Conversation>> listener) {
        getExecutor().execute(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.callbackListenerMap.put(requestId, listener);
            }
            SendPacketUtils.xChatGetConversation(requestId, StringeeClient.this, convIds);
        });
    }
}
