package com.stringee.call;

import android.Manifest.permission;
import android.content.Context;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;

import androidx.core.content.ContextCompat;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.Constant;
import com.stringee.common.PrefUtils;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.StringeeConstant;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.exception.StringeeException;
import com.stringee.listener.StatusListener;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.network.processor.CallStart;
import com.stringee.network.tcpclient.packet.Packet;
import com.stringee.network.tcpclient.packet.ServiceType;
import com.stringee.video.TextureViewRenderer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.EglBase;
import org.webrtc.MediaStream;
import org.webrtc.RendererCommon.RendererEvents;
import org.webrtc.RendererCommon.ScalingType;
import org.webrtc.SurfaceViewRenderer;

import java.util.LinkedList;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * Created by luannguyen on 7/15/2017.
 */

public class StringeeCall {

    public interface StringeeCallListener {
        void onSignalingStateChange(StringeeCall stringeeCall, SignalingState signalingState, String reason, int sipCode, String sipReason);

        void onError(StringeeCall stringeeCall, int code, String description);

        void onHandledOnAnotherDevice(StringeeCall stringeeCall, SignalingState signalingState, String description);

        void onMediaStateChange(StringeeCall stringeeCall, MediaState mediaState);

        void onLocalStream(StringeeCall stringeeCall);

        void onRemoteStream(StringeeCall stringeeCall);

        void onCallInfo(StringeeCall stringeeCall, JSONObject callInfo);
    }

    public interface CaptureSessionListener {
        void onCapturerStarted();

        void onCapturerStopped();
    }

    public static class StringeeCallStats {
        public int callPacketsLost;
        public int callPacketsReceived;
        public int callBytesReceived;
        public int videoPacketsLost;
        public int videoPacketsReceived;
        public int videoBytesReceived;
        public long timeStamp;

        public StringeeCallStats() {
        }
    }

    public interface CallStatsListener {
        void onCallStats(StringeeCallStats statsReport);
    }

    public enum SignalingState {
        CALLING(0),
        RINGING(1),
        ANSWERED(2),
        BUSY(3),
        ENDED(4);

        private final short value;

        SignalingState(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    public enum MediaState {
        CONNECTED(0),
        DISCONNECTED(1);

        private final short value;

        MediaState(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    private String callId;
    private String from;
    private String to;
    private String fromAlias;
    private String toAlias;
    private String encryptNumber;
    private int callStatus;
    private CallType callType = CallType.APP_TO_APP_OUTGOING;
    private VideoQuality videoQuality;
    private boolean isMute;
    private int requestId;
    private boolean isVideoCall;
    private String custom;
    private JSONObject encryptPhone;
    private boolean isCaller;
    private String deviceId;
    private boolean answeredOnAnotherDevice;
    private String customDataFromYourServer;
    private String customId;
    private SignalingState state;
    private boolean isP2P;
    private int audioBandwidth;
    private int videoBandwidth;

    private StringeeCallListener callListener;

    private SurfaceViewRenderer localRenderer;
    private TextureViewRenderer localRenderer2;
    private SurfaceViewRenderer remoteRenderer;
    private TextureViewRenderer remoteRenderer2;
    private MediaStream localStream;
    private MediaStream remoteStream;
    private EglBase.Context rootContext;

    private Context context;

    private final StringeeClient client;
    private StringeeCallFactory callFactory;
    private boolean initComplete;
    private boolean isLocalSdpSet;
    private boolean isRemoteSdpSet;
    private Timer timer;
    private TimerTask timerTask;
    private long mByteReceived;
    private int noByteReceivedCount;
    private boolean isHold;
    private Timer statsTimer;
    private TimerTask statsTimerTask;
    private JSONArray statsArray;

    private boolean canSwitch = false;
    private boolean isFrontCamera = true;
    String frontCameraName = "";
    String rearCameraName = "";

    private Timer checkExitsTimer;
    private static final long timeCheckExits = 1000 * 60 * 3;
    private CaptureSessionListener captureSessionListener;
    private LinkedList<StringeeIceServer> iceServers = new LinkedList<>();

    private final StringeeCallFactory.StringeeCallListener mCallListener = new StringeeCallFactory.StringeeCallListener() {
        @Override
        public void onSetSdpSuccess(StringeeSessionDescription sessionDescription) {
            sendSdpCandidate(sessionDescription);
        }

        @Override
        public void onCreateIceCandidate(StringeeIceCandidate iceCandidate) {
            //send cho peer con lai
            try {
                JSONObject data = new JSONObject();
                data.put("sdpMid", iceCandidate.sdpMid);
                data.put("sdpMLineIndex", iceCandidate.sdpMLineIndex);
                data.put("candidate", iceCandidate.sdp);
                Packet packet = new Packet(ServiceType.CALL_SDP_CANDIDATE);
                packet.setField("callId", callId);
                synchronized (Common.lock) {
                    packet.setField("requestId", ++Common.requestId);
                }
                packet.setField("type", "candidate");
                packet.setField("data", data);
                if (callStatus < StringeeConstant.ICE_CONNECTED) {
                    client.getPacketSenderThread().send(packet);
                } else {
                    if (client.isConnected()) {
                        client.getPacketSenderThread().send(packet);
                    }
                }
            } catch (JSONException e) {
                Utils.reportException(StringeeCall.class, e);
            }
        }

        @Override
        public void onChangeConnectionState(final String callId, StringeeConnectionState state) {
            if (state == StringeeConnectionState.CONNECTED) {
                if (callStatus == StringeeConstant.SIP_CODE_OK) {
                    setCallStatus(StringeeConstant.ICE_CONNECTED);
                }

                if (callListener != null) {
                    callListener.onMediaStateChange(StringeeCall.this, MediaState.CONNECTED);
                }
            }

            if (state == StringeeConnectionState.DISCONNECTED) {
                if (callListener != null) {
                    callListener.onMediaStateChange(StringeeCall.this, MediaState.DISCONNECTED);
                }
            }
        }

        @Override
        public void onAddStream(MediaStream mediaStream) {
            remoteStream = mediaStream;
            if (callListener != null) {
                callListener.onRemoteStream(StringeeCall.this);
            }

            callFactory.mute(isMute);
        }
    };

    public void setCaptureSessionListener(CaptureSessionListener captureSessionListener) {
        this.captureSessionListener = captureSessionListener;
        if (callFactory != null) {
            callFactory.setCaptureSessionListener(this.captureSessionListener);
        }
    }

    public StringeeCall(StringeeClient client, String from, String to) {
        this.context = client.getContext();
        this.client = client;
        this.from = from;
        this.to = to;
        isCaller = true;
        getCameraName();
    }

    public StringeeCall(StringeeClient client, String callId, String from, String to) {
        this.callId = callId;
        this.context = client.getContext();
        this.client = client;
        this.from = from;
        this.to = to;
        getCameraName();
        client.getCallMap().put(callId, this);
        startTimerCheckExitCall();
    }

    public SurfaceViewRenderer getLocalView() {
        if (localRenderer == null) {
            localRenderer = new SurfaceViewRenderer(context);
        }
        return localRenderer;
    }

    public TextureViewRenderer getLocalView2() {
        if (localRenderer2 == null) {
            localRenderer2 = new TextureViewRenderer(context);
        }
        return localRenderer2;
    }

    public SurfaceViewRenderer getRemoteView() {
        if (remoteRenderer == null) {
            remoteRenderer = new SurfaceViewRenderer(context);
        }
        return remoteRenderer;
    }

    public TextureViewRenderer getRemoteView2() {
        if (remoteRenderer2 == null) {
            remoteRenderer2 = new TextureViewRenderer(context);
        }
        return remoteRenderer2;
    }

    public Context getContext() {
        return context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    @Deprecated
    public void setQuality(int quality) {
        switch (quality) {
            case StringeeConstant.QUALITY_HD:
                this.videoQuality = VideoQuality.QUALITY_720P;
                break;
            case StringeeConstant.QUALITY_FULLHD:
                this.videoQuality = VideoQuality.QUALITY_1080P;
                break;
            default:
                this.videoQuality = VideoQuality.QUALITY_480P;
                break;
        }
    }

    public VideoQuality getVideoQuality() {
        return videoQuality;
    }

    public void setVideoQuality(VideoQuality videoQuality) {
        this.videoQuality = videoQuality;
    }

    public String getCallId() {
        return callId;
    }

    public void setCallId(String callId) {
        this.callId = callId;
    }

    public String getFrom() {
        return from;
    }

    public void setFrom(String from) {
        this.from = from;
    }

    public String getTo() {
        return to;
    }

    public void setTo(String to) {
        this.to = to;
    }

    public int getRequestId() {
        return requestId;
    }

    public void setRequestId(int requestId) {
        this.requestId = requestId;
    }

    public boolean isVideoCall() {
        return isVideoCall;
    }

    public void setVideoCall(boolean videoCall) {
        this.isVideoCall = videoCall;
    }

    @Deprecated
    public boolean isPhoneToAppCall() {
        return callType == CallType.PHONE_TO_APP;
    }

    public CallType getCallType() {
        return callType;
    }

    public void setCallType(CallType callType) {
        this.callType = callType;
    }

    @Deprecated
    public boolean isAppToPhoneCall() {
        return callType == CallType.APP_TO_PHONE;
    }

    public String getCustom() {
        return custom;
    }

    public void setCustom(String custom) {
        this.custom = custom;
    }

    public JSONObject getEncryptPhone() {
        return encryptPhone;
    }

    public void setEncryptPhone(JSONObject encryptPhone) {
        this.encryptPhone = encryptPhone;
    }

    public int getCallStatus() {
        return callStatus;
    }

    public void setCallStatus(int callStatus) {
        this.callStatus = callStatus;
    }

    public StringeeCallFactory getCallFactory() {
        return callFactory;
    }

    public void setCallFactory(StringeeCallFactory callFactory) {
        this.callFactory = callFactory;
    }

    public boolean isMute() {
        return isMute;
    }

    public void setLocalStream(MediaStream localStream) {
        this.localStream = localStream;
    }

    public void setRemoteStream(MediaStream remoteStream) {
        this.remoteStream = remoteStream;
    }

    public void setRootContext(EglBase.Context rootContext) {
        this.rootContext = rootContext;
    }

    public boolean isInitComplete() {
        return initComplete;
    }

    public void setInitComplete(boolean initComplete) {
        this.initComplete = initComplete;
    }

    public String getFromAlias() {
        return fromAlias;
    }

    public void setFromAlias(String fromAlias) {
        this.fromAlias = fromAlias;
    }

    public String getToAlias() {
        return toAlias;
    }

    public void setToAlias(String toAlias) {
        this.toAlias = toAlias;
    }

    public String getEncryptNumber() {
        return encryptNumber;
    }

    public void setEncryptNumber(String encryptNumber) {
        this.encryptNumber = encryptNumber;
    }

    public boolean isCaller() {
        return isCaller;
    }

    public void setCaller(boolean caller) {
        isCaller = caller;
    }

    public boolean isLocalSdpSet() {
        return isLocalSdpSet;
    }

    public void setLocalSdpSet(boolean localSdpSet) {
        isLocalSdpSet = localSdpSet;
    }

    public boolean isRemoteSdpSet() {
        return isRemoteSdpSet;
    }

    public void setRemoteSdpSet(boolean remoteSdpSet) {
        isRemoteSdpSet = remoteSdpSet;
    }

    public String getDeviceId() {
        return deviceId;
    }

    public void setDeviceId(String deviceId) {
        this.deviceId = deviceId;
    }

    public boolean isAnsweredOnAnotherDevice() {
        return answeredOnAnotherDevice;
    }

    public void setAnsweredOnAnotherDevice(boolean answeredOnAnotherDevice) {
        this.answeredOnAnotherDevice = answeredOnAnotherDevice;
    }

    public String getCustomDataFromYourServer() {
        return customDataFromYourServer;
    }

    public void setCustomDataFromYourServer(String customDataFromYourServer) {
        this.customDataFromYourServer = customDataFromYourServer;
    }

    public String getCustomId() {
        return customId;
    }

    public void setCustomId(String customId) {
        this.customId = customId;
    }

    public StringeeCallListener getCallListener() {
        return callListener;
    }

    public void setCallListener(StringeeCallListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> callListener = listener);
    }

    public StringeeClient getClient() {
        return client;
    }

    public SignalingState getState() {
        return state;
    }

    public void setState(SignalingState state) {
        this.state = state;
    }

    public boolean isP2P() {
        return isP2P;
    }

    public void setP2P(boolean p2P) {
        isP2P = p2P;
    }

    public int getAudioBandwidth() {
        return audioBandwidth;
    }

    public void setAudioBandwidth(int audioBandwidth) {
        this.audioBandwidth = audioBandwidth;
    }

    public int getVideoBandwidth() {
        return videoBandwidth;
    }

    public void setVideoBandwidth(int videoBandwidth) {
        this.videoBandwidth = videoBandwidth;
    }

    public LinkedList<StringeeIceServer> getIceServers() {
        return iceServers;
    }

    public void setIceServers(LinkedList<StringeeIceServer> iceServers) {
        this.iceServers = iceServers;
    }

    /**
     * Get cameraId
     */
    private void getCameraName() {
        // get cameraId
        CameraEnumerator enumerator = Camera2Enumerator.isSupported(context) ? new Camera2Enumerator(context) : new Camera1Enumerator();
        String[] cameraNames = enumerator.getDeviceNames();
        if (cameraNames.length > 0) {
            // first front id is main front camera
            for (String name : cameraNames) {
                boolean isFrontFace = enumerator.isFrontFacing(name);
                if (isFrontFace) {
                    frontCameraName = name;
                    break;
                }
            }

            // first rear id is main rear camera
            for (String cameraName : cameraNames) {
                boolean isBackFace = enumerator.isBackFacing(cameraName);
                if (isBackFace) {
                    rearCameraName = cameraName;
                    break;
                }
            }
        }

        canSwitch = (!Utils.isEmpty(frontCameraName) && !Utils.isEmpty(rearCameraName));
    }

    /**
     * Send DTMF
     *
     * @param dtmf     DTMF
     * @param listener StatusListener
     */
    public void sendDTMF(String dtmf, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendDTMF(client, callId, dtmf, requestId);
    }

    /**
     * Make a 1-1 call
     */
    public void makeCall(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        boolean isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
        if (isVideoCall) {
            isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED && ContextCompat.checkSelfPermission(context, permission.CAMERA) == PackageManager.PERMISSION_GRANTED;
        }
        if (!isPermissionGranted) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Permission is not granted"));
            }
            if (callListener != null) {
                callListener.onError(this, -1, "Permission is not granted");
            }
            return;
        }

        if (!client.isAlreadyConnected()) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeClient has not been connected yet"));
            }
            if (callListener != null) {
                callListener.onError(StringeeCall.this, -1, "StringeeClient has not been connected yet.");
            }
            return;
        }

        client.executeExecutor(() -> {
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            CallbackListener<StringeeCall> callbackListener = new CallbackListener<StringeeCall>() {
                @Override
                public void onSuccess(StringeeCall stringeeCall) {
                    isP2P = stringeeCall.isP2P();
                    state = stringeeCall.getState();
                    callId = stringeeCall.getCallId();
                    callType = stringeeCall.getCallType();
                    customDataFromYourServer = stringeeCall.getCustomDataFromYourServer();
                    iceServers = stringeeCall.getIceServers();
                    startTimerCheckExitCall();
                    client.getCallMap().put(callId, StringeeCall.this);
                    // Start call
                    callFactory = new StringeeCallFactory(StringeeCall.this, iceServers, new StringeeCallFactory.StringeeCallListener() {
                        @Override
                        public void onSetSdpSuccess(StringeeSessionDescription sessionDescription) {
                            sendSdpCandidate(sessionDescription);
                        }

                        @Override
                        public void onCreateIceCandidate(StringeeIceCandidate iceCandidate) {
                            //send cho peer con lai
                            try {
                                JSONObject data = new JSONObject();
                                data.put("sdpMid", iceCandidate.sdpMid);
                                data.put("sdpMLineIndex", iceCandidate.sdpMLineIndex);
                                data.put("candidate", iceCandidate.sdp);
                                Packet packet1 = new Packet(ServiceType.CALL_SDP_CANDIDATE);
                                packet1.setField("callId", callId);
                                synchronized (Common.lock) {
                                    packet1.setField("requestId", ++Common.requestId);
                                }
                                packet1.setField("type", "candidate");
                                packet1.setField("data", data);
                                client.getPacketSenderThread().send(packet1);
                            } catch (JSONException e) {
                                Utils.reportException(CallStart.class, e);
                            }
                        }

                        @Override
                        public void onChangeConnectionState(final String callId, StringeeConnectionState state) {
                            if (state == StringeeConnectionState.CONNECTED) {
                                if (callStatus == StringeeConstant.SIP_CODE_OK) {
                                    StringeeCall.this.setCallStatus(StringeeConstant.ICE_CONNECTED);
                                }

                                if (callListener != null) {
                                    callListener.onMediaStateChange(StringeeCall.this, StringeeCall.MediaState.CONNECTED);
                                }
                            }

                            if (state == StringeeConnectionState.DISCONNECTED) {
                                if (callListener != null) {
                                    callListener.onMediaStateChange(StringeeCall.this, StringeeCall.MediaState.DISCONNECTED);
                                }
                            }
                        }

                        @Override
                        public void onAddStream(MediaStream mediaStream) {
                            StringeeCall.this.setRemoteStream(mediaStream);
                            if (callListener != null) {
                                callListener.onRemoteStream(StringeeCall.this);
                            }
                            callFactory.mute(StringeeCall.this.isMute());
                        }
                    });
                    if (captureSessionListener != null) {
                        callFactory.setCaptureSessionListener(captureSessionListener);
                    }
                    callFactory.startCall(true);
                    if (callListener != null && callType != CallType.APP_TO_PHONE) {
                        callListener.onSignalingStateChange(StringeeCall.this, StringeeCall.SignalingState.CALLING, "Calling", StringeeConstant.SIP_CODE_CONNECTING, "Trying");
                    }
                    if (listener != null) {
                        listener.onSuccess();
                    }
                }

                @Override
                public void onError(StringeeError errorInfo) {
                    super.onError(errorInfo);
                    if (listener != null) {
                        listener.onError(errorInfo);
                    }
                    if (callListener != null) {
                        callListener.onError(StringeeCall.this, errorInfo.getCode(), errorInfo.getMessage());
                    }
                }
            };
            Common.callbackListenerMap.put(requestId, callbackListener);
            client.getCallRequest().put(requestId, StringeeCall.this);
            SendPacketUtils.startCall(client, StringeeCall.this);
        });
    }

    private void sendSdpCandidate(StringeeSessionDescription sessionDescription) {
        try {
            JSONObject data = new JSONObject();
            data.put("sdp", sessionDescription.description);
            if (sessionDescription.type == StringeeSessionDescription.Type.OFFER) {
                data.put("type", "offer");
            } else if (sessionDescription.type == StringeeSessionDescription.Type.ANSWER) {
                data.put("type", "answer");
            } else if (sessionDescription.type == StringeeSessionDescription.Type.PRANSWER) {
                data.put("type", "pranswer");
            }
            Packet packet = new Packet(ServiceType.CALL_SDP_CANDIDATE);
            packet.setField("callId", callId);
            packet.setField("data", data);
            synchronized (Common.lock) {
                packet.setField("requestId", ++Common.requestId);
            }
            packet.setField("type", "sdp");
            client.getPacketSenderThread().send(packet);
        } catch (JSONException e) {
            Utils.reportException(CallStart.class, e);
        }
    }

    /**
     * Send ringing
     */
    public void ringing(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.callChangeState(client, requestId, callId, StringeeConstant.SIP_CODE_RINGING, "Call ringing function");
            callStatus = StringeeConstant.SIP_CODE_RINGING;
        });
    }

    /**
     * Answer a call 1-1
     */
    public void answer(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        boolean isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
        if (isVideoCall) {
            isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED && ContextCompat.checkSelfPermission(context, permission.CAMERA) == PackageManager.PERMISSION_GRANTED;
        }
        if (!isPermissionGranted) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Permission is not granted"));
            }
            if (callListener != null) {
                callListener.onError(this, -1, "Permission is not granted");
            }
            return;
        }
        client.executeExecutor(() -> {
            if (!client.isAlreadyConnected()) {
                if (callListener != null) {
                    callListener.onError(StringeeCall.this, -1, "StringeeClient has not been connected yet.");
                }
                return;
            }

            if (answeredOnAnotherDevice) {
                if (callListener != null) {
                    callListener.onError(StringeeCall.this, -2, "This call is answered from another device.");
                }
                return;
            }

            if (state == SignalingState.ENDED) {
                if (callListener != null) {
                    callListener.onSignalingStateChange(StringeeCall.this, SignalingState.ENDED, "Ended", -1, "");
                }
                return;
            }

            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }

            Packet packet = client.getCallInPacket().get(callId);
            if (packet != null) {
                try {
                    packet.decodeJson();
                    boolean fromInternal = packet.getBoolean("fromInternal");
                    JSONArray jsonArray = (JSONArray) packet.getField("iceServers");
                    LinkedList<StringeeIceServer> iceServers = new LinkedList<>();
                    if (!Utils.isEmpty(jsonArray)) {
                        for (int i = 0; i < jsonArray.length(); i++) {
                            JSONObject jsonObject = jsonArray.getJSONObject(i);
                            StringeeIceServer iceServer = new StringeeIceServer(jsonObject.getString("urls"), jsonObject.getString("username"), jsonObject.getString("credential"));
                            iceServers.add(iceServer);
                        }
                    }
                    boolean isCallIn = !fromInternal;
                    setCallType(isCallIn ? CallType.PHONE_TO_APP : CallType.APP_TO_APP_INCOMING);

                    callFactory = new StringeeCallFactory(StringeeCall.this, iceServers, mCallListener);
                    if (captureSessionListener != null) {
                        callFactory.setCaptureSessionListener(captureSessionListener);
                    }
                    callFactory.startCall(false);
                } catch (JSONException ex) {
                    Utils.reportException(StringeeCall.class, ex);
                }
            }

            callStatus = StringeeConstant.SIP_CODE_OK;
            state = SignalingState.ANSWERED;
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.callChangeState(client, requestId, callId, StringeeConstant.SIP_CODE_OK, "Call answer function");
            if (callListener != null) {
                callListener.onSignalingStateChange(StringeeCall.this, SignalingState.ANSWERED, "Starting", -1, "");
            }

            String id = callId + deviceId;
            // Set remote sdp
            StringeeCallData sdpData = client.getSdpMap().get(id);
            if (sdpData != null) {
                try {
                    JSONObject dataObject = new JSONObject(sdpData.getData());
                    callFactory.processSdp(StringeeCall.this, dataObject);
                    client.getSdpMap().remove(id);
                } catch (JSONException e) {
                    Utils.reportException(StringeeCall.class, e);
                }
            }

            // Add candidates
            if (isLocalSdpSet) {
                LinkedBlockingQueue<StringeeCallData> candidates = client.getCandidatesMap().get(id);
                if (candidates != null) {
                    while (!candidates.isEmpty()) {
                        StringeeCallData callData = candidates.poll();
                        try {
                            if (callData != null) {
                                JSONObject dataObject = new JSONObject(callData.getData());
                                callFactory.processCandidate(dataObject);
                            }
                        } catch (JSONException e) {
                            Utils.reportException(StringeeCall.class, e);
                        }
                    }
                }
            }
            if (!isP2P) {
                checkToRestartICE();
            }
            startStatsTimer();
        });
    }

    /**
     * End call and release local resource
     */
    public void hangup(StatusListener listener) {
        hangup("Call hangup function", listener);
    }

    /**
     * End call and release local resource
     */
    public void hangup(String reason, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(new Runnable() {
            @Override
            public void run() {
                if (callId != null) {
                    if (!answeredOnAnotherDevice) {
                        if (!(callStatus == StringeeConstant.SIP_CODE_BUSY || callStatus == StringeeConstant.SIP_CODE_BUSY_2 || callStatus == StringeeConstant.SIP_CODE_ENDED)) {
                            synchronized (Common.lock) {
                                requestId = ++Common.requestId;
                            }
                            Common.statusListenerMap.put(requestId, new StatusListener() {
                                @Override
                                public void onSuccess() {
                                    if (listener != null) {
                                        listener.onSuccess();
                                    }
                                }

                                @Override
                                public void onError(StringeeError errorInfo) {
                                    super.onError(errorInfo);
                                    if (listener != null) {
                                        listener.onError(errorInfo);
                                    }
                                }
                            });
                            if (callListener != null) {
                                callListener.onSignalingStateChange(StringeeCall.this, StringeeCall.SignalingState.ENDED, "Ended", -1, "");
                            }
                            if (isCaller || callStatus == StringeeConstant.ICE_CONNECTED || callStatus == StringeeConstant.SIP_CODE_OK) {
                                SendPacketUtils.endCall(client, callId, requestId, reason);
                            } else {
                                SendPacketUtils.callChangeState(client, requestId, callId, StringeeConstant.SIP_CODE_BUSY, reason);
                            }
                        }
                    }
                }

                callStatus = StringeeConstant.SIP_CODE_ENDED;
                state = SignalingState.ENDED;

                uploadStats();

                release(true);
            }
        });
    }


    /**
     * Reject a call
     */
    public void reject(StatusListener listener) {
        reject("Call reject function", listener);
    }

    /**
     * Reject a call
     */
    public void reject(String reason, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(new Runnable() {
            @Override
            public void run() {
                synchronized (Common.lock) {
                    requestId = ++Common.requestId;
                }
                Common.statusListenerMap.put(requestId, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        if (listener != null) {
                            listener.onSuccess();
                        }
                        if (callListener != null) {
                            callListener.onSignalingStateChange(StringeeCall.this, StringeeCall.SignalingState.ENDED, "Ended", -1, "");
                        }
                    }

                    @Override
                    public void onError(StringeeError errorInfo) {
                        super.onError(errorInfo);
                        if (listener != null) {
                            listener.onError(errorInfo);
                        }
                    }
                });
                SendPacketUtils.callChangeState(client, requestId, callId, StringeeConstant.SIP_CODE_BUSY, reason);

                release(true);
            }
        });
    }

    /**
     * Release the call which answered from another device
     */
    public void release(boolean isHangup) {
        if (statsTimer != null) {
            statsTimer.cancel();
            statsTimer = null;
        }

        if (timerTask != null) {
            timerTask.cancel();
            timerTask = null;
        }

        if (timer != null) {
            timer.cancel();
            timer = null;
        }

        if (statsTimerTask != null) {
            statsTimerTask.cancel();
            statsTimerTask = null;
        }

        if (checkExitsTimer != null) {
            checkExitsTimer.cancel();
            checkExitsTimer = null;
        }

        if (client == null) {
            throw new StringeeException("StringeeClient has not been initialized.");
        }

        if (callId != null) {
            if (!Utils.isEmpty(deviceId)) {
                String id = callId + deviceId;
                client.getSdpMap().remove(id);
                client.getCandidatesMap().remove(id);
            }
            client.getCallInPacket().remove(callId);
            if (isHangup) {
                client.getCallMap().remove(callId);
            }
        }

        if (callFactory != null) {
            callFactory.stopCall(true);
            callFactory = null;
        }

        if (localRenderer != null) {
            localRenderer.release();
            localRenderer = null;
        }

        if (localRenderer2 != null) {
            localRenderer2.release();
            localRenderer2 = null;
        }

        if (remoteRenderer != null) {
            remoteRenderer.release();
            remoteRenderer = null;
        }

        if (remoteRenderer2 != null) {
            remoteRenderer2.release();
            remoteRenderer2 = null;
        }

        uploadStats();
    }

    /**
     * Mute/unmute call
     *
     * @param mute true: mute, false: unmute
     */
    public void mute(boolean mute) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            isMute = mute;
            if (callFactory != null) {
                callFactory.mute(isMute);
            }
        });
    }

    /**
     * Display local video
     */
    public void renderLocalView(boolean isOverlay) {
        this.renderLocalView(isOverlay, ScalingType.SCALE_ASPECT_FIT, null);
    }

    /**
     * Display local video
     */
    public void renderLocalView(boolean isOverlay, RendererEvents rendererEvents) {
        this.renderLocalView(isOverlay, ScalingType.SCALE_ASPECT_FIT, rendererEvents);
    }

    /**
     * Display local video
     */
    public void renderLocalView(boolean isOverlay, ScalingType scalingType) {
        this.renderLocalView(isOverlay, scalingType, null);
    }

    /**
     * Display local video
     */
    public void renderLocalView(boolean isOverlay, ScalingType scalingType, RendererEvents rendererEvents) {
        if (localStream != null && !Utils.isEmpty(localStream.videoTracks)) {
            SurfaceViewRenderer renderer = getLocalView();
            renderer.release();
            renderer.init(rootContext, rendererEvents);
            if (scalingType != null) {
                switch (scalingType) {
                    case SCALE_ASPECT_FIT:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);
                        break;
                    case SCALE_ASPECT_FILL:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);
                        break;
                    case SCALE_ASPECT_BALANCED:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_BALANCED);
                        break;
                }
            }
            renderer.setMirror(true);
            renderer.setZOrderMediaOverlay(isOverlay);
            renderer.setEnableHardwareScaler(true);
            localStream.videoTracks.get(0).addSink(renderer);
        }
    }

    /**
     * Display local video
     */
    public void renderLocalView2() {
        this.renderLocalView2(ScalingType.SCALE_ASPECT_FIT, null);
    }

    /**
     * Display local video
     */
    public void renderLocalView2(RendererEvents rendererEvents) {
        this.renderLocalView2(ScalingType.SCALE_ASPECT_FIT, rendererEvents);
    }

    /**
     * Display local video
     */
    public void renderLocalView2(ScalingType scalingType) {
        this.renderLocalView2(scalingType, null);
    }

    /**
     * Display local video
     */
    public void renderLocalView2(ScalingType scalingType, RendererEvents rendererEvents) {
        if (localStream != null && !Utils.isEmpty(localStream.videoTracks)) {
            TextureViewRenderer renderer = getLocalView2();
            renderer.release();
            renderer.init(rootContext, rendererEvents);
            if (scalingType != null) {
                switch (scalingType) {
                    case SCALE_ASPECT_FIT:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);
                        break;
                    case SCALE_ASPECT_FILL:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);
                        break;
                    case SCALE_ASPECT_BALANCED:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_BALANCED);
                        break;
                }
            }
            renderer.setMirror(true);
            renderer.setEnableHardwareScaler(true);
            localStream.videoTracks.get(0).addSink(renderer);
        }
    }

    /**
     * Display remote video
     */
    public void renderRemoteView(boolean isOverlay) {
        this.renderRemoteView(isOverlay, ScalingType.SCALE_ASPECT_FIT, null);
    }

    /**
     * Display remote video
     */
    public void renderRemoteView(boolean isOverlay, RendererEvents rendererEvents) {
        this.renderRemoteView(isOverlay, ScalingType.SCALE_ASPECT_FIT, rendererEvents);
    }

    /**
     * Display remote video
     */
    public void renderRemoteView(boolean isOverlay, ScalingType scalingType) {
        this.renderRemoteView(isOverlay, scalingType, null);
    }

    /**
     * Display remote video
     */
    public void renderRemoteView(boolean isOverlay, ScalingType scalingType, RendererEvents rendererEvents) {
        if (remoteStream != null && !Utils.isEmpty(remoteStream.videoTracks)) {
            SurfaceViewRenderer renderer = getRemoteView();
            renderer.release();
            renderer.init(rootContext, rendererEvents);
            if (scalingType != null) {
                switch (scalingType) {
                    case SCALE_ASPECT_FIT:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);
                        break;
                    case SCALE_ASPECT_FILL:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);
                        break;
                    case SCALE_ASPECT_BALANCED:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_BALANCED);
                        break;
                }
            }
            renderer.setMirror(false);
            renderer.setZOrderMediaOverlay(isOverlay);
            renderer.setEnableHardwareScaler(true);
            remoteStream.videoTracks.get(0).addSink(renderer);
        }
    }

    /**
     * Display remote video
     */
    public void renderRemoteView2() {
        this.renderRemoteView2(ScalingType.SCALE_ASPECT_FIT, null);
    }

    /**
     * Display remote video
     */
    public void renderRemoteView2(RendererEvents rendererEvents) {
        this.renderRemoteView2(ScalingType.SCALE_ASPECT_FIT, rendererEvents);
    }

    /**
     * Display remote video
     */
    public void renderRemoteView2(ScalingType scalingType) {
        this.renderRemoteView2(scalingType, null);
    }

    /**
     * Display remote video
     */
    public void renderRemoteView2(ScalingType scalingType, RendererEvents rendererEvents) {
        if (remoteStream != null && !Utils.isEmpty(remoteStream.videoTracks)) {
            TextureViewRenderer renderer = getRemoteView2();
            renderer.release();
            renderer.init(rootContext, rendererEvents);
            if (scalingType != null) {
                switch (scalingType) {
                    case SCALE_ASPECT_FIT:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);
                        break;
                    case SCALE_ASPECT_FILL:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);
                        break;
                    case SCALE_ASPECT_BALANCED:
                        renderer.setScalingType(ScalingType.SCALE_ASPECT_BALANCED);
                        break;
                }
            }
            renderer.setMirror(false);
            renderer.setEnableHardwareScaler(true);
            remoteStream.videoTracks.get(0).addSink(renderer);
        }
    }

    /**
     * Get call statistics
     *
     * @param statsListener callback
     */
    public void getStats(final CallStatsListener statsListener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (callFactory != null) {
                callFactory.getStats(statsListener);
            }
        });
    }

    /**
     * Enable/disable video
     *
     * @param enabled true: enable, false: disable
     */
    public void enableVideo(boolean enabled) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (callFactory != null) {
                callFactory.enableVideo(enabled);
            }
        });
    }

    /**
     * Switch camera
     *
     * @param listener callback
     */
    public void switchCamera(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (canSwitch) {
            client.executeExecutor(new Runnable() {
                @Override
                public void run() {
                    if (callFactory != null) {
                        callFactory.switchCamera(new StatusListener() {
                            @Override
                            public void onSuccess() {
                                isFrontCamera = !isFrontCamera;
                                if (listener != null) {
                                    listener.onSuccess();
                                }
                            }

                            @Override
                            public void onError(StringeeError errorInfo) {
                                super.onError(errorInfo);
                                if (listener != null) {
                                    listener.onError(errorInfo);
                                }
                            }
                        }, isFrontCamera ? rearCameraName : frontCameraName);
                    }
                }
            });
        } else {
            if (Utils.isEmpty(frontCameraName) && Utils.isEmpty(rearCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have any cameras"));
            } else if (Utils.isEmpty(rearCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have a rear camera"));
            } else if (Utils.isEmpty(frontCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have a front camera"));
            }
        }
    }

    /**
     * Switch to a specific camera
     *
     * @param listener   callback
     * @param cameraName camera name
     */
    public void switchCamera(StatusListener listener, String cameraName) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (callFactory != null) {
                callFactory.switchCamera(listener, cameraName);
            }
        });
    }

    /**
     * Send call info
     *
     * @param info JSONObject
     */
    public void sendCallInfo(JSONObject info, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendCallInfo(requestId, client, info, callId);
    }

    /**
     * Transfer the call to another user
     *
     * @param userId   user id
     * @param listener callback
     */
    public void transferToUserId(String userId, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (!(callStatus == StringeeConstant.SIP_CODE_OK || callStatus == StringeeConstant.ICE_CONNECTED)) {
            listener.onError(new StringeeError(-2, "Call not answered."));
            return;
        }

        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendTransfer(client, callId, userId, requestId);
    }

    public void transferToPhone(String from, String to, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (!(callStatus == StringeeConstant.SIP_CODE_OK || callStatus == StringeeConstant.ICE_CONNECTED)) {
            listener.onError(new StringeeError(-2, "Call not answered."));
            return;
        }

        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.transferToPhone(client, requestId, callId, from, to);
    }

    /**
     * Hold a call
     *
     * @param listener callback
     */
    public void hold2(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (isVideoCall) {
            listener.onError(new StringeeError(-1, "Only can hold a voice call."));
            return;
        }
        if (!(callStatus == StringeeConstant.SIP_CODE_OK || callStatus == StringeeConstant.ICE_CONNECTED)) {
            listener.onError(new StringeeError(-2, "Call not answered."));
            return;
        }
        client.executeExecutor(() -> {
            if (callFactory != null) {
                callFactory.hold(true);
                listener.onSuccess();
                isHold = true;
            }
        });
    }

    /**
     * Hold a call
     *
     * @param listener callback
     */
    public void hold(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (isVideoCall) {
            listener.onError(new StringeeError(-1, "Only can hold a voice call."));
            return;
        }
        if (!(callStatus == StringeeConstant.SIP_CODE_OK || callStatus == StringeeConstant.ICE_CONNECTED)) {
            listener.onError(new StringeeError(-2, "Call not answered."));
            return;
        }

        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.hold(client, requestId, callId, true);
        isHold = true;
    }

    /**
     * Unhold a call
     *
     * @param listener callback
     */
    public void unHold2(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }

        if (isVideoCall) {
            listener.onError(new StringeeError(-1, "Only can unhold a voice call."));
            return;
        }

        if (!(callStatus == StringeeConstant.SIP_CODE_OK || callStatus == StringeeConstant.ICE_CONNECTED)) {
            listener.onError(new StringeeError(-2, "Call not answered."));
            return;
        }

        client.executeExecutor(() -> {
            if (callFactory != null) {
                callFactory.hold(false);
                listener.onSuccess();
                isHold = false;
            }
        });
    }

    public void unHold(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }

        if (isVideoCall) {
            listener.onError(new StringeeError(-1, "Only can unhold a voice call."));
            return;
        }

        if (!(callStatus == StringeeConstant.SIP_CODE_OK || callStatus == StringeeConstant.ICE_CONNECTED)) {
            listener.onError(new StringeeError(-2, "Call not answered."));
            return;
        }

        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.hold(client, requestId, callId, false);
        isHold = false;
    }

    /**
     * Restart ice
     */
    public void restartICE() {
        if (callFactory != null) {
            callFactory.restartICE();
        }
    }

    public void checkToRestartICE() {
        timer = new Timer();
        timerTask = new TimerTask() {
            @Override
            public void run() {
                getStats(statsReport -> {
                    long bytesReceived = statsReport.callBytesReceived;
                    long bandwidth = bytesReceived - mByteReceived;
                    mByteReceived = bytesReceived;
                    if (bandwidth <= 0 && client.isConnected() && !isHold) {
                        noByteReceivedCount++;
                        if (noByteReceivedCount > Constant.MAX_CHECK_RESTART_ICE) {
                            noByteReceivedCount = 0;
                            restartICE();
                        }
                    } else {
                        noByteReceivedCount = 0;
                    }
                });
            }
        };
        timer.schedule(timerTask, 0, 1000);
    }

    public void startStatsTimer() {
        statsArray = new JSONArray();
        statsTimer = new Timer();
        statsTimerTask = new TimerTask() {
            @Override
            public void run() {
                getStats(statsReport -> {
                    try {
                        JSONObject statsObject = new JSONObject();
                        statsObject.put("bytesReceived", (long) statsReport.callBytesReceived);
                        JSONObject jsonObject = new JSONObject();
                        jsonObject.put(String.valueOf(statsReport.timeStamp), statsObject);
                        statsArray.put(jsonObject);
                    } catch (JSONException e) {
                        Utils.reportException(StringeeCall.class, e);
                    }
                });
            }
        };
        statsTimer.schedule(statsTimerTask, 0, 3000);
    }

    public void uploadStats() {
        if (statsArray == null || statsArray.length() == 0 || callId == null) {
            return;
        }
        String strStats = PrefUtils.getInstance(context).getString(Constant.STATS);
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put(callId, statsArray);
            JSONArray jsonArray;
            if (!Utils.isEmpty(strStats)) {
                jsonArray = new JSONArray(strStats);
                jsonArray.put(jsonObject);
            } else {
                jsonArray = new JSONArray();
                jsonArray.put(jsonObject);
            }
            PrefUtils.getInstance(context).putString(Constant.STATS, jsonArray.toString());

            if (statsArray.length() <= 60) {
                Utils.uploadStatsArray(context, client.getToken(), callId, Utils.getDeviceId(context), statsArray);
            } else {
                JSONArray uploadArray = new JSONArray();
                for (int i = 0; i < statsArray.length(); i++) {
                    uploadArray.put(statsArray.getJSONObject(i));
                    if ((i > 0 && i % 60 == 0) || (i == (statsArray.length() - 1))) {
                        Utils.uploadStatsArray(context, client.getToken(), callId, Utils.getDeviceId(context), uploadArray);
                        uploadArray = new JSONArray();
                    }
                }
            }
        } catch (JSONException e) {
            Utils.reportException(StringeeCall.class, e);
        }
    }

    /**
     * Resume camera
     */
    public void resumeVideo() {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (callFactory != null) {
                callFactory.upgradeVideo(true, true);
            }
        });
    }

    /**
     * Take a snapshot while calling
     *
     * @param listener callback
     */
    public void snapShot(CallbackListener<Bitmap> listener) {
        if (callFactory != null) {
            callFactory.snapshot(listener);
        }
    }

    public void startTimerCheckExitCall() {
        if (checkExitsTimer == null) {
            checkExitsTimer = new Timer();
            checkExitsTimer.schedule(new TimerTask() {
                @Override
                public void run() {
                    int requestId;
                    synchronized (Common.lock) {
                        requestId = ++Common.requestId;
                    }
                    Common.callbackListenerMap.put(requestId, new CallbackListener<Boolean>() {
                        @Override
                        public void onSuccess(Boolean exits) {
                            if (!exits) {
                                hangup("Call does not exist on the server", null);
                            }
                        }
                    });
                    SendPacketUtils.checkExitsCall(client, requestId, callId);
                }
            }, timeCheckExits, timeCheckExits);
        }
    }
}
