package com.stringee.call;

import static org.webrtc.RendererCommon.RendererEvents;

import android.Manifest.permission;
import android.content.Context;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.os.Build;

import androidx.annotation.RequiresApi;
import androidx.core.content.ContextCompat;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.StringeeConstant;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.exception.StringeeException;
import com.stringee.listener.StatusListener;
import com.stringee.messaging.listeners.CallbackListener;
import com.stringee.video.StringeeRoom;
import com.stringee.video.StringeeScreenCapture;
import com.stringee.video.StringeeVideo;
import com.stringee.video.StringeeVideoTrack;
import com.stringee.video.StringeeVideoTrack.MediaType;
import com.stringee.video.TextureViewRenderer;
import com.stringee.video.VideoDimensions;

import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.PeerConnection;
import org.webrtc.RTCStats;
import org.webrtc.RendererCommon.ScalingType;
import org.webrtc.SurfaceViewRenderer;

import java.math.BigInteger;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

public class StringeeCall2 {
    public interface StringeeCallListener {
        void onSignalingStateChange(StringeeCall2 stringeeCall2, StringeeCall2.SignalingState signalingState, String reason, int sipCode, String sipReason);

        void onError(StringeeCall2 stringeeCall2, int code, String description);

        void onHandledOnAnotherDevice(StringeeCall2 stringeeCall2, StringeeCall2.SignalingState signalingState, String description);

        void onMediaStateChange(StringeeCall2 stringeeCall2, StringeeCall2.MediaState mediaState);

        @Deprecated
        void onLocalStream(StringeeCall2 stringeeCall2);

        @Deprecated
        void onRemoteStream(StringeeCall2 stringeeCall2);

        @Deprecated
        void onVideoTrackAdded(StringeeVideoTrack videoTrack);

        @Deprecated
        void onVideoTrackRemoved(StringeeVideoTrack videoTrack);

        void onLocalTrackAdded(StringeeCall2 stringeeCall2, StringeeVideoTrack stringeeVideoTrack);

        void onRemoteTrackAdded(StringeeCall2 stringeeCall2, StringeeVideoTrack stringeeVideoTrack);

        void onCallInfo(StringeeCall2 stringeeCall, JSONObject callInfo);

        void onTrackMediaStateChange(String from, MediaType mediaType, boolean enable);
    }

    public static class StringeeCallStats {
        public int callPacketsLost;
        public int callPacketsReceived;
        public int callBytesReceived;
        public int videoPacketsLost;
        public int videoPacketsReceived;
        public int videoBytesReceived;
        public long timeStamp;

        public StringeeCallStats() {
        }
    }

    public interface CallStatsListener {
        void onCallStats(StringeeCall2.StringeeCallStats statsReport);
    }

    public enum SignalingState {
        CALLING(0), RINGING(1), ANSWERED(2), BUSY(3), ENDED(4);

        private final short value;

        SignalingState(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    public enum MediaState {
        CONNECTED(0), DISCONNECTED(1);

        private final short value;

        MediaState(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    @Deprecated
    public enum VideoQuality {
        QUALITY_288P(0), QUALITY_480P(1), QUALITY_720P(2), QUALITY_1080P(3);

        private final short value;

        VideoQuality(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }
    }

    public enum EndCallCause {
        NORMAL("NORMAL"), NOT_ENOUGH_MONEY("NOT_ENOUGH_MONEY"), MAX_CONNECT_TIME("MAX_CONNECT_TIME"), TIMEOUT_CLOSE_CONNECTION("TIMEOUT_CLOSE_CONNECTION"), USER_BUSY("USER_BUSY"), USER_END_CALL("USER_END_CALL"), USER_CANCEL("USER_CANCEL"), TIMEOUT_WAIT_SDP_TO_MAKE_CALL("TIMEOUT_WAIT_SDP_TO_MAKE_CALL"), CAN_NOT_MAKE_CALL("CAN_NOT_MAKE_CALL"), USER_TEMPORARILY_UNAVAILABLE("USER_TEMPORARILY_UNAVAILABLE"), TIMEOUT_MAKE_CALL("TIMEOUT_MAKE_CALL"), TIMEOUT_ANSWER("TIMEOUT_ANSWER"), DTMF_END("DTMF_END"), VOICE_MAIL_END("VOICE_MAIL_END"), USER_MAKE_ANOTHER_CALL("USER_MAKE_ANOTHER_CALL"), REST_API_STOP("REST_API_STOP"), STRINGEE_CLIENT_DISCONNECTED("STRINGEE_CLIENT_DISCONNECTED"), UNKNOWN("UNKNOWN"), CALL_NOT_EXIST_IN_SERVER("CALL_NOT_EXIST_IN_SERVER");

        private final String value;

        EndCallCause(final String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    private String from;
    private String to;
    private String fromAlias;
    private String toAlias;
    private CallType callType = CallType.APP_TO_APP_OUTGOING;
    private String encryptNumber;
    private String roomId;
    private String roomToken;
    private boolean isVideoCall;
    private String custom;
    private JSONObject encryptPhone;
    private String customDataFromYourServer;
    private String callId;
    private boolean isCaller;
    private StringeeCall2.SignalingState state;
    private int callStatus;
    private boolean answeredOnAnotherDevice;
    private boolean isMute;
    private boolean isVideoEnable;
    private com.stringee.call.VideoQuality videoQuality;

    private final Context context;
    private final StringeeClient client;
    private StringeeRoom room;
    private StringeeVideoTrack localVideoTrack;
    private StringeeVideoTrack localCaptureTrack;
    private StringeeVideoTrack remoteVideoTrack;
    private StringeeCall2.StringeeCallListener callListener;

    private boolean canSwitch = true;
    private boolean isFrontCamera = true;
    String frontCameraName = "";
    String rearCameraName = "";

    private boolean autoSendTrackMediaStateChangeEvent = false;

    private Timer timer;
    private int timeout;
    private double mPrevCallTimestamp = 0;
    private long mPrevCallBytes = 0;
    private long mCallBw = 0;
    private boolean needRestartTrack = false;
    private boolean reJoining = false;

    private String currentIp;

    private Timer checkExitsTimer;
    private static final long timeCheckExits = 1000 * 60 * 3;
    private StringeeVideoTrack.CaptureSessionListener captureSessionListener;

    public StringeeCall2(StringeeClient client, String from, String to) {
        this.context = client.getContext();
        this.client = client;
        this.from = from;
        this.to = to;
        this.isCaller = true;
        this.currentIp = client.getClientIp();
        getCameraName();
    }

    public StringeeCall2(StringeeClient client, String from, String to, boolean isVideoCall) {
        this.context = client.getContext();
        this.client = client;
        this.from = from;
        this.to = to;
        this.isVideoCall = isVideoCall;
        this.isVideoEnable = isVideoCall;
        this.currentIp = client.getClientIp();
        getCameraName();
    }

    public StringeeCall2(StringeeClient client, String callId, String from, String to) {
        this.context = client.getContext();
        this.callId = callId;
        this.client = client;
        this.from = from;
        this.to = to;
        this.currentIp = client.getClientIp();
        getCameraName();
        client.getCallMap2().put(callId, this);
        startTimerCheckExitCall();
    }

    public String getCallId() {
        return callId;
    }

    public void setCallId(String callId) {
        this.callId = callId;
    }

    public String getFrom() {
        return from;
    }

    public void setFrom(String from) {
        this.from = from;
    }

    public String getTo() {
        return to;
    }

    public void setTo(String to) {
        this.to = to;
    }

    public String getFromAlias() {
        return fromAlias;
    }

    public void setFromAlias(String fromAlias) {
        this.fromAlias = fromAlias;
    }

    public String getToAlias() {
        return toAlias;
    }

    public void setToAlias(String toAlias) {
        this.toAlias = toAlias;
    }

    public CallType getCallType() {
        return callType;
    }

    public void setCallType(CallType callType) {
        this.callType = callType;
    }

    public String getEncryptNumber() {
        return encryptNumber;
    }

    public void setEncryptNumber(String encryptNumber) {
        this.encryptNumber = encryptNumber;
    }

    public String getRoomId() {
        return roomId;
    }

    public void setRoomId(String roomId) {
        this.roomId = roomId;
    }

    public String getRoomToken() {
        return roomToken;
    }

    public void setRoomToken(String roomToken) {
        this.roomToken = roomToken;
    }

    public boolean isVideoCall() {
        return isVideoCall;
    }

    public void setVideoCall(boolean videoCall) {
        isVideoCall = videoCall;
        isVideoEnable = videoCall;
    }

    public String getCustom() {
        return custom;
    }

    public void setCustom(String custom) {
        this.custom = custom;
    }

    public JSONObject getEncryptPhone() {
        return encryptPhone;
    }

    public void setEncryptPhone(JSONObject encryptPhone) {
        this.encryptPhone = encryptPhone;
    }

    public boolean isCaller() {
        return isCaller;
    }

    public SignalingState getState() {
        return state;
    }

    public void setState(SignalingState state) {
        this.state = state;
    }

    public int getCallStatus() {
        return callStatus;
    }

    public void setCallStatus(int callStatus) {
        this.callStatus = callStatus;
    }

    public StringeeCall2.StringeeCallListener getCallListener() {
        return callListener;
    }

    public void setCallListener(StringeeCall2.StringeeCallListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> callListener = listener);
    }

    public String getCustomDataFromYourServer() {
        return customDataFromYourServer;
    }

    public void setCustomDataFromYourServer(String customDataFromYourServer) {
        this.customDataFromYourServer = customDataFromYourServer;
    }

    public StringeeRoom getRoom() {
        return room;
    }

    public void setRoom(StringeeRoom room) {
        this.room = room;
    }

    public void setLocalVideoTrack(StringeeVideoTrack localVideoTrack) {
        this.localVideoTrack = localVideoTrack;
    }

    public void setRemoteVideoTrack(StringeeVideoTrack remoteVideoTrack) {
        this.remoteVideoTrack = remoteVideoTrack;
    }

    public StringeeVideoTrack getRemoteVideoTrack() {
        return remoteVideoTrack;
    }

    public boolean isMute() {
        return isMute;
    }

    public boolean isVideoEnable() {
        return isVideoEnable;
    }

    public boolean isAnsweredOnAnotherDevice() {
        return answeredOnAnotherDevice;
    }

    public void setAnsweredOnAnotherDevice(boolean answeredOnAnotherDevice) {
        this.answeredOnAnotherDevice = answeredOnAnotherDevice;
    }

    public void setAutoSendTrackMediaStateChangeEvent(boolean autoSendTrackMediaStateChangeEvent) {
        this.autoSendTrackMediaStateChangeEvent = autoSendTrackMediaStateChangeEvent;
    }

    @Deprecated
    public void setQuality(VideoQuality quality) {
        switch (quality) {
            case QUALITY_288P:
                this.videoQuality = com.stringee.call.VideoQuality.QUALITY_288P;
                break;
            case QUALITY_480P:
                this.videoQuality = com.stringee.call.VideoQuality.QUALITY_480P;
                break;
            case QUALITY_720P:
                this.videoQuality = com.stringee.call.VideoQuality.QUALITY_720P;
                break;
            case QUALITY_1080P:
                this.videoQuality = com.stringee.call.VideoQuality.QUALITY_1080P;
                break;
        }
    }

    public com.stringee.call.VideoQuality getVideoQuality() {
        return videoQuality;
    }

    public void setVideoQuality(com.stringee.call.VideoQuality videoQuality) {
        this.videoQuality = videoQuality;
    }

    public StringeeVideoTrack getLocalCaptureTrack() {
        return localCaptureTrack;
    }

    public void setLocalCaptureTrack(StringeeVideoTrack localCaptureTrack) {
        this.localCaptureTrack = localCaptureTrack;
    }

    public void setReJoining(boolean reJoining) {
        this.reJoining = reJoining;
    }

    public StringeeVideoTrack.CaptureSessionListener getCaptureSessionListener() {
        return captureSessionListener;
    }

    public void setCaptureSessionListener(StringeeVideoTrack.CaptureSessionListener captureSessionListener) {
        this.captureSessionListener = captureSessionListener;
    }

    /**
     * Get cameraId
     */
    private void getCameraName() {
        // get cameraId
        CameraEnumerator enumerator = Camera2Enumerator.isSupported(context) ? new Camera2Enumerator(context) : new Camera1Enumerator();
        String[] cameraNames = enumerator.getDeviceNames();
        if (cameraNames.length > 0) {
            // first front id is main front camera
            for (String name : cameraNames) {
                boolean isFrontFace = enumerator.isFrontFacing(name);
                if (isFrontFace) {
                    frontCameraName = name;
                    break;
                }
            }

            // first rear id is main rear camera
            for (String cameraName : cameraNames) {
                boolean isBackFace = enumerator.isBackFacing(cameraName);
                if (isBackFace) {
                    rearCameraName = cameraName;
                    break;
                }
            }
        }

        canSwitch = (!Utils.isEmpty(frontCameraName) && !Utils.isEmpty(rearCameraName));
    }

    /**
     * Make a call
     *
     * @param listener StatusListener
     */
    public void makeCall(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        boolean isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
        if (isVideoCall) {
            isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED && ContextCompat.checkSelfPermission(context, permission.CAMERA) == PackageManager.PERMISSION_GRANTED;
        }
        if (!isPermissionGranted) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Permission is not granted"));
            }
            if (callListener != null) {
                callListener.onError(this, -1, "Permission is not granted");
            }
            return;
        }

        if (!client.isAlreadyConnected()) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeClient has not been connected yet"));
            }
            if (callListener != null) {
                callListener.onError(StringeeCall2.this, -1, "StringeeClient has not been connected yet");
            }
            return;
        }

        client.executeExecutor(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            CallbackListener<StringeeCall2> callbackListener = new CallbackListener<StringeeCall2>() {
                @Override
                public void onSuccess(StringeeCall2 stringeeCall2) {
                    callId = stringeeCall2.getCallId();
                    callType = stringeeCall2.getCallType();
                    roomId = stringeeCall2.getRoomId();
                    roomToken = stringeeCall2.getRoomToken();
                    StringeeVideo.initRoomCall(client, StringeeCall2.this);
                    startTimerCheckExitCall();
                    client.getCallMap2().put(callId, StringeeCall2.this);
                    if (listener != null) {
                        listener.onSuccess();
                    }
                    if (callListener != null) {
                        callListener.onSignalingStateChange(StringeeCall2.this, StringeeCall2.SignalingState.CALLING, "Calling", -1, "");
                    }
                }

                @Override
                public void onError(StringeeError errorInfo) {
                    super.onError(errorInfo);
                    if (listener != null) {
                        listener.onError(errorInfo);
                    }
                    if (callListener != null) {
                        callListener.onError(StringeeCall2.this, errorInfo.getCode(), errorInfo.getMessage());
                    }
                }
            };
            Common.callbackListenerMap.put(requestId, callbackListener);
            client.getCallRequest2().put(requestId, StringeeCall2.this);
            SendPacketUtils.startCall2(client, requestId, StringeeCall2.this);
        });
    }

    /**
     * Send ringing
     */
    public void ringing(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            callStatus = StringeeConstant.SIP_CODE_RINGING;
            SendPacketUtils.callChangeState2(client, requestId, callId, StringeeConstant.SIP_CODE_RINGING, "Call ringing function");
        });
    }

    /**
     * Answer a call
     */
    public void answer(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        boolean isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
        if (isVideoCall) {
            isPermissionGranted = ContextCompat.checkSelfPermission(context, permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED && ContextCompat.checkSelfPermission(context, permission.CAMERA) == PackageManager.PERMISSION_GRANTED;
        }
        if (!isPermissionGranted) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Permission is not granted"));
            }
            if (callListener != null) {
                callListener.onError(this, -1, "Permission is not granted");
            }
            return;
        }
        client.executeExecutor(() -> {
            if (!client.isAlreadyConnected()) {
                if (callListener != null) {
                    callListener.onError(StringeeCall2.this, -1, "StringeeClient has not been connected yet.");
                }
                return;
            }

            if (answeredOnAnotherDevice) {
                if (callListener != null) {
                    callListener.onHandledOnAnotherDevice(StringeeCall2.this, state, "This call is handled on another device.");
                }
                return;
            }

            if (state == SignalingState.ENDED) {
                if (callListener != null) {
                    callListener.onSignalingStateChange(StringeeCall2.this, SignalingState.ENDED, EndCallCause.CALL_NOT_EXIST_IN_SERVER.getValue(), -1, "");
                }
                return;
            }

            callStatus = StringeeConstant.SIP_CODE_OK;
            state = SignalingState.ANSWERED;
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.callChangeState2(client, requestId, callId, StringeeConstant.SIP_CODE_OK, "Call answer function");
            if (callListener != null) {
                callListener.onSignalingStateChange(StringeeCall2.this, SignalingState.ANSWERED, "Starting", -1, "");
            }

            StringeeVideo.initRoomCall(client, StringeeCall2.this);
        });
    }

    /**
     * End a call
     */
    public void hangup(StatusListener listener) {
        hangup("Call hangup function", listener);
    }

    /**
     * End a call
     */
    public void hangup(String reason, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (answeredOnAnotherDevice) {
                return;
            }
            if (state == SignalingState.ENDED || state == SignalingState.BUSY) {
                return;
            }
            if (callId != null) {
                int requestId;
                synchronized (Common.lock) {
                    requestId = ++Common.requestId;
                }
                if (isCaller || state == SignalingState.ANSWERED) {
                    if (listener != null) {
                        Common.statusListenerMap.put(requestId, listener);
                    }
                    SendPacketUtils.endCall2(client, requestId, callId, reason);
                } else {
                    Common.statusListenerMap.put(requestId, new StatusListener() {
                        @Override
                        public void onSuccess() {
                            if (listener != null) {
                                listener.onSuccess();
                            }
                        }

                        @Override
                        public void onError(StringeeError errorInfo) {
                            super.onError(errorInfo);
                            if (listener != null) {
                                listener.onError(errorInfo);
                            }
                        }
                    });
                    SendPacketUtils.callChangeState2(client, requestId, callId, StringeeConstant.SIP_CODE_BUSY, reason);
                }
                if (callListener != null) {
                    callListener.onSignalingStateChange(StringeeCall2.this, StringeeCall2.SignalingState.ENDED, EndCallCause.NORMAL.getValue(), -1, "");
                }
            }

            callStatus = StringeeConstant.SIP_CODE_ENDED;
            state = StringeeCall2.SignalingState.ENDED;

            release();
        });
    }

    public void release() {
        if (client == null) {
            throw new StringeeException("StringeeClient has not been initialized.");
        }
        client.executeExecutor(() -> {
            if (callId != null) {
                client.getCallMap2().remove(callId);
            }

            if (room != null) {
                StringeeVideo.release(context, room);
                room = null;
            }

            timeout = 0;
            if (timer != null) {
                timer.cancel();
                timer = null;
            }

            if (checkExitsTimer != null) {
                checkExitsTimer.cancel();
                checkExitsTimer = null;
            }
        });
    }

    /**
     * Reject a call
     */
    public void reject(StatusListener listener) {
        reject("Call reject function", listener);
    }

    /**
     * Reject a call
     */
    public void reject(String reason, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            Common.statusListenerMap.put(requestId, new StatusListener() {
                @Override
                public void onSuccess() {
                    if (listener != null) {
                        listener.onSuccess();
                    }
                    if (callListener != null) {
                        callListener.onSignalingStateChange(StringeeCall2.this, StringeeCall2.SignalingState.ENDED, EndCallCause.NORMAL.getValue(), -1, "");
                    }
                    release();
                }

                @Override
                public void onError(StringeeError errorInfo) {
                    super.onError(errorInfo);
                    if (listener != null) {
                        listener.onError(errorInfo);
                    }
                }
            });
            SendPacketUtils.callChangeState2(client, requestId, callId, StringeeConstant.SIP_CODE_BUSY, reason);

            callStatus = StringeeConstant.SIP_CODE_ENDED;
            state = StringeeCall2.SignalingState.ENDED;
        });
    }

    /**
     * Get local video view
     */
    public SurfaceViewRenderer getLocalView() {
        if (localVideoTrack == null) {
            return null;
        }
        return localVideoTrack.getView(context);
    }

    /**
     * Get local video view
     */
    public TextureViewRenderer getLocalView2() {
        if (localVideoTrack == null) {
            return null;
        }
        return localVideoTrack.getView2(context);
    }

    /**
     * Get remote video view
     */
    public SurfaceViewRenderer getRemoteView() {
        if (remoteVideoTrack == null) {
            return null;
        }
        return remoteVideoTrack.getView(context);
    }

    /**
     * Get remote video view
     */
    public TextureViewRenderer getRemoteView2() {
        if (remoteVideoTrack == null) {
            return null;
        }
        return remoteVideoTrack.getView2(context);
    }

    /**
     * Render local video
     */
    public void renderLocalView(boolean isOverlay) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView(isOverlay);
        }
    }

    /**
     * Render local video
     */
    public void renderLocalView(boolean isOverlay, RendererEvents rendererEvents) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView(isOverlay, rendererEvents);
        }
    }

    /**
     * Render local video with scale
     */
    public void renderLocalView(boolean isOverlay, ScalingType scalingType) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView(isOverlay, scalingType, null);
        }
    }

    /**
     * Render local video with scale
     */
    public void renderLocalView(boolean isOverlay, ScalingType scalingType, RendererEvents rendererEvents) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView(isOverlay, scalingType, rendererEvents);
        }
    }

    /**
     * Render local video
     */
    public void renderLocalView2() {
        if (localVideoTrack != null) {
            localVideoTrack.renderView2();
        }
    }

    /**
     * Render local video
     */
    public void renderLocalView2(RendererEvents rendererEvents) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView2(rendererEvents);
        }
    }

    /**
     * Render local video with scale
     */
    public void renderLocalView2(ScalingType scalingType) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView2(scalingType, null);
        }
    }

    /**
     * Render local video with scale
     */
    public void renderLocalView2(ScalingType scalingType, RendererEvents rendererEvents) {
        if (localVideoTrack != null) {
            localVideoTrack.renderView2(scalingType, rendererEvents);
        }
    }

    /**
     * Render remote video
     */
    public void renderRemoteView(boolean isOverlay) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView(isOverlay);
        }
    }

    /**
     * Render remote video
     */
    public void renderRemoteView(boolean isOverlay, RendererEvents rendererEvents) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView(isOverlay, rendererEvents);
        }
    }

    /**
     * Render remote video with scale
     */
    public void renderRemoteView(boolean isOverlay, ScalingType scalingType) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView(isOverlay, scalingType, null);
        }
    }

    /**
     * Render remote video with scale
     */
    public void renderRemoteView(boolean isOverlay, ScalingType scalingType, RendererEvents rendererEvents) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView(isOverlay, scalingType, rendererEvents);
        }
    }

    /**
     * Render remote video
     */
    public void renderRemoteView2() {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView2();
        }
    }

    /**
     * Render remote video
     */
    public void renderRemoteView2(RendererEvents rendererEvents) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView2(rendererEvents);
        }
    }

    /**
     * Render remote video with scale
     *
     * @param scalingType scale type
     */
    public void renderRemoteView2(ScalingType scalingType) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView2(scalingType, null);
        }
    }

    /**
     * Render remote video with scale
     *
     * @param scalingType scale type
     */
    public void renderRemoteView2(ScalingType scalingType, RendererEvents rendererEvents) {
        if (remoteVideoTrack != null) {
            remoteVideoTrack.renderView2(scalingType, rendererEvents);
        }
    }

    /**
     * Enable/disable audio
     *
     * @param mute true: mute, false: unmute
     */
    public void mute(boolean mute) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            isMute = mute;
            if (autoSendTrackMediaStateChangeEvent) {
                if (localVideoTrack != null) {
                    localVideoTrack.sendAudioEnableNotification(!isMute, new StatusListener() {
                        @Override
                        public void onSuccess() {

                        }
                    });
                }
            }
            if (localVideoTrack != null) {
                localVideoTrack.mute(isMute);
            }
        });
    }

    /**
     * Enable/disable video
     *
     * @param videoEnabled true: enable, false: disable
     */
    public void enableVideo(boolean videoEnabled) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            isVideoEnable = videoEnabled;
            if (localVideoTrack != null) {
                if (autoSendTrackMediaStateChangeEvent) {
                    localVideoTrack.sendVideoEnableNotification(isVideoEnable, new StatusListener() {
                        @Override
                        public void onSuccess() {

                        }
                    });
                }
                localVideoTrack.enableVideo(isVideoEnable);
            }
        });
    }

    /**
     * Switch camera
     *
     * @param listener StatusListener
     */
    public void switchCamera(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (canSwitch) {
            client.executeExecutor(() -> {
                if (localVideoTrack != null) {
                    localVideoTrack.switchCamera(new StatusListener() {
                        @Override
                        public void onSuccess() {
                            isFrontCamera = !isFrontCamera;
                            if (listener != null) {
                                listener.onSuccess();
                            }
                        }

                        @Override
                        public void onError(StringeeError errorInfo) {
                            super.onError(errorInfo);
                            if (listener != null) {
                                listener.onError(errorInfo);
                            }
                        }
                    }, isFrontCamera ? rearCameraName : frontCameraName);
                }
            });
        } else {
            if (Utils.isEmpty(frontCameraName) && Utils.isEmpty(rearCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have any cameras"));
            } else if (Utils.isEmpty(rearCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have a rear camera"));
            } else if (Utils.isEmpty(frontCameraName)) {
                listener.onError(new StringeeError(-1, "The device does not have a front camera"));
            }
        }
    }

    /**
     * Switch to a specific camera
     *
     * @param listener   StatusListener
     * @param cameraName camera name
     */
    public void switchCamera(StatusListener listener, String cameraName) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (localVideoTrack != null) {
                localVideoTrack.switchCamera(listener, cameraName);
            }
        });
    }

    /**
     * Get call statistic
     *
     * @param statsListener CallStatsListener
     */
    public void getStats(final StringeeCall2.CallStatsListener statsListener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (remoteVideoTrack != null) {
                PeerConnection peerConnection = remoteVideoTrack.getPeerConnection();
                if (peerConnection != null) {
                    peerConnection.getStats(rtcStatsReport -> {
                        StringeeCallStats stringeeCallStats = new StringeeCallStats();
                        stringeeCallStats.timeStamp = System.currentTimeMillis();
                        for (Map.Entry<String, RTCStats> e : rtcStatsReport.getStatsMap().entrySet()) {
                            Map<String, Object> members = e.getValue().getMembers();
                            if (e.getValue().getType().equals("inbound-rtp") && members.containsKey("kind")) {
                                String kind = (String) members.get("kind");
                                if (kind != null) {
                                    BigInteger bytesReceived = (BigInteger) members.get("bytesReceived");
                                    Long packetsReceived = (Long) members.get("packetsReceived");
                                    Integer packetsLost = (Integer) members.get("packetsLost");

                                    if (kind.equals("audio")) {
                                        stringeeCallStats.callBytesReceived = bytesReceived != null ? bytesReceived.intValue() : 0;
                                        stringeeCallStats.callPacketsReceived = packetsReceived != null ? packetsReceived.intValue() : 0;
                                        stringeeCallStats.callPacketsLost = packetsLost != null ? packetsLost : 0;
                                    }
                                    if (kind.equals("video")) {
                                        stringeeCallStats.videoBytesReceived = bytesReceived != null ? bytesReceived.intValue() : 0;
                                        stringeeCallStats.videoPacketsReceived = packetsReceived != null ? packetsReceived.intValue() : 0;
                                        stringeeCallStats.videoPacketsLost = packetsLost != null ? packetsLost : 0;
                                    }
                                }
                            }
                        }
                        statsListener.onCallStats(stringeeCallStats);
                    });
                }
            }
        });
    }

    /**
     * Resume camera
     */
    public void resumeVideo() {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (room != null) {
                if (localVideoTrack != null) {
                    room.unpublish(localVideoTrack, new StatusListener() {
                        @Override
                        public void onSuccess() {

                        }
                    });
                }

                StringeeVideoTrack.Options options = new StringeeVideoTrack.Options();
                options.audio(true);
                options.video(true);
                options.screen(false);
                com.stringee.call.VideoQuality videoQuality = getVideoQuality();
                if (videoQuality != null) {
                    switch (videoQuality) {
                        case QUALITY_288P:
                            options.videoDimensions(VideoDimensions.CIF_VIDEO_DIMENSIONS);
                            break;
                        case QUALITY_480P:
                            options.videoDimensions(VideoDimensions.WVGA_VIDEO_DIMENSIONS);
                            break;
                        case QUALITY_720P:
                            options.videoDimensions(VideoDimensions.HD_720P_VIDEO_DIMENSIONS);
                            break;
                        case QUALITY_1080P:
                            options.videoDimensions(VideoDimensions.HD_1080P_VIDEO_DIMENSIONS);
                            break;
                    }
                }
                localVideoTrack = StringeeVideo.createLocalVideoTrack(client.getContext(), options, captureSessionListener, null);
                if (localVideoTrack != null) {
                    room.publish(localVideoTrack, new StatusListener() {
                        @Override
                        public void onSuccess() {
                            if (callListener != null) {
                                callListener.onLocalStream(StringeeCall2.this);
                                callListener.onLocalTrackAdded(StringeeCall2.this, localVideoTrack);
                            }
                        }

                        @Override
                        public void onError(StringeeError errorInfo) {
                            if (callListener != null) {
                                callListener.onError(StringeeCall2.this, -1, "Can not publish the local video track");
                            }
                            release();
                        }
                    });
                    localVideoTrack.mute(isMute);
                    localVideoTrack.enableVideo(isVideoEnable);
                } else {
                    if (callListener != null) {
                        callListener.onError(StringeeCall2.this, -1, "Can not publish the local video track");
                    }
                    release();
                }
            }
        });
    }

    /**
     * Send call info
     *
     * @param info JSONObject
     */
    public void sendCallInfo(JSONObject info, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            SendPacketUtils.sendCallInfo(requestId, client, info, callId);
        });
    }

    /**
     * Start capture screen
     */
    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public void startCaptureScreen(StringeeScreenCapture capture, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        if (capture == null) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "StringeeScreenCapture is null"));
            }
            return;
        }
        client.executeExecutor(() -> {
            StringeeVideoTrack videoTrack = capture.getCaptureTrack();
            if (videoTrack == null) {
                if (listener != null) {
                    listener.onError(new StringeeError(-1, "Can not get the video track from the screen capture"));
                }
                return;
            }

            room.publish(videoTrack, new StatusListener() {
                @Override
                public void onSuccess() {
                    localCaptureTrack = videoTrack;
                    client.getScreenCaptureMap().put(localCaptureTrack.getLocalId(), capture);
                    if (callListener != null) {
                        callListener.onVideoTrackAdded(localCaptureTrack);
                        callListener.onLocalTrackAdded(StringeeCall2.this, localCaptureTrack);
                    }
                    if (listener != null) {
                        listener.onSuccess();
                    }
                }

                @Override
                public void onError(StringeeError errorInfo) {
                    if (listener != null) {
                        listener.onError(errorInfo);
                    }
                }
            });
        });
    }

    /**
     * Stop capture screen
     *
     * @param listener callback
     */
    public void stopCaptureScreen(StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        client.executeExecutor(() -> {
            if (localCaptureTrack != null) {
                room.unpublish(localCaptureTrack, new StatusListener() {
                    @Override
                    public void onSuccess() {
                        localCaptureTrack.release();
                        client.getScreenCaptureMap().remove(localCaptureTrack.getLocalId());
                        if (callListener != null) {
                            callListener.onVideoTrackRemoved(localCaptureTrack);
                        }
                        localCaptureTrack = null;
                        if (listener != null) {
                            listener.onSuccess();
                        }
                    }

                    @Override
                    public void onError(StringeeError errorInfo) {
                        if (listener != null) {
                            listener.onError(errorInfo);
                        }
                    }
                });

            }
        });
    }

    /**
     * Send DTMF
     *
     * @param dtmf     DTMF
     * @param listener callback
     */
    public void sendDTMF(String dtmf, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is null");
        }
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        if (listener != null) {
            Common.statusListenerMap.put(requestId, listener);
        }
        SendPacketUtils.sendDTMF2(client, callId, dtmf, requestId);
    }

    public void restartICE() {
        if (localVideoTrack != null) {
            localVideoTrack.restartICE();
        }
    }

    /**
     * Take a snapshot while calling
     *
     * @param listener callback
     */
    public void snapshotLocal(CallbackListener<Bitmap> listener) {
        if (localVideoTrack != null) {
            localVideoTrack.snapshotLocal(listener);
        }
    }

    public void snapshotLocal(String fromUser, String requestId) {
        if (localVideoTrack != null) {
            localVideoTrack.snapshotLocal(new CallbackListener<Bitmap>() {
                @Override
                public void onSuccess(Bitmap bitmap) {
                    try {
                        JSONObject customMessage = new JSONObject();
                        customMessage.put("type", "capture-result");
                        customMessage.put("requestId", requestId);
                        customMessage.put("status", 1);
                        customMessage.put("image-base64", Utils.decodeToBase64String(bitmap));
                        client.sendCustomMessage(fromUser, customMessage, new StatusListener() {
                            @Override
                            public void onSuccess() {

                            }
                        });
                    } catch (JSONException e) {
                        Utils.reportException(StringeeCall2.class, e);
                    }
                }

                @Override
                public void onError(StringeeError errorInfo) {
                    super.onError(errorInfo);
                    try {
                        JSONObject customMessage = new JSONObject();
                        customMessage.put("type", "capture-result");
                        customMessage.put("requestId", requestId);
                        customMessage.put("status", 0);
                        client.sendCustomMessage(fromUser, customMessage, new StatusListener() {
                            @Override
                            public void onSuccess() {

                            }
                        });
                    } catch (JSONException e) {
                        Utils.reportException(StringeeCall2.class, e);
                    }
                }
            });
        }
    }

    /**
     * Take a screenshot while calling
     *
     * @param listener callback
     */
    public void snapshotScreen(CallbackListener<Bitmap> listener) {
        if (localVideoTrack != null) {
            localVideoTrack.snapshotScreen(listener);
        }
    }

    public void startTimer() {
        if (timer == null) {
            timer = new Timer();

            timer.schedule(new TimerTask() {
                @Override
                public void run() {
                    if (remoteVideoTrack != null) {
                        PeerConnection peerConnection = remoteVideoTrack.getPeerConnection();
                        if (peerConnection != null) {
                            peerConnection.getStats(rtcStatsReport -> {
                                if (rtcStatsReport.getStatsMap() == null) {
                                    return;
                                }
                                int bytesReceived = 0;
                                for (Map.Entry<String, RTCStats> e : rtcStatsReport.getStatsMap().entrySet()) {
                                    Map<String, Object> members = e.getValue().getMembers();
                                    if (e.getValue().getType().equals("inbound-rtp") && members.containsKey("kind")) {
                                        String kind = (String) members.get("kind");
                                        if (kind != null) {
                                            if (kind.equals("audio")) {
                                                BigInteger value = (BigInteger) members.get("bytesReceived");
                                                bytesReceived = value != null ? value.intValue() : 0;
                                            }
                                        }
                                    }
                                }
                                double videoTimestamp = (double) System.currentTimeMillis() / 1000;

                                //initialize values
                                if (mPrevCallTimestamp != 0) {
                                    //calculate video bandwidth
                                    mCallBw = (long) ((8 * (bytesReceived - mPrevCallBytes)) / (videoTimestamp - mPrevCallTimestamp));
                                }
                                mPrevCallTimestamp = videoTimestamp;
                                mPrevCallBytes = bytesReceived;
                                if (mCallBw == 0) {
                                    if (!client.isConnected()) {
                                        timeout++;
                                    }
                                    if (timeout > 10 && !reJoining) {
                                        needRestartTrack = true;
                                    }
                                } else {
                                    timeout = 0;
                                    needRestartTrack = false;
                                }
                                if (isNetworkChange() && !reJoining) {
                                    needRestartTrack = true;
                                }
                                if (needRestartTrack && client.isConnected()) {
                                    reJoinRoom();
                                }
                            });
                        }
                    } else {
                        if ((!client.isConnected() || isNetworkChange()) && !reJoining) {
                            needRestartTrack = true;
                        }
                        if (needRestartTrack && client.isConnected()) {
                            reJoinRoom();
                        }
                    }
                }
            }, 0, 1000);
        }
    }

    private boolean isNetworkChange() {
        if (Utils.isNetworkAvailable(context)) {
            String ip = client.getClientIp();
            if (!currentIp.equals(ip)) {
                currentIp = ip;
                return true;
            }
        }
        return false;
    }

    private void reJoinRoom() {
        if (reJoining) {
            return;
        }
        timeout = 0;
        needRestartTrack = false;
        reJoining = true;
        if (client == null) {
            throw new StringeeException("StringeeClient has not been initialized.");
        }
        client.executeExecutor(() -> {
            if (localVideoTrack != null) {
                room.unpublish(localVideoTrack, new StatusListener() {
                    @Override
                    public void onSuccess() {

                    }
                });
            }
            if (localCaptureTrack != null) {
                room.unpublish(localCaptureTrack, new StatusListener() {
                    @Override
                    public void onSuccess() {
                    }
                });
            }
            if (room != null) {
                room.leave(false, new StatusListener() {
                    @Override
                    public void onSuccess() {

                    }
                });
                StringeeVideo.release(context, room);
                StringeeVideo.initRoomCall(client, StringeeCall2.this);
            }
        });
    }

    public void startTimerCheckExitCall() {
        if (checkExitsTimer == null) {
            checkExitsTimer = new Timer();
            checkExitsTimer.schedule(new TimerTask() {
                @Override
                public void run() {
                    int requestId;
                    synchronized (Common.lock) {
                        requestId = ++Common.requestId;
                    }
                    Common.callbackListenerMap.put(requestId, new CallbackListener<Boolean>() {
                        @Override
                        public void onSuccess(Boolean exits) {
                            if (!exits) {
                                hangup("Call does not exist on the server", null);
                            }
                        }
                    });
                    SendPacketUtils.checkExitsCall(client, requestId, callId);
                }
            }, timeCheckExits, timeCheckExits);
        }
    }
}
