package com.stringee.common.event;

import androidx.annotation.VisibleForTesting;

import com.stringee.StringeeClient;
import com.stringee.call.StringeeCall;
import com.stringee.call.StringeeCall2;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.listener.StringeeConnectionListener;
import com.stringee.messaging.ChatRequest;
import com.stringee.messaging.Conversation;
import com.stringee.messaging.StringeeChange;
import com.stringee.messaging.StringeeObject;
import com.stringee.messaging.User;
import com.stringee.messaging.listeners.ChangeEventListener;
import com.stringee.messaging.listeners.LiveChatEventListener;
import com.stringee.messaging.listeners.UserTypingEventListener;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class EventManager implements EventSubject {
    @VisibleForTesting
    private static volatile EventManager instance;
    private final Map<String, List<EventReceiver>> eventMap = new HashMap<>();
    private int receiverId = 0;

    public EventManager() {
    }

    public static EventManager getInstance() {
        if (instance == null) {
            synchronized (EventManager.class) {
                if (instance == null) {
                    instance = new EventManager();
                }
            }
        }
        return instance;
    }

    @Override
    public void registerEvent(EventReceiver receiver, String flag) {
        Utils.post(() -> {
            int id;
            synchronized (EventManager.class) {
                id = ++receiverId;
            }
            receiver.create(flag, id);
            List<EventReceiver> eventReceivers = eventMap.get(flag);
            if (Utils.isEmpty(eventReceivers)) {
                eventReceivers = new ArrayList<>();
            }
            eventReceivers.add(receiver);
            eventMap.put(flag, eventReceivers);
        });
    }

    @Override
    public void unregisterEvent(EventReceiver receiver) {
        Utils.post(() -> {
            if (receiver != null) {
                List<EventReceiver> eventReceivers = eventMap.get(receiver.getFlag());
                if (!Utils.isEmpty(eventReceivers)) {
                    for (int i = 0; i < eventReceivers.size(); i++) {
                        if (eventReceivers.get(i).getId() == receiver.getId()) {
                            eventReceivers.remove(i);
                            break;
                        }
                    }
                    eventMap.put(receiver.getFlag(), eventReceivers);
                }
            }
        });
    }

    @Override
    public void sendEvent(Event event) {
        Utils.post(() -> {
            List<EventReceiver> observers = eventMap.get(event.getTag());
            if (!Utils.isEmpty(observers)) {
                for (int i = 0; i < observers.size(); i++) {
                    EventReceiver observer = observers.get(i);
                    if (observer != null) {
                        observer.onReceive(event);
                    }
                }
            }
        });
    }

    public static void sendClientConnectedEvent(StringeeClient client, boolean isReconnecting) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onConnectionConnected(client, isReconnecting));
            }
        }
        Event event = new Event(EventNotify.CLIENT_CONNECTION_CONNECTED.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.IS_RECONNECTING, isReconnecting);
        getInstance().sendEvent(event);
    }

    public static void sendClientDisconnectedEvent(StringeeClient client, boolean isReconnecting) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onConnectionDisconnected(client, isReconnecting));
            }
        }
        Event event = new Event(EventNotify.CLIENT_CONNECTION_DISCONNECTED.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.IS_RECONNECTING, isReconnecting);
        getInstance().sendEvent(event);
    }

    public static void sendClientIncomingCallEvent(StringeeClient client, StringeeCall stringeeCall) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onIncomingCall(stringeeCall));
            }
        }
        Event event = new Event(EventNotify.CLIENT_INCOMING_CALL.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.STRINGEE_CALL, stringeeCall);
        getInstance().sendEvent(event);
    }

    public static void sendClientIncomingCall2Event(StringeeClient client, StringeeCall2 stringeeCall) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onIncomingCall2(stringeeCall));
            }
        }
        Event event = new Event(EventNotify.CLIENT_INCOMING_CALL2.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.STRINGEE_CALL2, stringeeCall);
        getInstance().sendEvent(event);
    }

    public static void sendClientErrorEvent(StringeeClient client, StringeeError error) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onConnectionError(client, error));
            }
        }
        Event event = new Event(EventNotify.CLIENT_CONNECTION_ERROR.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.STRINGEE_ERROR, error);
        getInstance().sendEvent(event);
    }

    public static void sendClientNewTokenEvent(StringeeClient client) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onRequestNewToken(client));
            }
        }
        Event event = new Event(EventNotify.CLIENT_REQUEST_NEW_TOKEN.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        getInstance().sendEvent(event);
    }

    public static void sendClientCustomMsgEvent(StringeeClient client, String from, JSONObject msg) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onCustomMessage(from, msg));
            }
        }
        Event event = new Event(EventNotify.CLIENT_RECEIVE_CUSTOM_MESSAGE.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.FROM, from);
        event.put(EventConstants.MESSAGE, msg);
        getInstance().sendEvent(event);
    }

    public static void sendClientTopicMsgEvent(StringeeClient client, String from, JSONObject msg) {
        List<StringeeConnectionListener> listeners = client.getConnectionListeners();
        if (!Utils.isEmpty(listeners)) {
            for (int i = 0; i < listeners.size(); i++) {
                StringeeConnectionListener listener = listeners.get(i);
                client.executeListenerExecutor(() -> listener.onTopicMessage(from, msg));
            }
        }
        Event event = new Event(EventNotify.CLIENT_RECEIVE_TOPIC_MESSAGE.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.FROM, from);
        event.put(EventConstants.MESSAGE, msg);
        getInstance().sendEvent(event);
    }

    public static void sendChatChangeEvent(StringeeClient client, StringeeChange change) {
        List<ChangeEventListener> changeEventListeners = client.getChangeEventListeners();
        if (!Utils.isEmpty(changeEventListeners)) {
            for (int i = 0; i < changeEventListeners.size(); i++) {
                ChangeEventListener listener = changeEventListeners.get(i);
                client.executeListenerExecutor(() -> listener.onChangeEvent(change));
            }
        }
        boolean isConversation = change.getObjectType() == StringeeObject.Type.CONVERSATION;
        Event event = new Event((isConversation ? EventNotify.CHAT_CONVERSATION_CHANGE : EventNotify.CHAT_MESSAGE_CHANGE).getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CHANGE_TYPE, change.getChangeType());
        event.put(isConversation ? EventConstants.CONVERSATION : EventConstants.MESSAGE, change.getObject());
        getInstance().sendEvent(event);
    }

    public static void sendUserTypingEvent(StringeeClient client, boolean isTypingm, Conversation conversation, User user) {
        List<UserTypingEventListener> userTypingEventListeners = client.getUserTypingListeners();
        if (!Utils.isEmpty(userTypingEventListeners)) {
            for (int i = 0; i < userTypingEventListeners.size(); i++) {
                UserTypingEventListener listener = userTypingEventListeners.get(i);
                client.executeListenerExecutor(() -> {
                    if (isTypingm) {
                        listener.onTyping(conversation, user);
                    } else {
                        listener.onEndTyping(conversation, user);
                    }
                });
            }
        }
        Event event = new Event((isTypingm ? EventNotify.CHAT_USER_TYPING : EventNotify.CHAT_USER_END_TYPING).getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CONVERSATION, conversation);
        event.put(EventConstants.USER, user);
        getInstance().sendEvent(event);
    }

    public static void sendChatSupportReceiceEvent(StringeeClient client, ChatRequest chatRequest) {
        List<LiveChatEventListener> liveChatListeners = client.getLiveChatListeners();
        if (!Utils.isEmpty(liveChatListeners)) {
            for (int i = 0; i < liveChatListeners.size(); i++) {
                LiveChatEventListener listener = liveChatListeners.get(i);
                client.executeListenerExecutor(() -> {
                    if (chatRequest.getRequestType() == ChatRequest.RequestType.NORMAL) {
                        listener.onReceiveChatRequest(chatRequest);
                    } else {
                        listener.onReceiveTransferChatRequest(chatRequest);
                    }
                });
            }
        }
        Event event = new Event(EventNotify.CHAT_SUPPORT_RECEIVE_CHAT_REQUEST.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CHAT_REQUEST, chatRequest);
        getInstance().sendEvent(event);
    }

    public static void sendChatSupportHandledEvent(StringeeClient client, ChatRequest chatRequest, ChatRequest.State state) {
        List<LiveChatEventListener> liveChatListeners = client.getLiveChatListeners();
        if (!Utils.isEmpty(liveChatListeners)) {
            for (int i = 0; i < liveChatListeners.size(); i++) {
                LiveChatEventListener listener = liveChatListeners.get(i);
                client.executeListenerExecutor(() -> listener.onHandleOnAnotherDevice(chatRequest, state));
            }
        }
        Event event = new Event(EventNotify.CHAT_SUPPORT_REQUEST_HANDLE_ON_ANOTHER_DEVICE.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CHAT_REQUEST, chatRequest);
        event.put(EventConstants.CHAT_REQUEST_STATE, state);
        EventManager.sendChatSupportReceiceEvent(client, chatRequest);
    }

    public static void sendChatSupportTimeoutAnswerEvent(StringeeClient client, ChatRequest chatRequest) {
        List<LiveChatEventListener> liveChatListeners = client.getLiveChatListeners();
        if (!Utils.isEmpty(liveChatListeners)) {
            for (int i = 0; i < liveChatListeners.size(); i++) {
                LiveChatEventListener listener = liveChatListeners.get(i);
                client.executeListenerExecutor(() -> listener.onTimeoutAnswerChat(chatRequest));
            }
        }
        Event event = new Event(EventNotify.CHAT_SUPPORT_TIMEOUT_ANSWER.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CHAT_REQUEST, chatRequest);
        getInstance().sendEvent(event);
    }

    public static void sendChatSupportTimeoutInQueueEvent(StringeeClient client, Conversation conversation) {
        List<LiveChatEventListener> liveChatListeners = client.getLiveChatListeners();
        if (!Utils.isEmpty(liveChatListeners)) {
            for (int i = 0; i < liveChatListeners.size(); i++) {
                LiveChatEventListener listener = liveChatListeners.get(i);
                client.executeListenerExecutor(() -> listener.onTimeoutInQueue(conversation));
            }
        }
        Event event = new Event(EventNotify.CHAT_SUPPORT_TIMEOUT_IN_QUEUE.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CONVERSATION, conversation);
        getInstance().sendEvent(event);
    }

    public static void sendChatSupportHandledEvent(StringeeClient client, Conversation conversation, User endedByUser) {
        List<LiveChatEventListener> liveChatEventListeners = client.getLiveChatListeners();
        if (!Utils.isEmpty(liveChatEventListeners)) {
            for (int i = 0; i < liveChatEventListeners.size(); i++) {
                LiveChatEventListener listener = liveChatEventListeners.get(i);
                client.executeListenerExecutor(() -> listener.onConversationEnded(conversation, endedByUser));
            }
        }
        Event event = new Event(EventNotify.CHAT_SUPPORT_CONVERSATION_ENDED.getValue());
        event.put(EventConstants.STRINGEE_CLIENT, client);
        event.put(EventConstants.CONVERSATION, conversation);
        event.put(EventConstants.USER, endedByUser);
        EventManager.getInstance().sendEvent(event);
    }
}
