package com.stringee.messaging;

import com.stringee.StringeeClient;
import com.stringee.common.Common;
import com.stringee.common.Constant;
import com.stringee.common.SendPacketUtils;
import com.stringee.common.Utils;
import com.stringee.exception.StringeeError;
import com.stringee.exception.StringeeException;
import com.stringee.listener.StatusListener;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.regex.Pattern;

public class Message extends StringeeObject implements Serializable {
    public enum Type {
        TEXT(1),
        PHOTO(2),
        VIDEO(3),
        AUDIO(4),
        FILE(5),
        LINK(6),
        CREATE_CONVERSATION(7),
        RENAME_CONVERSATION(8),
        LOCATION(9),
        CONTACT(10),
        STICKER(11),
        RATING(14),
        NOTIFICATION(100),
        TEMP_DATE(1000);

        private final short value;

        Type(final int value) {
            this.value = (short) value;
        }

        public static Type getType(int value) {
            switch (value) {
                case 2:
                    return PHOTO;
                case 3:
                    return VIDEO;
                case 4:
                    return AUDIO;
                case 5:
                    return FILE;
                case 6:
                    return LINK;
                case 7:
                    return CREATE_CONVERSATION;
                case 8:
                    return RENAME_CONVERSATION;
                case 9:
                    return LOCATION;
                case 10:
                    return CONTACT;
                case 11:
                    return STICKER;
                case 14:
                    return RATING;
                case 100:
                    return NOTIFICATION;
                case 1000:
                    return TEMP_DATE;
                default:
                    return TEXT;
            }
        }

        public short getValue() {
            return value;
        }
    }

    public enum MsgType {
        SEND(0),
        RECEIVE(1);

        private final short value;

        MsgType(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static MsgType getType(int value) {
            if (value == 1) {
                return RECEIVE;
            }
            return SEND;
        }
    }

    public enum State {
        INITIALIZE(0),
        SENDING(1),
        SENT(2),
        DELIVERED(3),
        READ(4);

        private final short value;

        State(final int value) {
            this.value = (short) value;
        }

        public short getValue() {
            return value;
        }

        public static State getState(int value) {
            switch (value) {
                case 1:
                    return SENDING;
                case 2:
                    return SENT;
                case 3:
                    return DELIVERED;
                case 4:
                    return READ;
                default:
                    return INITIALIZE;

            }
        }
    }

    private int id;
    private String sId;
    private String localId;
    private String conversationId;
    private String convLocalId;
    private long createdOnLocalAt;
    private long createdAt;
    private long updateAt;
    private State state;
    private long sequence;
    private Type type;
    private String text;
    private String thumbnail;
    private String filePath;
    private double latitude;
    private double longitude;
    private String address;
    private MsgType msgType = MsgType.SEND;
    private String fileUrl;
    private String thumbnailUrl;
    private int duration;
    private float imageRatio;
    private String contact;
    private String fileName;
    private long fileLength;
    private String clientId;
    private String stickerName;
    private String stickerCategory;
    private String customData;
    private boolean deleted;
    private User sender;
    private String fileId;

    public Message() {
        state = State.INITIALIZE;
    }

    public Message(Type type) {
        state = State.INITIALIZE;
        createdOnLocalAt = System.currentTimeMillis();
        this.type = type;
    }

    public Message(String text) {
        this(Pattern.compile("(?i)\\\\b(?:https?|ftp)://\\\\S+(?:/|\\\\b)").matcher(text).matches() ? Type.LINK : Type.TEXT);
        this.text = text;
    }

    public int getDbId() {
        return id;
    }

    public void setDbId(int id) {
        this.id = id;
    }

    public String getId() {
        return sId;
    }

    public void setId(String sId) {
        this.sId = sId;
    }

    public String getLocalId() {
        return localId;
    }

    public void setLocalId(String sId) {
        this.localId = sId;
    }

    public String getConversationId() {
        return conversationId;
    }

    public void setConversationId(String conversationId) {
        this.conversationId = conversationId;
    }

    public String getConversationLocalId() {
        return convLocalId;
    }

    public void setConvLocalId(String convLocalId) {
        this.convLocalId = convLocalId;
    }

    @Deprecated
    public String getSenderId() {
        if (sender != null) {
            return sender.getUserId();
        }
        return "";
    }

    @Deprecated
    public String getSenderName() {
        if (sender != null) {
            return sender.getName();
        }
        return "";
    }

    @Deprecated
    public String getSenderAvatar() {
        if (sender != null) {
            return sender.getAvatarUrl();
        }
        return "";
    }

    public long getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(long createdAt) {
        this.createdAt = createdAt;
    }

    public long getUpdateAt() {
        return updateAt;
    }

    public void setUpdateAt(long updateAt) {
        this.updateAt = updateAt;
    }

    public State getState() {
        return state;
    }

    public void setState(State state) {
        this.state = state;
    }

    public long getSequence() {
        return sequence;
    }

    public long getCreatedOnLocalAt() {
        return createdOnLocalAt;
    }

    public void setCreatedOnLocalAt(long createdOnLocalAt) {
        this.createdOnLocalAt = createdOnLocalAt;
    }

    public void setSequence(long sequence) {
        this.sequence = sequence;
    }

    public Type getType() {
        return type;
    }

    public void setType(Type type) {
        this.type = type;
    }

    public String getText() {
        return text;
    }

    public void setText(String text) {
        this.text = text;
    }

    public String getThumbnail() {
        return thumbnail;
    }

    public void setThumbnail(String thumbnail) {
        this.thumbnail = thumbnail;
    }

    public double getLatitude() {
        return latitude;
    }

    public void setLatitude(double latitude) {
        this.latitude = latitude;
    }

    public double getLongitude() {
        return longitude;
    }

    public void setLongitude(double longitude) {
        this.longitude = longitude;
    }

    public String getAddress() {
        return address;
    }

    public void setAddress(String address) {
        this.address = address;
    }

    public MsgType getMsgType() {
        return msgType;
    }

    public void setMsgType(MsgType msgType) {
        this.msgType = msgType;
    }

    public String getFilePath() {
        return filePath;
    }

    public void setFilePath(String filePath) {
        this.filePath = filePath;
    }

    public String getFileUrl() {
        return fileUrl;
    }

    public void setFileUrl(String fileUrl) {
        this.fileUrl = fileUrl;
    }

    public String getThumbnailUrl() {
        return thumbnailUrl;
    }

    public void setThumbnailUrl(String thumbnailUrl) {
        this.thumbnailUrl = thumbnailUrl;
    }

    public int getDuration() {
        return duration;
    }

    public void setDuration(int duration) {
        this.duration = duration;
    }

    public float getImageRatio() {
        return imageRatio;
    }

    public void setImageRatio(float imageRatio) {
        this.imageRatio = imageRatio;
    }

    public String getContact() {
        return contact;
    }

    public void setContact(String contact) {
        this.contact = contact;
    }

    public String getFileName() {
        return fileName;
    }

    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    public long getFileLength() {
        return fileLength;
    }

    public void setFileLength(long fileLength) {
        this.fileLength = fileLength;
    }

    public String getClientId() {
        return clientId;
    }

    public void setClientId(String clientId) {
        this.clientId = clientId;
    }

    public String getStickerName() {
        return stickerName;
    }

    public void setStickerName(String stickerName) {
        this.stickerName = stickerName;
    }

    public String getStickerCategory() {
        return stickerCategory;
    }

    public void setStickerCategory(String stickerCategory) {
        this.stickerCategory = stickerCategory;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public void setDeleted(boolean deleted) {
        this.deleted = deleted;
    }

    public JSONObject getCustomData() {
        if (customData != null) {
            try {
                return new JSONObject(customData);
            } catch (JSONException e) {
                Utils.reportException(Message.class, e);
            }
        }
        return null;
    }

    public void setCustomData(JSONObject jsonObject) {
        if (jsonObject != null) {
            customData = jsonObject.toString();
        }
    }

    public User getSender() {
        return sender;
    }

    public void setSender(User sender) {
        this.sender = sender;
    }

    public String getFileId() {
        return fileId;
    }

    public void setFileId(String fileId) {
        this.fileId = fileId;
    }

    /**
     * Mark a message as read
     *
     * @param client   StringeeClient
     * @param listener StatusListener
     */
    @Deprecated
    public void markAsRead(final StringeeClient client, final StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (conversationId == null || sequence == 0) {
            listener.onError(new StringeeError(-1, "Conversation's id is null or message sequence is undefined."));
            return;
        }
        client.executeExecutor(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            client.getMessagesMap().put(requestId, Message.this);
            SendPacketUtils.sendMessageReport(client, requestId, Message.this, Constant.MESSAGE_STATE_READ);

//                String clientId = client.getUserId();
//                if (clientId != null) {
//                    DBHandler dbHandler = DBHandler.getIntance(client.getContext());
//                    int totalUnread = dbHandler.getTotalUnread(conversationId, clientId, sequence);
//                    if (totalUnread > 0) {
//                        dbHandler.decreaseTotalUnread(conversationId, client.getUserId(), totalUnread);
//                        Conversation conversation = dbHandler.getConversationByRemoteId(conversationId, clientId);
//                        if (conversation != null) {
//                            ChangeEventListenter changeEventListenter = client.getChangeEventListenter();
//                            if (changeEventListenter != null) {
//                                StringeeChange change = new StringeeChange(StringeeChange.Type.UPDATE, conversation);
//                                changeEventListenter.onChangeEvent(change);
//                            }
//                        }
//                    }
//                    dbHandler.updateMessagesState(Constant.MESSAGE_STATE_READ, sequence, conversationId, clientId, msgType);
//                }
        });
    }

    /**
     * Pin or unpin or message
     *
     * @param client     StringeeClient
     * @param pinOrUnpin boolean
     * @param listener   StatusListener
     */
    public void pinOrUnpin(StringeeClient client, boolean pinOrUnpin, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        client.executeExecutor(() -> {
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            client.getMessagesMap().put(requestId, Message.this);
            SendPacketUtils.pinMessage(client, requestId, this, pinOrUnpin);
        });
    }

    /**
     * Edit a text message
     *
     * @param client   StringeeClient
     * @param content  String
     * @param listener StatusListener
     */
    public void edit(StringeeClient client, String content, StatusListener listener) {
        if (client == null) {
            throw new StringeeException("StringeeClient is not initialized");
        }
        if (content == null) {
            if (listener != null) {
                listener.onError(new StringeeError(-1, "Content can not be null."));
            }
            return;
        }
        client.executeExecutor(() -> {
            text = content;
            int requestId;
            synchronized (Common.lock) {
                requestId = ++Common.requestId;
            }
            if (listener != null) {
                Common.statusListenerMap.put(requestId, listener);
            }
            client.getMessagesMap().put(requestId, Message.this);
            SendPacketUtils.editMessage(client, requestId, this, content);
        });
    }
}
