package com.stringee.common;

import android.graphics.Point;

import com.stringee.StringeeClient;
import com.stringee.call.StringeeCall;
import com.stringee.call.StringeeCall2;
import com.stringee.call.StringeeIceCandidate;
import com.stringee.call.StringeeSessionDescription;
import com.stringee.messaging.ChannelType;
import com.stringee.messaging.Conversation;
import com.stringee.messaging.ConversationFilter.ConversationFilterChatSupportStatus;
import com.stringee.messaging.Message;
import com.stringee.messaging.User;
import com.stringee.messaging.User.Role;
import com.stringee.network.tcpclient.packet.Packet;
import com.stringee.network.tcpclient.packet.ServiceType;
import com.stringee.video.StringeeVideoTrack;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.PeerConnection;

import java.util.List;

/**
 * Created by luannb on 5/24/2016.
 */
public class SendPacketUtils {

    /**
     * Reply ping
     */
    public static void sendPing(StringeeClient client) {
        Packet packet = new Packet(ServiceType.PING);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    /**
     * Log in by token
     */
    public static void login(StringeeClient client) {
        Packet packet = new Packet(ServiceType.AUTHEN);
        packet.setField("accesstoken", client.getToken());
        synchronized (Common.lock) {
            packet.setField("requestId", ++Common.requestId);
        }
        packet.setField("sdkVersion", "2.1.3");
        Point screenSize = Utils.getDisplaySize(client.getContext());
        packet.setField("screenSize", screenSize.x + "x" + screenSize.y);
        packet.setField("platformVersion", android.os.Build.VERSION.RELEASE);
        packet.setField("deviceName", android.os.Build.MODEL);
        packet.setField("platform", 2);
        packet.setField("deviceId", Utils.getDeviceId(client.getContext()));
        if (!client.isAlreadyAuthenticated()) {
            String sessionId = Utils.getDeviceId(client.getContext()) + "-" + System.currentTimeMillis();
            client.setSessionsId(sessionId);
            packet.setField("sessionId", sessionId);
        } else {
            packet.setField("sessionId", client.getSessionId());
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    /**
     * Join a meeting
     *
     * @param roomId room id
     */
    public static void joinRoom(StringeeClient client, int roomId, String password) {
        Packet packet = new Packet(ServiceType.VIDEO_CONFERENCE_JOIN_ROOM);
        synchronized (Common.lock) {
            packet.setField("requestId", ++Common.requestId);
        }
        packet.setField("roomId", roomId);
        if (!Utils.isEmpty(password)) {
            packet.setField("roomPassword", password);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void joinPublicRoom(StringeeClient client, String alias) {
        Packet packet = new Packet(ServiceType.VIDEO_CONFERENCE_JOIN_ROOM);
        packet.setField("fixRoomAlias", alias);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    /**
     * @param roomType room type
     */
    public static void makeRoom(StringeeClient client, int requestId, int roomType, String password) {
        Packet packet = new Packet(ServiceType.VIDEO_CONFERENCE_MAKE_ROOM);
        packet.setField("requestId", requestId);
        packet.setField("roomType", roomType);
        if (!Utils.isEmpty(password)) {
            packet.setField("roomPassword", password);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendCallOut(StringeeClient client, StringeeCall stringeeCall) {
        Packet packet = new Packet(ServiceType.CALLOUT_START_CALL);
        packet.setField("requestId", stringeeCall.getRequestId());
        packet.setField("toPhoneNumber", stringeeCall.getTo());
        packet.setField("fromPhoneNumber", stringeeCall.getFrom());
        packet.setField("sipTrunkId", 0);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void endCallOut(StringeeClient client, String callId) {
        Packet packet = new Packet(ServiceType.CALLOUT_STOP_CALL);
        synchronized (Common.lock) {
            packet.setField("requestId", ++Common.requestId);
        }
        packet.setField("callId", callId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void destroyRoom(StringeeClient client, int roomId) {
        Packet packet = new Packet(ServiceType.VIDEO_CONFERENCE_TERMINATE_ROOM);
        packet.setField("roomId", roomId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void startCall(StringeeClient client, StringeeCall stringeeCall) {
        Packet packet = new Packet(ServiceType.CALL_START);
        packet.setField("requestId", stringeeCall.getRequestId());
        packet.setField("toNumber", stringeeCall.getTo());
        packet.setField("fromNumber", stringeeCall.getFrom());
        packet.setField("video", stringeeCall.isVideoCall());
        if (!Utils.isEmpty(stringeeCall.getCustom())) {
            packet.setField("custom", stringeeCall.getCustom());
        }
        if (stringeeCall.getEncryptPhone() != null) {
            packet.setField("encryptPhone", stringeeCall.getEncryptPhone());
        }
        if (client.getPacketSenderThread() == null) {
            return;
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void endCall(StringeeClient client, String callId, int requestId, String reason) {
        Packet packet = new Packet(ServiceType.CALL_STOP);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("reason", reason);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void endCall2(StringeeClient client, int requestId, String callId, String reason) {
        Packet packet = new Packet(ServiceType.VIDEO_ROOM_END_CALL);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("reason", reason);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void callChangeState(StringeeClient client, int requestId, String callId, int code, String reason) {
        Packet packet = new Packet(ServiceType.CALL_STATE);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("code", code);
        packet.setField("reason", reason);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void callChangeState2(StringeeClient client, int requestId, String callId, int code, String reason) {
        Packet packet = new Packet(ServiceType.VIDEO_ROOM_CALL_STATE);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("code", code);
        packet.setField("reason", reason);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendDTMF(StringeeClient client, String callId, String s, int requestId) {
        Packet packet = new Packet(ServiceType.CALL_DTMF);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("digits", s);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendDTMF2(StringeeClient client, String callId, String s, int requestId) {
        Packet packet = new Packet(ServiceType.CALL_DTMF2);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("digits", s);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendCallInfo(int requestId, StringeeClient client, JSONObject info, String callId) {
        Packet packet = new Packet(ServiceType.CALL_INFO);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("info", info);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }


    public static void registerPushToken(StringeeClient client, String token, String packageName, int requestId) {
        Packet packet = new Packet(ServiceType.PUSH_DEVICE_TOKEN_REGISTER);
        packet.setField("requestId", requestId);
        packet.setField("deviceType", 2);
        packet.setField("production", true);
        packet.setField("voip", true);
        packet.setField("packageName", packageName);
        packet.setField("deviceToken", token);
        if (client.getPacketSenderThread() != null && !Utils.isEmpty(token)) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void registerPushTokenAndDeleteOthers(StringeeClient client, String token, String packageName, List<String> packageNames, int requestId) {
        Packet packet = new Packet(ServiceType.PUSH_DEVICE_TOKEN_REGISTER);
        packet.setField("requestId", requestId);
        packet.setField("deviceType", 2);
        packet.setField("production", true);
        packet.setField("voip", true);
        packet.setField("packageName", packageName);
        packet.setField("deviceToken", token);
        packet.setField("deleteOthers", true);
        if (!Utils.isEmpty(packageNames)) {
            JSONArray packageNamesArray = new JSONArray();
            for (int i = 0; i < packageNames.size(); i++) {
                packageNamesArray.put(packageNames.get(i));
            }
            packet.setField("packageNames", packageNamesArray);
        }
        if (client.getPacketSenderThread() != null && !Utils.isEmpty(token)) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void unregisterPushToken(StringeeClient client, String token, String packageName, String userId, int projectId, int requestId) {
        Packet packet = new Packet(ServiceType.PUSH_DEVICE_TOKEN_UNREGISTER);
        packet.setField("requestId", requestId);
        packet.setField("deviceType", 2);
        packet.setField("packageName", packageName);
        packet.setField("deviceToken", token);
        packet.setField("projectId", projectId);
        packet.setField("userId", userId);
        if (client.getPacketSenderThread() != null && !Utils.isEmpty(token)) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendCustomMessage(StringeeClient client, int requestId, String toUser, JSONObject jsonObject) {
        Packet packet = new Packet(ServiceType.CUSTOM_MESSAGE);
        packet.setField("requestId", requestId);
        packet.setField("toUser", toUser);
        packet.setField("message", jsonObject);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendTransfer(StringeeClient client, String callId, String userId, int requestId) {
        Packet packet = new Packet(ServiceType.TRANSFER_CALL);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("number", userId);
            jsonObject.put("type", "internal");
            jsonObject.put("alias", userId);
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }
        packet.setField("to", jsonObject);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void transferToPhone(StringeeClient client, int requestId, String callId, String from, String to) {
        Packet packet = new Packet(ServiceType.TRANSFER_CALL);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        JSONObject fromObject = new JSONObject();
        try {
            fromObject.put("number", from);
            fromObject.put("type", "external");
            fromObject.put("alias", from);
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }
        packet.setField("from", fromObject);

        JSONObject toObject = new JSONObject();
        try {
            toObject.put("number", to);
            toObject.put("type", "external");
            toObject.put("alias", to);
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }
        packet.setField("to", toObject);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void changeAttribute(StringeeClient client, int requestId, String name, String newValue) {
        Packet packet = new Packet(ServiceType.CHANGE_ATTRIBUTE);
        packet.setField("requestId", requestId);
        packet.setField("attribute", name);
        packet.setField("value", newValue);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void hold(StringeeClient client, int requestId, String callId, boolean hold) {
        Packet packet = new Packet(ServiceType.HOLD);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        packet.setField("hold", hold);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void createConversation(int requestId, StringeeClient client, Conversation conversation) {
        Packet packet = new Packet(ServiceType.CHAT_CREATE_CONVERSATION);
        packet.setField("requestId", requestId);
        packet.setField("localDbId", conversation.getLocalId());
        packet.setField("distinct", conversation.isDistinct());
        packet.setField("isGroup", conversation.isGroup());
        String name = conversation.getName();
        if (name == null) {
            name = "";
        }
        packet.setField("name", name);
        JSONArray infoArray = new JSONArray();
        JSONArray participantsArray = new JSONArray();
        String myUserId = conversation.getClientId();
        List<User> participants = conversation.getParticipants();
        if (!Utils.isEmpty(participants)) {
            for (int i = 0; i < participants.size(); i++) {
                User identity = participants.get(i);
                String userId = identity.getUserId();
                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("userId", userId);
                    jsonObject.put("display_name", identity.getName());
                    jsonObject.put("avatar_url", identity.getAvatarUrl());
                    infoArray.put(jsonObject);
                } catch (JSONException e) {
                    Utils.reportException(SendPacketUtils.class, e);
                }

                if (userId != null && userId.equals(myUserId)) {
                    continue;
                }
                participantsArray.put(identity.getUserId());
            }
        }
        packet.setField("participants", participantsArray);
        packet.setField("participantsInfo", infoArray);
        packet.setField("oaId", conversation.getOaId());
        packet.setField("customData", conversation.getCustomData());
        packet.setField("creatorId", conversation.getCreator());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendMessage(StringeeClient client, int requestId, Message message) {
        Packet packet = new Packet(ServiceType.CHAT_MESSAGE);
        packet.setField("requestId", requestId);
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("content", message.getText());
            jsonObject.put("text", message.getText());
            jsonObject.put("messageType", message.getType().getValue());
            switch (message.getType()) {
                case LOCATION:
                    JSONObject locationObject = new JSONObject();
                    locationObject.put("lat", message.getLatitude());
                    locationObject.put("lon", message.getLongitude());
                    jsonObject.put("location", locationObject);
                    break;
                case CONTACT:
                    JSONObject contactObject = new JSONObject();
                    contactObject.put("vcard", message.getContact());
                    jsonObject.put("contact", contactObject);
                    break;
                case PHOTO:
                    JSONObject photoObject = new JSONObject();
                    photoObject.put("filePath", message.getFileUrl());
                    photoObject.put("thumbnail", message.getThumbnailUrl());
                    if (!Utils.isEmpty(message.getFilePath())) {
                        photoObject.put("ratio", Utils.getRatio(message.getFilePath()));
                    } else {
                        photoObject.put("ratio", message.getImageRatio());
                    }
                    jsonObject.put("photo", photoObject);
                    break;
                case AUDIO:
                    JSONObject audioObject = new JSONObject();
                    audioObject.put("filePath", message.getFileUrl());
                    audioObject.put("duration", message.getDuration());
                    jsonObject.put("audio", audioObject);
                    break;
                case VIDEO:
                    JSONObject videoObject = new JSONObject();
                    videoObject.put("filePath", message.getFileUrl());
                    videoObject.put("thumbnail", message.getThumbnailUrl());
                    if (!Utils.isEmpty(message.getThumbnail())) {
                        videoObject.put("ratio", Utils.getRatio(message.getThumbnail()));
                    } else {
                        videoObject.put("ratio", message.getImageRatio());
                    }
                    videoObject.put("duration", message.getDuration());
                    jsonObject.put("video", videoObject);
                    break;
                case FILE:
                    JSONObject fileObject = new JSONObject();
                    fileObject.put("filePath", message.getFileUrl());
                    fileObject.put("filename", message.getFileName());
                    fileObject.put("length", message.getFileLength());
                    jsonObject.put("file", fileObject);
                    break;
                case STICKER:
                    JSONObject stickerObject = new JSONObject();
                    stickerObject.put("category", message.getStickerCategory());
                    stickerObject.put("name", message.getStickerName());
                    jsonObject.put("sticker", stickerObject);
                    break;
            }

            JSONObject customObject = message.getCustomData();
            if (customObject != null) {
                jsonObject.put("metadata", customObject);
            }
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }

        packet.setField("message", jsonObject);
        packet.setField("type", message.getType().getValue());
        packet.setField("convId", message.getConversationId());
        packet.setField("localDbId", message.getLocalId());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendMessageReport(StringeeClient client, int requestId, Message message, int state) {
        Packet packet = new Packet(ServiceType.CHAT_MESSAGE_REPORT);
        packet.setField("lastMsgSeq", message.getSequence());
        packet.setField("lastMsgTimestamp", message.getCreatedAt());
        packet.setField("status", state);
        packet.setField("convId", message.getConversationId());
        packet.setField("requestId", requestId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendMessageReport(StringeeClient client, int requestId, Conversation conversation, int state) {
        Packet packet = new Packet(ServiceType.CHAT_MESSAGE_REPORT);
        packet.setField("lastMsgSeq", conversation.getLastSequence());
        packet.setField("lastMsgTimestamp", conversation.getLastTimeNewMsg());
        packet.setField("status", state);
        packet.setField("convId", conversation.getId());
        packet.setField("requestId", requestId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void loadConversations(int requestId, StringeeClient client, long greater, long smaller, int limit, boolean loadAll, boolean unread, String oaId, List<ChannelType> channelTypes, ConversationFilterChatSupportStatus chatSupportStatus) {
        Packet packet = new Packet(ServiceType.CHAT_CONVERSATION_LOAD);
        packet.setField("lastUpdateGreater", greater);
        packet.setField("lastUpdateSmaller", smaller);
        packet.setField("limit", limit);
        packet.setField("requestId", requestId);
        if (loadAll) {
            packet.setField("loadAllConversations", true);
        }
        if (unread) {
            packet.setField("unread", true);
        }
        if (!Utils.isEmpty(oaId)) {
            packet.setField("oaId", oaId);
        }
        if (!Utils.isEmpty(channelTypes)) {
            JSONArray channelTypesArray = new JSONArray();
            for (int i = 0; i < channelTypes.size(); i++) {
                ChannelType type = channelTypes.get(i);
                channelTypesArray.put(type.getValue());
            }
            packet.setField("channelTypes", channelTypesArray);
        }

        if (chatSupportStatus != null) {
            packet.setField("ended", chatSupportStatus.getValue());
        }

        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void deleteConversation(int requestId, StringeeClient client, String convId, long seq) {
        Packet packet = new Packet(ServiceType.CHAT_DELETE_CONVERSATION);
        packet.setField("requestId", requestId);
        packet.setField("seq", seq);
        packet.setField("convId", convId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void loadMessages(int requestId, StringeeClient client, String convId, long seqGreater, long seqSmaller, int limit, String sortBy, int loadDeletedMsg, int loadDeletedMessageContent, boolean loadAll) {
        Packet packet = new Packet(ServiceType.CHAT_MESSAGE_LOAD);
        packet.setField("requestId", requestId);
        packet.setField("seqGreater", seqGreater);
        packet.setField("seqSmaller", seqSmaller);
        packet.setField("convId", convId);
        packet.setField("sort", sortBy);
        packet.setField("limit", limit);
        packet.setField("loadDeletedMessage", loadDeletedMsg);
        packet.setField("loadDeletedMessageContent", loadDeletedMessageContent);
        if (loadAll) {
            packet.setField("allowToLoadHistory", loadAll);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void getAttachmentMessages(StringeeClient client, int requestId, String convId, Message.Type type, long createdSmaller, int limit) {
        Packet packet = new Packet(ServiceType.CHAT_CONVERSATION_ATTACHMENT);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("msgType", type.getValue());
        packet.setField("createdSmaller", createdSmaller);
        packet.setField("limit", limit);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void getConversation(int requestId, StringeeClient client, String convIds) {
        Packet packet = new Packet(ServiceType.GET_CONVERSATION);
        packet.setField("convIds", convIds);
        packet.setField("requestId", requestId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void addParticipants(int requestId, StringeeClient client, String convId, List<User> participants) {
        Packet packet = new Packet(ServiceType.ADD_PARTICIPANTS);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        JSONArray jsonArray = new JSONArray();
        for (int i = 0; i < participants.size(); i++) {
            jsonArray.put(participants.get(i).getUserId());
        }
        packet.setField("userIds", jsonArray);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void removeParticipants(int requestId, StringeeClient client, String convId, List<User> participants) {
        Packet packet = new Packet(ServiceType.REMOVE_PARTICIPANTS);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        JSONArray jsonArray = new JSONArray();
        for (int i = 0; i < participants.size(); i++) {
            jsonArray.put(participants.get(i).getUserId());
        }
        packet.setField("userIds", jsonArray);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }


    public static void deleteMessages(int requestId, StringeeClient client, String convId, List<Message> messages) {
        Packet packet = new Packet(ServiceType.DELETE_MESSAGES);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        JSONArray jsonArray = new JSONArray();
        for (int i = 0; i < messages.size(); i++) {
            jsonArray.put(messages.get(i).getId());
        }
        packet.setField("messageIds", jsonArray);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void deleteMessages(int requestId, StringeeClient client, String convId, JSONArray messageIds) {
        Packet packet = new Packet(ServiceType.DELETE_MESSAGES);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("messageIds", messageIds);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendChatResponse(StringeeClient client, int requestId, String convId, boolean accept, int channelType) {
        Packet packet = new Packet(ServiceType.CHAT_AGENT_RESPONSE);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("answer", accept);
        packet.setField("channelType", channelType);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void createLiveChat(StringeeClient client, int requestId, String queueId, String customData, String serviceId) {
        String localId = Utils.getDeviceId(client.getContext()) + "-conversation-" + System.currentTimeMillis();
        Packet packet = new Packet(ServiceType.CHAT_CREATE_CONVERSATION);
        packet.setField("requestId", requestId);
        packet.setField("localDbId", localId);
        packet.setField("distinct", false);
        packet.setField("isGroup", true);
        packet.setField("name", queueId);
        packet.setField("participants", new JSONArray());
        if (customData != null) {
            packet.setField("customData", customData);
        }
        if (!Utils.isEmpty(serviceId)) {
            packet.setField("serviceId", serviceId);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void endChat(StringeeClient client, int requestId, String convId) {
        Packet packet = new Packet(ServiceType.END_CHAT);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void rateChat(StringeeClient client, int requestId, String convId, String comment, int rate) {
        Packet packet = new Packet(ServiceType.RATE_CHAT);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        if (!Utils.isEmpty(comment)) {
            packet.setField("comment", comment);
        }
        packet.setField("rating", rate);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void updateConversation(int requestId, StringeeClient client, String convId, String name, String avatar) {
        Packet packet = new Packet(ServiceType.UPDATE_CONVERSATION);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        if (name != null) {
            packet.setField("groupName", name);
        }
        if (avatar != null) {
            packet.setField("imageUrl", avatar);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void block(StringeeClient client, int requestId, String userId, String convId) {
        Packet packet = new Packet(ServiceType.CHAT_BLOCK_USER);
        packet.setField("requestId", requestId);
        if (userId != null) {
            packet.setField("blockUser", userId);
        }
        if (convId != null) {
            packet.setField("block_group_invite", convId);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }


    public static void getConversations(StringeeClient client, int requestId, JSONArray jsonArray) {
        Packet packet = new Packet(ServiceType.CHAT_CONVERSATION_FOR_USERS);
        packet.setField("requestId", requestId);
        packet.setField("participants", jsonArray);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void getTotalUnread(StringeeClient client, int requestId) {
        Packet packet = new Packet(ServiceType.CHAT_UNREAD_CONVERSATION_COUNT);
        packet.setField("requestId", requestId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void pinMessage(StringeeClient client, int requestId, Message message, boolean isPin) {
        Packet packet = new Packet(ServiceType.CHAT_PIN_MESSAGE);
        packet.setField("requestId", requestId);
        packet.setField("convId", message.getConversationId());
        packet.setField("msgId", message.getId());
        packet.setField("isPin", isPin);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void editMessage(StringeeClient client, int requestId, Message message, String newContent) {
        Packet packet = new Packet(ServiceType.CHAT_EDIT_MESSAGE);
        packet.setField("requestId", requestId);
        packet.setField("convId", message.getConversationId());
        packet.setField("msgId", message.getId());
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("content", newContent);
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }
        packet.setField("newContent", jsonObject);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void revokeMessages(StringeeClient client, int requestId, String convId, JSONArray msgIds, boolean delete) {
        Packet packet = new Packet(ServiceType.CHAT_REVOKE_MESSAGE);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("msgIds", msgIds);
        packet.setField("delete", delete ? 1 : 0);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void setRole(StringeeClient client, int requestId, String convId, String userId, Role role) {
        Packet packet = new Packet(ServiceType.CHAT_SET_ROLE);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("userId", userId);
        packet.setField("role", role.getValue());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void getMessagesByIds(StringeeClient client, int requestId, String convId, String[] msgIds) {
        Packet packet = new Packet(ServiceType.GET_MESSAGES_INFO);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        StringBuilder strMsgId = new StringBuilder();
        for (String msgId : msgIds) {
            strMsgId.append(msgId).append(",");
        }
        if (strMsgId.length() > 0) {
            strMsgId = new StringBuilder(strMsgId.substring(0, strMsgId.length() - 1));
        }
        packet.setField("msgIds", strMsgId.toString());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void transferChat(StringeeClient client, int requestId, String convId, String toUserId, String customerId, String customerName) {
        Packet packet = new Packet(ServiceType.CHAT_TRANSFER);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("toUserId", toUserId);
        packet.setField("customerId", customerId);
        packet.setField("customerName", customerName);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void continueChatting(StringeeClient client, int requestId, String convId) {
        Packet packet = new Packet(ServiceType.AGENT_CONTINUE_CHATTING);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendChatTransferResponse(StringeeClient client, int requestId, String convId, boolean accept) {
        Packet packet = new Packet(ServiceType.CHAT_TRANSFER_RESPONSE);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("answer", accept);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendChatTranscriptResponse(StringeeClient client, int requestId, String convId, String email, String domain) {
        Packet packet = new Packet(ServiceType.CHAT_SEND_TRANSCRIPT);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("email", email);
        packet.setField("domain", domain);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void updateUser(StringeeClient client, int requestId, User user) {
        Packet packet = new Packet(ServiceType.CHAT_USER_UPDATE_INFO);
        packet.setField("requestId", requestId);
        packet.setField("display_name", user.getName());
        packet.setField("email", user.getEmail());
        packet.setField("avatar_url", user.getAvatarUrl());
        packet.setField("phone", user.getPhone());
        packet.setField("location", user.getLocation());
        packet.setField("browser", user.getBrowser());
        packet.setField("platform", user.getPlatform());
        packet.setField("device", user.getDevice());
        packet.setField("ipaddress", user.getIpAddress());
        packet.setField("hostname", user.getHostName());
        packet.setField("useragent", user.getUserAgent());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void connectRoom(StringeeClient client, int requestId, String token) {
        Packet packet = new Packet(ServiceType.VIDEO_ENDPOINT_JOIN_ROOM);
        packet.setField("requestId", requestId);
        packet.setField("roomToken", token);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void publishTrack(StringeeClient client, int requestId, String roomId, StringeeVideoTrack videoTrack) {
        Packet packet = new Packet(ServiceType.VIDEO_PUBLISH_TRACK);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("localPcId", videoTrack.getLocalId());
        packet.setField("audio", videoTrack.audioEnabled());
        packet.setField("video", videoTrack.videoEnabled());
        packet.setField("screen", videoTrack.isScreenCapture());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendVideoSDP(StringeeClient client, int requestId, StringeeVideoTrack videoTrack, StringeeSessionDescription description) {
        Packet packet = new Packet(ServiceType.VIDEO_ENDPOINT_SDP);
        packet.setField("requestId", requestId);
        packet.setField("serverTrackId", videoTrack.getId());
        packet.setField("serverPcId", videoTrack.getPcId());
        packet.setField("roomId", videoTrack.getRoomId());
        packet.setField("isLocal", true);
        JSONObject data = new JSONObject();
        try {
            data.put("sdp", description.description);
            data.put("type", "offer");
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }
        packet.setField("data", data);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendVideoCandidate(StringeeClient client, int requestId, StringeeVideoTrack videoTrack, StringeeIceCandidate iceCandidate) {
        Packet packet = new Packet(ServiceType.VIDEO_ENDPOINT_CANDIDATE);
        packet.setField("requestId", requestId);
        packet.setField("serverTrackId", videoTrack.getId());
        packet.setField("serverPcId", videoTrack.getPcId());
        packet.setField("roomId", videoTrack.getRoomId());
        packet.setField("isLocal", videoTrack.isLocal());
        JSONObject data = new JSONObject();
        try {
            data.put("sdpMid", iceCandidate.sdpMid);
            data.put("sdpMLineIndex", iceCandidate.sdpMLineIndex);
            data.put("candidate", iceCandidate.sdp);
        } catch (JSONException e) {
            Utils.reportException(SendPacketUtils.class, e);
        }
        packet.setField("data", data);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void leaveRoom(StringeeClient client, int requestId, String roomId, boolean allClient) {
        Packet packet = new Packet(ServiceType.VIDEO_LEFT_ROOM);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("allStringeeClient", allClient);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void subscribeTrack(StringeeClient client, int requestId, String roomId, StringeeVideoTrack videoTrack) {
        Packet packet = new Packet(ServiceType.VIDEO_SUBSCRIBE_TRACK);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("subscribeTrackId", videoTrack.getId());
        packet.setField("localPcId", videoTrack.getLocalId());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void unpublish(StringeeClient client, int requestId, String roomId, String trackId) {
        Packet packet = new Packet(ServiceType.VIDEO_UNPUBLISH_TRACK);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("serverTrackId", trackId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void unsubscribe(StringeeClient client, int requestId, String roomId, StringeeVideoTrack videoTrack) {
        Packet packet = new Packet(ServiceType.VIDEO_UNSUBSCRIBE_TRACK);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("serverTrackId", videoTrack.getId());
        packet.setField("serverPcId", videoTrack.getPcId());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendRoomMsg(StringeeClient client, int requestId, String roomId, JSONObject jsonObject) {
        Packet packet = new Packet(ServiceType.VIDEO_ROOM_SEND_MSG);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("msg", jsonObject);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendVideoRoomEnableDisableAudioVideo(StringeeClient client, int requestId, String roomId, int mediaType, boolean enable, String trackId) {
        Packet packet = new Packet(ServiceType.VIDEO_ROOM_ENABLE_DISABLE_AUDIO_VIDEO);
        packet.setField("requestId", requestId);
        packet.setField("roomId", roomId);
        packet.setField("mediaType", mediaType);
        packet.setField("enable", enable);
        packet.setField("serverTrackId", trackId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void startCall2(StringeeClient client, int requestId, StringeeCall2 stringeeCall) {
        Packet packet = new Packet(ServiceType.VIDEO_ROOM_MAKE_CALL);
        packet.setField("requestId", requestId);
        packet.setField("toNumber", stringeeCall.getTo());
        packet.setField("fromNumber", stringeeCall.getFrom());
        packet.setField("video", stringeeCall.isVideoCall());
        if (!Utils.isEmpty(stringeeCall.getCustom())) {
            packet.setField("custom", stringeeCall.getCustom());
        }
        if (stringeeCall.getEncryptPhone() != null) {
            packet.setField("encryptPhone", stringeeCall.getEncryptPhone());
        }
        if (client.getPacketSenderThread() == null) {
            return;
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendBeginTyping(StringeeClient client, int requestId, String convId) {
        Packet packet = new Packet(ServiceType.CHAT_USER_BEGIN_TYPING);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("userId", client.getUserId());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void sendEndTyping(StringeeClient client, int requestId, String convId) {
        Packet packet = new Packet(ServiceType.CHAT_USER_END_TYPING);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("userId", client.getUserId());
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void loadChatRequest(StringeeClient client, int requestId) {
        Packet packet = new Packet(ServiceType.CHAT_REQUEST_LOAD);
        packet.setField("requestId", requestId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void joinOaConversation(StringeeClient client, int requestId, String convId) {
        Packet packet = new Packet(ServiceType.CHAT_JOIN_OA_CONVERSATION);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    /**
     * Send the media state of a video track to server
     *
     * @param videoTrack         the video track
     * @param iceConnectionState the state of the video track
     */
    public static void sendMediaState(StringeeVideoTrack videoTrack, PeerConnection.IceConnectionState iceConnectionState) {
        int requestId;
        synchronized (Common.lock) {
            requestId = ++Common.requestId;
        }
        Packet packet = new Packet(ServiceType.VIDEO_ROOM_MEDIA_STATE);
        packet.setField("requestId", requestId);
        packet.setField("serverTrackId", videoTrack.getId());
        String state = "";
        switch (iceConnectionState) {
            case NEW:
                state = "RTCIceConnectionStateNew";
                break;
            case CHECKING:
                state = "RTCIceConnectionStateChecking";
                break;
            case CONNECTED:
                state = "RTCIceConnectionStateConnected";
                break;
            case COMPLETED:
                state = "RTCIceConnectionStateCompleted";
                break;
            case FAILED:
                state = "RTCIceConnectionStateFailed";
                break;
            case DISCONNECTED:
                state = "RTCIceConnectionStateDisconnected";
                break;
            case CLOSED:
                state = "RTCIceConnectionStateClosed";
                break;
        }
        packet.setField("state", state);
        packet.setField("serverPcId", videoTrack.getPcId());
        packet.setField("localId", videoTrack.getLocalId());
        packet.setField("roomId", videoTrack.getRoomId());
        packet.setField("isLocal", videoTrack.isLocal());
        StringeeClient client = videoTrack.getClient();
        if (client != null && client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void getUserInfo(StringeeClient client, int requestId, String userIds) {
        Packet packet = new Packet(ServiceType.GET_USER_INFO);
        packet.setField("requestId", requestId);
        packet.setField("userIds", userIds);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void checkExitsCall(StringeeClient client, int requestId, String callId) {
        Packet packet = new Packet(ServiceType.CHECK_EXIST_CALL);
        packet.setField("requestId", requestId);
        packet.setField("callId", callId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void xChatLoadCustomerConversations(int requestId, StringeeClient client, long greater, long smaller, int limit, String customerId, boolean isAscending) {
        Packet packet = new Packet(ServiceType.XCHAT_LOAD_CUSTOMER_CONVERSATION_LIST);
        packet.setField("requestId", requestId);
        packet.setField("customerId", customerId);
        packet.setField("lastUpdateGreater", greater);
        packet.setField("lastUpdateSmaller", smaller);
        packet.setField("limit", limit);
        packet.setField("order", isAscending ? "ASC" : "DESC");

        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void xChatLoadConversations(int requestId, StringeeClient client, long greater, long smaller, int limit, boolean loadAll, boolean unread, boolean isAscending) {
        Packet packet = new Packet(ServiceType.XCHAT_LOAD_CONVERSATION);
        packet.setField("requestId", requestId);
        packet.setField("lastUpdateGreater", greater);
        packet.setField("lastUpdateSmaller", smaller);
        packet.setField("limit", limit);
        packet.setField("order", isAscending ? "ASC" : "DESC");
        if (loadAll) {
            packet.setField("loadAllConversations", true);
        }
        if (unread) {
            packet.setField("unread", true);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void xChatLoadMessages(int requestId, StringeeClient client, String convId, long seqGreater, long seqSmaller, int limit, boolean isAscending, int loadDeletedMsg, int loadDeletedMessageContent, boolean loadAll) {
        Packet packet = new Packet(ServiceType.XCHAT_LOAD_MESSAGE);
        packet.setField("requestId", requestId);
        packet.setField("convId", convId);
        packet.setField("seqGreater", seqGreater);
        packet.setField("seqSmaller", seqSmaller);
        packet.setField("limit", limit);
        packet.setField("order", isAscending ? "ASC" : "DESC");
        packet.setField("loadDeletedMessage", loadDeletedMsg);
        packet.setField("loadDeletedMessageContent", loadDeletedMessageContent);
        if (loadAll) {
            packet.setField("allowToLoadHistory", loadAll);
        }
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }

    public static void xChatGetConversation(int requestId, StringeeClient client, List<String> convIdList) {
        Packet packet = new Packet(ServiceType.XCHAT_GET_CONVERSATION_INFO);
        StringBuilder convIds = new StringBuilder();
        if (!Utils.isEmpty(convIdList)) {
            for (int i = 0; i < convIdList.size(); i++) {
                String convId = convIdList.get(i);
                convIds.append(convId).append(",");
            }
        }
        convIds = new StringBuilder(convIds.substring(0, convIds.length() - 1));
        packet.setField("convIds", convIds.toString());
        packet.setField("requestId", requestId);
        if (client.getPacketSenderThread() != null) {
            client.getPacketSenderThread().send(packet);
        }
    }
}
